// -------------------------------- pointCloud.h -------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_POINT_CLOUD_H
#define CAPY_POINT_CLOUD_H
#include "externalHeaders.h"
#include "cext.h"
#include "mathfun.h"
#include "capymath.h"
#include "bezier.h"
#include "btree.h"

// Description:
// Point cloud class.

// Structure to memorise information about a link between two points in a
// point cloud
typedef struct CapyPointCloudLink {

  // Indices of the linked points
  size_t iPoints[2];

  // Weight of the link
  double weight;
} CapyPointCloudLink;

// PointCloud object.
typedef struct CapyPointCloud {

  // Dimension of a point in the cloud.
  size_t dim;

  // Number of points in the cloud.
  size_t size;

  // Points
  CapyVec* points;

  // Number of links in the cloud.
  size_t nbLink;

  // Arrays of links between points
  CapyPointCloudLink* links;

  // Mean vector
  CapyVec mean;

  // Standard deviation vector
  CapyVec stdDev;

  // Covariance matrix
  CapyMat covariance;

  // Pearson correlation matrix
  CapyMat pearsonCorrelation;

  // Principal components (one component per column, ordered from the most
  // significant)
  CapyMat principalComponent;

  // Eigen values of the principal components
  CapyVec eigenValue;

  // Range of values in each dimension
  CapyRangeDouble* range;

  // Destructor
  void (*destruct)(void);

  // Get the Bezier approximating the point cloud.
  // Input:
  //   nbIn: the number of dimension from the first dimension of a point
  //         treated as inputs of the Bezier (the remaining ones being treated
  //         as the output).
  //   order: the Bezier order.
  // Output:
  //   Return a new CapyBezier which takes the first 'nbIn' values of a point
  //   as input and return as output the approximation of the remaining values
  //   of that point. Return NULL if the Bezier couldn't be created.
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyBezier* (*getApproxBezier)(
               size_t const nbIn,
    CapyBezierOrder_t const order);

  // Update the mean vector
  // Output:
  //   that->mean is updated.
  void (*updateMean)(void);

  // Update the stdDev vector
  // Output:
  //   that->mean and that->stdDev is updated.
  void (*updateStdDev)(void);

  // Update the covariance matrix
  // Output:
  //   that->covariance and that->mean are updated.
  void (*updateCovariance)(void);

  // Update the pearson correlation matrix
  // Output:
  //   that->pearsonCorrelation, that->stdDev, that->covariance and
  //   that->mean are updated.
  void (*updatePearsonCorrelation)(void);

  // Update the principal components
  // Output:
  //   that->principalComponent, that eigenValue, that->covariance and
  //   that->mean are updated.
  void (*updatePrincipalComponent)(void);

  // Update the ranges
  // Output:
  //   that->range is updated.
  void (*updateRange)(void);

  // Get an estimate of the minimum sum of weights along paths linking
  // two points (cf CapyPathFinder)
  // Inputs:
  //   iPointFrom: the start node of the path
  //   iPointTo: the end node of the path
  // Output:
  //   Return the estimate, which must be less or equal to the actual minimum
  //   sum of weights.
  double (*estimateWeightPath)(
    size_t const iNodeFrom,
    size_t const iNodeTo);
} CapyPointCloud;

// Create a CapyPointCloud.
// Input:
//   dim: the dimension of a point in the cloud
// Output:
//   Return a CapyPointCloud
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloud CapyPointCloudCreate(size_t const dim);

// Allocate memory for a new CapyPointCloud and create it
// Input:
//   dim: the dimension of a point in the cloud
// Output:
//   Return a CapyPointCloud
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloud* CapyPointCloudAlloc(size_t const dim);

// Free the memory used by a CapyPointCloud* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPointCloud to free
void CapyPointCloudFree(CapyPointCloud** const that);

// Structure to memorise the result of the query of the nearest point
typedef struct CapyPointCloudNearestNeighbourRes {

  // Index of the point in the point cloud
  size_t iPoint;

  // Distance to the query
  double dist;

  // Counter for the number of point tested (for performance evaluation)
  size_t nbTestedPoint;

  // Destructor
  void (*destruct)(void);
} CapyPointCloudNearestNeighbourRes;

// BTree of CapyPointCloudNearestNeighbourRes
CapyDecBTree(
  BTreePointCloudNearestNeighbourRes, CapyPointCloudNearestNeighbourRes)

// PointCloudNearestNeighbour object.
typedef struct CapyPointCloudNearestNeighbour {

  // Reference to the poin cloud
  CapyPointCloud const* pointCloud;

  // BTree representing the point cloud
  BTreePointCloudNearestNeighbourRes* btree;

  // Reference point for distance calculation
  CapyVec origin;

  // Destructor
  void (*destruct)(void);

  // Query the nearest point in the point cloud
  // Input:
  //   query: the query point
  //   iInitPoint: index of the point in the point cloud used for
  //               initialisation
  // Output:
  //   Return the index of the nearest point in the point cloud and its
  //   distance to the query.
  CapyPointCloudNearestNeighbourRes (*query)(
    CapyVec const* const query,
            size_t const iInitPoint);
} CapyPointCloudNearestNeighbour;

// Create a CapyPointCloudCloudNearestNeighbour.
// Input:
//   pointCloud: the point cloud
//   nbMaxElem: max number of elements in the underlying BTree
//   origin: reference point for distance calculation
// Output:
//   Return a CapyPointCloudCloudNearestNeighbour
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloudNearestNeighbour CapyPointCloudNearestNeighbourCreate(
    CapyPointCloud const* const pointCloud,
                   size_t const nbMaxElem,
           CapyVec const* const origin);

// Allocate memory for a new CapyPointCloudCloudNearestNeighbour and create it
// Input:
//   pointCloud: the point cloud
//   nbMaxElem: max number of elements in the underlying BTree
//   origin: reference point for distance calculation
// Output:
//   Return a CapyPointCloudCloudNearestNeighbour
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloudNearestNeighbour* CapyPointCloudNearestNeighbourAlloc(
    CapyPointCloud const* const pointCloud,
                   size_t const nbMaxElem,
           CapyVec const* const origin);

// Free the memory used by a CapyPointCloudNearestNeighbour* and reset
// '*that' to NULL
// Input:
//   that: a pointer to the CapyPointCloudNearestNeighbour to free
void CapyPointCloudNearestNeighbourFree(
  CapyPointCloudNearestNeighbour** const that);
#endif
