// -------------------------- poissonSampling.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "poissonSampling.h"
#include "idxCombination.h"

// Structure for efficient search of sample validity
typedef struct SampleGrid {
  double const** samples;
  size_t* dims;
  double cellSize;
  size_t nbCell;
} SampleGrid;

// Get the indices in a grid for a position
static void GridGetIdxFromPos(
  CapyPoissonSampling const* const that,
           SampleGrid const* const grid,
               double const* const pos,
                     size_t* const idx) {
  loop(iDim, that->dim) {
    idx[iDim] =
      (size_t)floor((pos[iDim] - that->domains[iDim].min) / grid->cellSize);
  }
}

// Get the cell index in a grid for a position indices
static size_t GridGetCellIdxFromPosIdx(
  CapyPoissonSampling const* const that,
           SampleGrid const* const grid,
               size_t const* const posIdx) {
  size_t idx = 0;
  loop(iDim, that->dim) {
    if(iDim > 0) idx *= grid->dims[iDim - 1];
    idx += posIdx[iDim];
  }
  return idx;
}

// Add a sample to the grid
static void GridAddSample(
  CapyPoissonSampling const* const that,
                 SampleGrid* const grid,
           CapySample const* const sample) {
  size_t posIdx[that->dim];
  GridGetIdxFromPos(that, grid, sample->vals, posIdx);
  size_t idx = GridGetCellIdxFromPosIdx(that, grid, posIdx);
  grid->samples[idx] = sample->vals;
}

// Get a sample values in the grid
static double const* GridGetSample(
  CapyPoissonSampling const* const that,
           SampleGrid const* const grid,
               size_t const* const posIdx) {
  size_t idx = GridGetCellIdxFromPosIdx(that, grid, posIdx);
  return grid->samples[idx];
}

// Check if a candidate sample is valid in term of its distance to other
// samples and sampling domains
static bool SampleIsValid(
  CapyPoissonSampling const* const that,
           CapySample const* const sample,
           SampleGrid const* const grid) {

  // Check the domain
  loop(iDim, that->dim) {
    if(sample->vals[iDim] < that->domains[iDim].min) return false;
    if(sample->vals[iDim] > that->domains[iDim].max) return false;
  }

  // Get the cell indices for the sample pos
  size_t posIdx[that->dim];
  GridGetIdxFromPos(that, grid, sample->vals, posIdx);

  // Loop on the neighbour cells
  CapyIdxCombination combi = CapyIdxCombinationCreate(that->dim);
  loop(iDim, that->dim) {
    if(posIdx[iDim] > 0) combi.ranges[iDim].min = posIdx[iDim] - 1;
    else combi.ranges[iDim].min = posIdx[iDim];
    if(posIdx[iDim] < grid->dims[iDim] - 1) {
      combi.ranges[iDim].max = posIdx[iDim] + 1;
    } else combi.ranges[iDim].max = posIdx[iDim];
  }
  forEach(posIdxNeighbour, combi) {

    // Get the sample position in the neighbour cell
    double const* neighbourPos = GridGetSample(that, grid, posIdxNeighbour);

    // If there was a neighbour
    if(neighbourPos != NULL) {

      // Check the distance to the sample
      double dist = 0.0;
      loop(iDim, that->dim) {
        double d = sample->vals[iDim] - neighbourPos[iDim];
        dist += d * d;
      }
      dist = sqrt(dist);
      if(dist < that->r) {
        $(&combi, destruct)();
        return false;
      }
    }
  }

  // Free memory
  $(&combi, destruct)();

  // If we reached here, the sample is valid
  return true;
}

// Get samples using Bridson's algorithm
// Output:
//   Return a new array of samples
static CapySamples* GetSamples(void) {
  methodOf(CapyPoissonSampling);

  // Initialise the array of result samples and array of active samples
  CapySamples* resultSamples = CapySamplesAlloc(0);
  CapySamples* activeSamples = CapySamplesAlloc(1);

  // Create the sample grid
  SampleGrid grid = {0};
  safeMalloc(grid.dims, that->dim);
  grid.cellSize = that->r / sqrt((double)(that->dim));
  grid.nbCell = 1;
  loop(iDim, that->dim) {
    grid.dims[iDim] =
      1 + (size_t)floor(
        (that->domains[iDim].max - that->domains[iDim].min) / grid.cellSize);
    grid.nbCell *= grid.dims[iDim];
  }
  safeMalloc(grid.samples, grid.nbCell);
  loop(i, grid.nbCell) grid.samples[i] = NULL;

  // Create the initial sample
  CapySample activeSample = CapySampleCreate(that->dim);
  loop(iDim, that->dim) {
    activeSample.vals[iDim] =
      $(&(that->rng), getDoubleRange)(that->domains + iDim);
  }

  // Add the initial sample to the array of active samples
  $(activeSamples, set)(0, &activeSample);
  GridAddSample(that, &grid, &activeSample);

  // While the array of active samples is not empty
  while($(activeSamples, getSize)() > 0) {

    // Shuffle the array and choose the current active sample as the first one
    $(activeSamples, shuffle)(&(that->rng));
    activeSample = $(activeSamples, get)(0);

    // Loop until we find a new sample from the active sample or we reach the
    // threshold
    size_t nbTry = 0;
    while(nbTry < that->k) {

      // Get a new sample from the active one
      CapySample candidateSample = CapySampleCreate(that->dim);
      double d = 0.0;
      loop(iDim, that->dim) {
        candidateSample.vals[iDim] = $(&(that->rng), getDouble)() - 0.5;
        d += candidateSample.vals[iDim] * candidateSample.vals[iDim];
      }
      d = sqrt(d);
      double r = (1.0 + $(&(that->rng), getDouble)()) * that->r;
      d = r / d;
      loop(iDim, that->dim) {
        candidateSample.vals[iDim] *= d;
        candidateSample.vals[iDim] += activeSample.vals[iDim];
      }

      // If the candidate respects the constraint distance with respect to
      // existing samples and domain
      bool isSampleValid = SampleIsValid(that, &candidateSample, &grid);
      if(isSampleValid) {

        // Add the candidate to the array of active samples
        size_t n = $(activeSamples, getSize)();
        $(activeSamples, resize)(n + 1);
        $(activeSamples, set)(n, &candidateSample);
        GridAddSample(that, &grid, &candidateSample);

        // Stop the search for a new candidate
        nbTry = that->k;

      // Else, the candidate doesn't respect the constraint distance
      } else {

        // Free the candidate
        $(&candidateSample, destruct)();

        // Increment the number of try
        ++nbTry;

        // If we have reached the threshold
        if(nbTry == that->k) {

          // Move the active sample from the active list to the result list
          size_t n = $(resultSamples, getSize)();
          $(resultSamples, resize)(n + 1);
          $(resultSamples, set)(n, &activeSample);
          n = $(activeSamples, getSize)();
          CapySample lastSample = $(activeSamples, get)(n - 1);
          $(activeSamples, set)(0, &lastSample);
          $(activeSamples, resize)(n - 1);
        }
      }
    }
  }

  // Free memory
  CapySamplesFree(&activeSamples);
  free(grid.samples);
  free(grid.dims);

  // Return the result samples
  return resultSamples;
}

// Free the memory used by a CapyPoissonSampling
static void Destruct(void) {
  methodOf(CapyPoissonSampling);
  $(that, destructCapySampling)();
}

// Create a CapyPoissonSampling
// Input:
//   seed: seed for the random number generator
//    dim: dimension of the samples
// Output:
//   Return a CapyPoissonSampling
CapyPoissonSampling CapyPoissonSamplingCreate(
  CapyRandomSeed_t const seed,
            size_t const dim) {

  // Create the Noise
  CapyPoissonSampling that = {
    .r = 1.0,
    .k = 30,
  };
  CapyInherits(that, CapySampling, (seed, dim));
  that.destruct = Destruct;
  that.getSamples = GetSamples;
  return that;
}

// Allocate memory for a new CapyPoissonSampling and create it
// Input:
//   seed: seed for the random number generator
//    dim: dimension of the samples
// Output:
//   Return a CapyPoissonSampling
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPoissonSampling* CapyPoissonSamplingAlloc(
  CapyRandomSeed_t const seed,
            size_t const dim) {

  // Allocate memory and create the new CapyPoissonSampling
  CapyPoissonSampling* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  CapyPoissonSampling b = CapyPoissonSamplingCreate(seed, dim);
  memcpy(that, &b, sizeof(CapyPoissonSampling));

  // Return the new CapyPoissonSampling
  return that;
}

// Free the memory used by a CapyPoissonSampling* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPoissonSampling to free
void CapyPoissonSamplingFree(CapyPoissonSampling** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
