// ------------------------------ polynomial.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "polynomial.h"

// Evaluate the function for a given input.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void Eval(
  double const* const in,
        double* const out) {
  methodOf(CapyPolynomial1D);
  out[0] = 0.0;
  double x = 1.0;
  loop(i, that->coeffs.dim) {
    out[0] += that->coeffs.vals[i] * x;
    x *= in[0];
  }
}

// Evaluate the derivative of the function for a given input and
// dimension
// Input:
//     in: the input vector
//   iDim: the derivation dimension
//    out: array of double updated with the result of derivation
// Exceptions:
//   May raise CapyExc_MallocFailed
static void EvalDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  methodOf(CapyPolynomial1D);
  if(iDim != 0) {
    out[0] = 0.0;
  } else {
    out[0] = 0.0;
    double x = 1.0;
    for(size_t i = 1; i < that->coeffs.dim; ++i) {
      out[0] += ((double)i) * that->coeffs.vals[i] * x;
      x *= in[0];
    }
  }
}

// Free the memory used by a CapyPolynomial1D
static void Destruct(void) {
  methodOf(CapyPolynomial1D);
  $(that, destructCapyMathFun)();
  CapyVecDestruct(&(that->coeffs));
}

// Create a CapyPolynomial1D
// Input:
//   coeffs: the coefficients of the polynomial in order x^0, x^1, x^2, ...
// Output:
//   Return a CapyPolynomial1D
CapyPolynomial1D CapyPolynomial1DCreate(CapyVec const* const coeffs) {
  CapyPolynomial1D that = {
    .coeffs = CapyVecCreate(coeffs->dim),
  };
  CapyInherits(that, CapyMathFun, (1, 1));
  that.destruct = Destruct;
  that.eval = Eval;
  that.evalDerivative = EvalDerivative;
  memcpy(that.coeffs.vals, coeffs->vals, coeffs->dim * sizeof(coeffs->vals[0]));
  return that;
}

// Allocate memory for a new CapyPolynomial1D and create it
// Input:
//   coeffs: the coefficients of the polynomial in order x^0, x^1, x^2, ...
// Output:
//   Return a CapyPolynomial1D
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPolynomial1D* CapyPolynomial1DAlloc(CapyVec const* const coeffs) {
  CapyPolynomial1D* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyPolynomial1DCreate(coeffs);
  return that;
}

// Free the memory used by a CapyPolynomial1D* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPolynomial1D to free
void CapyPolynomial1DFree(CapyPolynomial1D** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
