// -------------------------- predictor.h -------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_PREDICTOR_H
#define CAPY_PREDICTOR_H
#include "externalHeaders.h"
#include "dataset.h"

// Description:
// Parent class and structures for the predictor classes.

// Type of predictor
typedef enum CapyPredictorType {
  capyPredictorType_categorical,
  capyPredictorType_numerical,
  capyPredictorType_nb,
} CapyPredictorType;

// Accuracy measure types for numerical predictor
// MAE (mean absolute error), RMSE (root mean square error) and R^2 (coefficient of determination) are calculated for numerical predictor
// accuracy is calculated for categorical predictors
typedef enum CapyPredictorAccuracyMeasure {
  capyPredictorAccuracyMeasure_mae,
  capyPredictorAccuracyMeasure_rmse,
  capyPredictorAccuracyMeasure_rSquared,
  capyPredictorAccuracyMeasure_accuracy,
  capyPredictorAccuracyMeasure_nb,
} CapyPredictorAccuracyMeasure;

// PredictorEvaluation, class to memorise the result of evaluation of a
// predictor
//
// Overall accuracy
// For categorical predictor it is the percentage of correct answers
// (in [0.0, 1.0]).
// For numerical predictor it is the MAE (mean absolute error)
// double accuracy;
//
// Confusion matrix, such as confusionMatrix[i * n + j] is the number of times
// the i-th category has been predicted as the j-th category, where n is the
// number of possible categories
// size_t* confusionMatrix;
//
// Return the evaluated accuracy of the predictor
// Input:
//   type: type of accuracy measure
// Output:
//   Return the accuracy.
// double (*getAccuracy)(CapyPredictorAccuracyMeasure const type);
#define CapyPredictorEvaluationDef                                       \
struct {                                                                 \
  double accuracies[capyPredictorAccuracyMeasure_nb];                    \
  size_t* confusionMatrix;                                               \
  void (*destruct)(void);                                                \
  double (*getAccuracy)(CapyPredictorAccuracyMeasure const type);        \
}

// CapyPredictorEvaluation declaration
typedef CapyPredictorEvaluationDef CapyPredictorEvaluation;

// Create a CapyPredictorEvaluation
// Output:
//   Return a CapyPredictorEvaluation
CapyPredictorEvaluation CapyPredictorEvaluationCreate(void);

// Allocate memory for a new CapyPredictorEvaluation and create it
// Output:
//   Return a CapyPredictorEvaluation
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPredictorEvaluation* CapyPredictorEvaluationAlloc(void);

// Free the memory used by a CapyPredictorEvaluation* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapyPredictorEvaluation to free
void CapyPredictorEvaluationFree(CapyPredictorEvaluation** const that);

// PredictorResult, structure to memorise the result of prediction by a
// Predictor
typedef struct CapyPredictorPrediction {

  // Predicted category/value
  union {
    size_t category;
    double val;
  };

  // Confidence of the prediction (in [0,+inf[, the higher the more confident)
  double confidence;
} CapyPredictorPrediction;

// Types of input features scaling
// minMaxNormalization: [min,max]->[0,1]
// minMaxNormalizationSym: [min,max]->[-1,1]
// meanNormalization: (x - avg) / (max - min)
// standardization: (x - avg) / sigma
typedef enum CapyPredictorFeatureScaling {
  capyPredictorFeatureScaling_none,
  capyPredictorFeatureScaling_minMaxNormalization,
  capyPredictorFeatureScaling_minMaxNormalizationSym,
  capyPredictorFeatureScaling_meanNormalization,
  capyPredictorFeatureScaling_standardization,
  capyPredictorFeatureScaling_nb,
} CapyPredictorFeatureScaling;

// Predictor, parent class for all predictors
//
// Index of the predicted output in the dataset (default: 0)
// size_t iOutput;
//
// Type of predictor (default: categorical)
// CapyPredictorType type;
//
// Type of feature scaling (default: none)
// CapyPredictorFeatureScaling featureScaling
//
// Number of inputs
// size_t nbInput;
//
// Train the predictor on a dataset
// Input:
//   dataset: the dataset
// Output:
//   The predictor is trained.
// Exception:
//   May raise CapyExc_UnsupportedFormat
// void (*train)(CapyDataset const* const dataset);
//
// Predict an input
// Input:
//   inp: the input
// Output:
//   Return the prediction
// CapyPredictorPrediction (*predict)(CapyVec const* const inp);
//
// Evaluate the predictor on a dataset
// Input:
//   dataset: the dataset
// Output:
//   Return the evaluation of the predictor.
// CapyPredictorEvaluation* (*evaluate)(CapyDataset const* const dataset);
//
// Convert a CapyDataset into a CapyMat usable by the predictor
// Input:
//   dataset: the dataset to be converted
// Output:
//   Return a matrix formatted as necessary
// CapyMat (*cvtDatasetToMat)(
//   CapyDataset const* const dataset);
//
// Clone a predictor
// Output:
//   Return a clone of the predictor.
// void* (*clone)(void);
//
// Export the predictor as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
//   dataset: the training dataset
// Output:
//   A ready to use C function implementing the predictor is written on the
//   stream. See the comment exported with the function to know how to use
//   the exported function.
//  void (*exportToCFun)(
//                 FILE* const stream,
//           char const* const name,
//    CapyDataset const* const dataset);
//
// Export the predictor as a HTML web app
// Input:
//   stream: the stream where to export
//   title: the title of the web app
//   dataset: the training dataset
//   expectedAccuracy: the expected accuracy of the predictor (in [0,1])
// Output:
//   A ready to use web app implementing the predictor is written on the
//   stream.
//  void (*exportToHtml)(
//                 FILE* const stream,
//           char const* const title,
//    CapyDataset const* const dataset,
//                double const expectedAccuracy);
//
// Preprocess the input features in the training data
// Input:
//   mat: the training data
//   dataset: the training dataset
// Output:
//   'mat' is updated
// void (*scaleTrainingInputFeatures)(
//             CapyMat* const mat,
//   CapyDataset const* const dataset);
//
// Preprocess the input features in the input vector
// Input:
//   inp: the input vector
// Output:
//   'inp' is updated
// void (*scaleInputFeatures)(CapyVec* const inp);
//
//  Export the input feature scaling as C code
//  Input:
//    stream: the stream on which the code is exported
//  Output:
//    The scaling code is written to the stream (to be used by the
//    exportToCFun method.
//  void (*exportScaleInputToCFun)(FILE* const stream);
//
//  Save the predictor to a stream
//  Input:
//    stream: the stream on which to save
//  Output:
//    The predictor data are saved on the stream
//  void (*save)(FILE* const stream);
#define CapyPredictorDef                                                   \
struct {                                                                   \
  size_t iOutput;                                                          \
  CapyPredictorType type;                                                  \
  CapyPad(CapyPredictorType, type);                                        \
  CapyPredictorFeatureScaling featureScaling;                              \
  CapyPad(CapyPredictorFeatureScaling, featureScaling);                    \
  size_t nbInput;                                                          \
  CapyRangeDouble* scalingFrom;                                            \
  CapyRangeDouble scalingTo;                                               \
  void (*destruct)(void);                                                  \
  void (*train)(CapyDataset const* const dataset);                         \
  CapyPredictorPrediction (*predict)(CapyVec const* const inp);            \
  CapyPredictorEvaluation* (*evaluate)(CapyDataset const* const dataset);  \
  CapyMat (*cvtDatasetToMat)(CapyDataset const* const dataset);            \
  void* (*clone)(void);                                                    \
  void (*exportToCFun)(                                                    \
                 FILE* const stream,                                       \
           char const* const name,                                         \
    CapyDataset const* const dataset);                                     \
  void (*exportToHtml)(                                                    \
                 FILE* const stream,                                       \
           char const* const title,                                        \
    CapyDataset const* const dataset,                                      \
                double const expectedAccuracy);                            \
  void (*scaleTrainingInputFeatures)(                                      \
              CapyMat* const mat,                                          \
    CapyDataset const* const dataset);                                     \
  void (*scaleInputFeatures)(CapyVec* const inp);                          \
  void (*exportScaleInputToCFun)(FILE* const stream);                      \
  void (*save)(FILE* const stream);                                        \
}

// Predictor structure
typedef CapyPredictorDef CapyPredictor;

// Create a CapyPredictor
// Input:
//   type: type of predictor
// Output:
//   Return a CapyPredictor
CapyPredictor CapyPredictorCreate(CapyPredictorType const type);

// Allocate memory for a new CapyPredictor and create it
// Input:
//   type: type of predictor
// Output:
//   Return a CapyPredictor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPredictor* CapyPredictorAlloc(CapyPredictorType const type);

// Free the memory used by a CapyPredictor* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapyPredictor to free
void CapyPredictorFree(CapyPredictor** const that);
#endif
