// ---------------------------- quaternion.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_QUATERNION_H
#define CAPY_QUATERNION_H
#include "externalHeaders.h"
#include "cext.h"
#include "capymath.h"

// Description:
// Quaternion class.

// Quaternion object
typedef struct CapyQuaternion CapyQuaternion;
struct CapyQuaternion {

  // Values (3 first values are axis, 4th is angle)
  // [1, 0, 0 ,0] is a rotation around x by 180 degrees
  // [0, 1, 0 ,0] is a rotation around y by 180 degrees
  // [0, 0, 1 ,0] is a rotation around z by 180 degrees
  double vals[4];

  // Destructor
  void (*destruct)(void);

  // Convert the quaternion 'that' to a rotation matrix
  // Input:
  //   res: the result rotation matrix
  // Output:
  //   'res' is updated.
  void (*toRotMat)(CapyMat* const res);

  // Calculate the quaternion equivalent to the rotation of 'that' followed by
  // the rotation of 'tho'
  void (*compose)(
    CapyQuaternion const* const tho,
          CapyQuaternion* const res);

  // Calculate the quaternion equivalent to the rotation necessary to convert
  // 'that' into 'tho'
  // Inputs:
  //   tho: the target quaternion
  //   res: the result quaternion, can be same as 'that' or 'tho'
  // Output:
  //   'res' is updated. tho = compose(difference(that, tho), that) TODO check
  void (*difference)(
    CapyQuaternion const* const tho,
          CapyQuaternion* const res);

  // Calculate the inverse quaternion of the quaternion 'that'
  // Input:
  //   res: the result quaternion, can be same as 'that'
  // Output:
  //   'res' is updated
  void (*inverse)(CapyQuaternion* const res);

  // Return true if 'that' and 'tho' are equals, false else
  // Input:
  //   tho: the quaternion to compare to
  // Output:
  //   Return true if the quaternions are equal
  bool (*isEqualTo)(CapyQuaternion const* const tho);

  // Rotate the vector 'v' by the quaternion 'that'
  // Inputs:
  //   v: the vector to be rotated
  //   res: the result vector, can be same as 'v'
  // Output:
  //   'res' is updated.
  void (*apply)(
    double const* const v,
          double* const res);

  // Normalise the quaternion
  // Output:
  //   The quaternion is normalised
  void (*normalise)(void);

  // Get the rotation axis of the quaternion 'that'
  // Input:
  //   res: the result rotation axis
  // Output:
  //   'res' is set to the rotation axis
  void (*getRotAxis)(double* const res);

  // Get the rotation angle (in radians) of the quaternion 'that'
  // Output:
  //   Return the rotation angle
  double (*getRotAngle)(void);
};

// Create a CapyQuaternion
// Output:
//   Return a CapyQuaternion with default null rotation
CapyQuaternion CapyQuaternionCreate(void);

// Allocate memory for a new CapyQuaternion and create it
// Output:
//   Return a CapyQuaternion
// Exception:
//   May raise CapyExc_MallocFailed.
CapyQuaternion* CapyQuaternionAlloc(void);

// Create a new static quaternion from the rotation matrix 'rotMat'
CapyQuaternion CapyQuaternionCreateFromRotMat(CapyMat const* const rotMat);

// Allocate memory and create a new Quaternion from the rotation matrix
// 'rotMat'
CapyQuaternion* CapyQuaternionAllocFromRotMat(CapyMat const* const rotMat);

// Create a new static quaternion corresponding to the rotation around
// 'axis' (must be normalized) by 'theta' (in radians)
CapyQuaternion CapyQuaternionCreateFromRotAxis(
  double const* const axis,
         double const theta);

// Allocate memory and create a new Quaternion corresponding to the rotation
// around 'axis' (must be normalized) by 'theta' (in radians)
CapyQuaternion* CapyQuaternionAllocFromRotAxis(
  double const* const axis,
         double const theta);

// Create a new static quaternion corresponding to the rotation bringing the
// vector 'from' to the vector 'to'
CapyQuaternion CapyQuaternionCreateRotFromVecToVec(
  double const* const from,
  double const* const to);

// Allocate memory and create a new quaternion corresponding to the rotation
// bringing the vector 'from' to the vector 'to'
CapyQuaternion* CapyQuaternionAllocRotFromVecToVec(
  double const* const from,
  double const* const to);

// Free the memory used by a CapyQuaternion* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyQuaternion to free
void CapyQuaternionFree(CapyQuaternion** const that);
#endif
