// ------------------------------ random.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "random.h"

// Inverse of UINT64_MAX to use multiplication instead of division
// for faster generation of random double values
#define INV_UINT64_MAX ((double)0x3bf0000000000000)

// Initialise the state of a CapyRandom using the SplitMix64 generator
// Input:
//   that: the CapyRandom to be initialised
//   seed: the seed to initialise the generator
static void Init(
       CapyRandom* const that,
  CapyRandomSeed_t const seed) {

  // Apply the SplitMix64 algorithm to the seed and initialise the state
  // with the result
  CapyRandomSeed_t result = seed + 0x9E3779B97F4A7C15;
  result = (result ^ (result >> 30)) * 0xBF58476D1CE4E5B9;
  result = (result ^ (result >> 27)) * 0x94D049BB133111EB;
  that->state = that->bits = result ^ (result >> 31);
}

// Step the generator using xorshift*
// Input:
//   that: the CapyRandom to be stepped
static void StepXorshiftStar(void) {
  methodOf(CapyRandom);

  // Apply the xorshift*64 algorithm
  CapyRandomState_t x = that->bits;
  x ^= x >> 12;
  x ^= x << 25;
  x ^= x >> 27;
  x *= 0x2545F4914F6CDD1DULL;
  that->state = that->bits = x;
}

// Function used by PCG-XSH-RR
static uint32_t rotr32(
  uint32_t const x,
   uint8_t const r) {
  return x >> r | x << (-r & 31);
}

// Step the generator using PCG-XSH-RR
// Input:
//   that: the CapyRandom to be stepped
static void StepPcgXshRr(void) {
  methodOf(CapyRandom);

  // Apply the PCG-XSH-RR algorithm (generate 64 bits at once)
  uint64_t const multiplier = 6364136223846793005u;
  uint64_t const increment = 1442695040888963407u;
  loop(i, 2) {
    CapyRandomState_t x = that->state;
    uint8_t count = (uint8_t)(x >> 59);
    that->state = x * multiplier + increment;
    x ^= x >> 18;
    ((uint32_t*)&(that->bits))[i] = rotr32((uint32_t)(x >> 27), count);
  }
}

// Return the next pseudo random number as an integer using the
// xorshift algorithm
// Output:
//   Return the random number as an integer
static uint64_t GetUInt64(void) {
  methodOf(CapyRandom);

  // Step the generator
  $(that, step)();

  // Return the state
  return that->bits;
}

// Return the next pseudo random number as an integer using the
// xorshift algorithm
// Output:
//   Return the random number as an integer
static size_t GetSize(void) {
  methodOf(CapyRandom);

  // Step the generator
  $(that, step)();

  // Return the state
  return (size_t)that->bits;
}

static uint8_t GetUInt8(void) {
  methodOf(CapyRandom);

  // If there are not enough bits in the buffer,
  // produce 64 new random bits
  if(that->nbRemainingBits < 8) {
    that->buffer = GetUInt64();
    that->nbRemainingBits = 64;
  }
  uint8_t randomBit = that->buffer & 0xFF;
  that->buffer = that->buffer >> 8;
  that->nbRemainingBits -= 8;
  return randomBit;
}

static uint16_t GetUInt16(void) {
  methodOf(CapyRandom);

  // If there are not enough bits in the buffer,
  // produce 64 new random bits
  if(that->nbRemainingBits < 16) {
    that->buffer = GetUInt64();
    that->nbRemainingBits = 64;
  }
  uint16_t randomBit = that->buffer & 0xFFFF;
  that->buffer = that->buffer >> 16;
  that->nbRemainingBits -= 16;
  return randomBit;
}

static uint32_t GetUInt32(void) {
  methodOf(CapyRandom);

  // If there are not enough bits in the buffer,
  // produce 64 new random bits
  if(that->nbRemainingBits < 32) {
    that->buffer = GetUInt64();
    that->nbRemainingBits = 64;
  }
  uint32_t randomBit = that->buffer & 0xFFFFFFFF;
  that->buffer = that->buffer >> 32;
  that->nbRemainingBits -= 32;
  return randomBit;
}

// Return the next pseudo random number as a boolean
// Output:
//   Return the random number as a boolean
static bool GetBool(void) {
  methodOf(CapyRandom);

  // If there are no bit any more in the buffer,
  // produce 64 new random bits
  if(that->nbRemainingBits == 0) {
    that->buffer = GetUInt64();
    that->nbRemainingBits = 64;
  }
  bool randomBit = that->buffer & 1;
  that->buffer = that->buffer >> 1;
  that->nbRemainingBits--;
  return randomBit;
}

// Return the next pseudo random number as a double using the Downey
// algorithm
// Output:
//   Return the random number as a double in [0.0, 1.0]
static double GetDouble(void) {

// Guard against big endian architecture for which the masks below
// are invalid
#if __BYTE_ORDER__ != __ORDER_LITTLE_ENDIAN__
  sorry_only_little_endian_architecture_supported;
#endif

  // Get the exponent of the result
  union { double f; uint64_t i;} low, high, result;
  low.f = 0.0;
  high.f = 1.0;
  uint64_t low_exp = (low.i >> 52) & 0x07FF;
  uint64_t high_exp = (high.i >> 52) & 0x07FF;
  uint64_t exp = high_exp - 1;
  while(exp > low_exp && !GetBool()) exp--;

  // Get the mantissa of the result
  uint64_t mant = GetUInt64() & 0x000FFFFFFFFFFFFF;
  if(mant == 0 && GetBool()) exp++;

  // Create the result and return it
  result.i = (exp << 52) | mant;
  return result.f;
}

// Return the next pseudo random number as a double using integer
// division
// Output:
//   Return the random number as a double in [0.0, 1.0]. It is around
//   two times faster than the Downey's algorithm but generates random
//   floating-point number less efficiently. See
//   https://baillehachepascal.dev/2021/random_float.php
static double GetDoubleFast(void) {
  methodOf(CapyRandom);

  // Step the generator
  $(that, step)();

  // Return the state
  return (double)that->bits / (double)UINT64_MAX;
}

// Return the next pseudo random number as an integer mapped into a range
// Input:
//   range: the range to map the random number to
// Output:
//   Return the random number as an integer in the given range, bounds are
//   inclusive
#define GetIntRange(name, type)                                          \
static type Get ## name ## Range(CapyRange ## name const* const range) { \
  return range->min +                                                    \
    (type)(GetUInt64() % (uint64_t)(range->max + 1 - range->min));       \
}

GetIntRange(UInt8, uint8_t)
GetIntRange(Int8, int8_t)
GetIntRange(UInt16, uint16_t)
GetIntRange(Int16, int16_t)
GetIntRange(UInt32, uint32_t)
GetIntRange(Int32, int32_t)
GetIntRange(UInt64, uint64_t)
GetIntRange(Int64, int64_t)
GetIntRange(Size, size_t)

// Return the next pseudo random number as an integer mapped into a range
// using Lemire's algorithm
// Input:
//   range: the range to map the random number to
// Output:
//   Return the random number as an integer in the given range, bounds are
//   inclusive
static uint32_t GetUInt32RangeLemire(CapyRangeUInt32 const* const range) {
  uint32_t s = range->max + 1 - range->min;
  uint32_t x = GetUInt32();
  uint64_t m = (uint64_t)x * (uint64_t)s;
  uint32_t l = (uint32_t)m;
  if (l < s) {
    uint32_t t = -s % s;
    while (l < t) {
      x = GetUInt32();
      m = (uint64_t)x * (uint64_t)s;
      l = (uint32_t)m;
    }
  }
  return (uint32_t)((m >> 32) + range->min);
}

// Return the next pseudo random number as a double mapped into a range
// Input:
//   range: the range to map the random number to
// Output:
//   Return the random number as a double in the given range, bounds are
//   inclusive
static double GetDoubleRange(CapyRangeDouble const* const range) {
  return range->min + GetDouble() * (range->max - range->min);
}

static double GetDoubleRangeFast(CapyRangeDouble const* const range) {
  return range->min + GetDoubleFast() * (range->max - range->min);
}

// Return the next pseudo random number as a CapyRatio
// Output:
//   Return a random CapyRatio in [0.0, 1.0]
static CapyRatio GetRatio(void) {
  methodOf(CapyRandom);

  // Variable to memorise the result
  CapyRatio res;

  // Set the base
  $(that, step)();
  res.base = 0;

  // Set the denominator
  res.den = UINT64_MAX;

  // Set the numerator
  $(that, step)();
  res.num = that->bits;

  // Reduce the ratio
  res = CapyRatioReduce(res);

  // Return the result
  return res;
}

// Get a pseudo-random uint32_t using the Squirrel3 algorithm.
// Input:
//   position: the position in the pseudo-random sequence
// Output:
//   Return the pseudo-random uint32_t at the given position in the
//   sequence. Note that this sequence is different from the one
//   generated by the other methods of CapyRandom. Note also that the
//   sequence is completely determined by the seed of the CapyRandom.
static uint32_t GetSquirrel3(uint32_t const position) {
  methodOf(CapyRandom);

  // Variable to memorise the result
  uint32_t mangled = position;

  // Variables to perform the calculation
  uint32_t const noise1 = 0xB5297A4D;
  uint32_t const noise2 = 0x68E31DA4;
  uint32_t const noise3 = 0x1B56C4E9;

  // Squirrel3 algorithm
  mangled *= noise1;
  mangled += (uint32_t)that->seed;
  mangled ^= (mangled >> 8);
  mangled += noise2;
  mangled ^= (mangled << 8);
  mangled *= noise3;
  mangled ^= (mangled >> 8);

  // Return the result
  return mangled;
}

// Get a random event in a statistical normal distribution
// Input:
//   that: the random generator
//   dist: the distribution
// Output:
//   Return a distribution event
static CapyDistEvt GetDistEvtNormal(
            CapyRandom* const that,
  CapyDistNormal const* const dist) {
  CapyDistEvt evt = {.vec = CapyVecCreate(dist->dimEvt)};
  loop(iDim, dist->dimEvt) {

    // Box-Muller transform
    // note that you can get another event for free as
    // otherEvt.vals[iDim] = v[2] * sin(M_PI * 2.0 * v[1]) + dist->mean[iDim];
    double v[3];
    do {
      v[0] = $(that, getDouble)();
    } while(v[0] <= DBL_EPSILON);
    v[1] = $(that, getDouble)();
    v[2] = dist->stdDev[iDim] * sqrt(-2.0 * log(v[0]));
    evt.vec.vals[iDim] = v[2] * cos(M_PI * 2.0 * v[1]) + dist->mean[iDim];
  }
  return evt;
}

// Get a random event in a statistical discrete distribution
// Input:
//   dist: the distribution
// Output:
//   Return a distribution event
static CapyDistEvt GetDistEvtDiscrete(
              CapyRandom* const that,
  CapyDistDiscrete const* const dist) {
  double sum = 0.0;
  loop(iEvt, dist->occ->size) sum += ($(dist->occ, get)(iEvt)).prob;
  CapyRangeDouble range = {.vals = {0.0, sum}};
  double x = $(that, getDoubleRange)(&range);
  size_t iEvt = 0;
  sum = 0.0;
  while (iEvt < dist->occ->size && sum + ($(dist->occ, get)(iEvt)).prob < x) {
    sum += ($(dist->occ, get)(iEvt)).prob;
    ++iEvt;
  }
  return ($(dist->occ, get)(iEvt)).evt;
}

// Get a random index according to a vector of probability per index
// Input:
//   prob: the vector of probabilities
// Output:
//   Return the index
static size_t GetIdxGivenProbVec(CapyVec const* const prob) {
  methodOf(CapyRandom);
  double sum = 0.0;
  loop(i, prob->dim) sum += prob->vals[i];
  CapyRangeDouble range = {.vals = {0.0, sum}};
  double const x = $(that, getDoubleRange)(&range);
  size_t idx = 0;
  sum = 0.0;
  while (idx < prob->dim && sum + prob->vals[idx] < x) {
    sum += prob->vals[idx];
    ++idx;
  }
  return idx;
}

// Get a random event in a statistical distribution
// Input:
//   dist: the distribution
// Output:
//   Return a distribution event
static CapyDistEvt GetDistEvt(CapyDist const* const dist) {
  methodOf(CapyRandom);
  if(dist->type == capyDistributionType_discrete) {
    CapyDistDiscrete const* distDiscrete = (CapyDistDiscrete const*)dist;
    return GetDistEvtDiscrete(that, distDiscrete);
  } else if(dist->type == capyDistributionType_normal) {
    CapyDistNormal const* distNormal = (CapyDistNormal const*)dist;
    return GetDistEvtNormal(that, distNormal);
  } else raiseExc(CapyExc_UndefinedExecution);
  return (CapyDistEvt){0};
}

// Free the memory used by a CapyRandom
static void Destruct(void) {
  return;
}

// Set the type of PRNG
// Input:
//   type: type of PRNG
// Output:
//   The type is updated and the CapyRandom is re-initialised.
// Exception:
//   May raise CapyExc_UndefinedExecution
static void SetPrng(CapyRandomPrng const type) {
  methodOf(CapyRandom);
  that->type = type;
  switch(that->type) {
    case capyRandomPrng_pcg_xsh_rr:
      that->step = StepPcgXshRr;
      break;
    case capyRandomPrng_xorshift_star:
      that->step = StepXorshiftStar;
      break;
    default:
      raiseExc(CapyExc_UndefinedExecution);
  }
  Init(that, that->seed);
}

// Create a CapyRandom
// Input:
//   seed: seed of the generator
// Output:
//   Return a CapyRandom
CapyRandom CapyRandomCreate(CapyRandomSeed_t const seed) {

  // Create the CapyRandom
  CapyRandom that = {
    .type = capyRandomPrng_pcg_xsh_rr,
    .seed = seed,
    .destruct = Destruct,
    .step = StepPcgXshRr,
    .setPrng = SetPrng,
    .getBool = GetBool,
    .getUInt8 = GetUInt8,
    .getUInt16 = GetUInt16,
    .getUInt32 = GetUInt32,
    .getUInt64 = GetUInt64,
    .getSize = GetSize,
    .getDouble = GetDouble,
    .getDoubleFast = GetDoubleFast,
    .getUInt8Range = GetUInt8Range,
    .getInt8Range = GetInt8Range,
    .getUInt16Range = GetUInt16Range,
    .getInt16Range = GetInt16Range,
    .getUInt32Range = GetUInt32Range,
    .getInt32Range = GetInt32Range,
    .getUInt64Range = GetUInt64Range,
    .getInt64Range = GetInt64Range,
    .getUInt32RangeLemire = GetUInt32RangeLemire,
    .getSizeRange = GetSizeRange,
    .getDoubleRange = GetDoubleRange,
    .getDoubleRangeFast = GetDoubleRangeFast,
    .getRatio = GetRatio,
    .getSquirrel3 = GetSquirrel3,
    .getDistEvt = GetDistEvt,
    .getIdxGivenProbVec = GetIdxGivenProbVec,
  };

  // Initialise its state using the SplitMix64 generator
  Init(&that, seed);

  // Initialise the buffer used for optimisation
  that.buffer = $(&that, getUInt64)();
  that.nbRemainingBits = 64;

  // Return the CapyRandom
  return that;
}

// Allocate memory for a new CapyRandom and create it
// Input:
//   seed: seed of the generator
// Output:
//   Return a CapyRandom
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRandom* CapyRandomAlloc(CapyRandomSeed_t const seed) {
  CapyRandom* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyRandomCreate(seed);
  return that;
}

// Free the memory used by a CapyRandom* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRandom to free
void CapyRandomFree(CapyRandom** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

#if CAPY_ARCH_X86 == 1

// Get a truly random value to seed a random generator
// Output:
//   Return the truly random value.
uint32_t CapyGetTrulyRandomValue(void) {
  uint32_t val = 0;
  while(_rdseed32_step(&val) != 1);
  return val;
}

#endif
