// ---------------------------------- random.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_RANDOM_H
#define CAPY_RANDOM_H
#include "externalHeaders.h"
#include "cext.h"
#include "range.h"
#include "ratio.h"
#include "distribution.h"

// Description:
// Pseudo random generation class.

// Type of PRNG algorithms supported by CapyRandom
typedef enum CapyRandomPrng {
  capyRandomPrng_pcg_xsh_rr,
  capyRandomPrng_xorshift_star,
} CapyRandomPrng;

// Type for the seed of a CapyRandom
typedef uint64_t CapyRandomSeed_t;

// Type for the state of a CapyRandom
typedef uint64_t CapyRandomState_t;

// Random object
typedef struct CapyRandom {

  // Type of algorithm (default: capyRandomPrng_pcg_xsh_rr)
  CapyRandomPrng type;
  CapyPad(CapyRandomPrng, type);

  // Seed
  CapyRandomSeed_t seed;

  // State
  CapyRandomState_t state;

  // Random bits
  CapyRandomState_t bits;

  // Buffer for optimisation
  CapyRandomState_t buffer;
  uint8_t nbRemainingBits;
  CapyPad(uint8_t, nbRemainingBits);

  // Destructor
  void (*destruct)(void);

  // Step the state
  void (*step)(void);

  // Set the type of PRNG
  // Input:
  //   type: type of PRNG
  // Output:
  //   The type is updated and the CapyRandom is re-initialised.
  void (*setPrng)(CapyRandomPrng const type);

  // Return the next pseudo random number as a boolean
  // Output:
  //   Return the random number as a boolean
  bool (*getBool)(void);

  // Return the next pseudo random number as an integer using the
  // xorshift algorithm
  // Output:
  //   Return the random number as an integer
  uint8_t (*getUInt8)(void);
  uint16_t (*getUInt16)(void);
  uint32_t (*getUInt32)(void);
  uint64_t (*getUInt64)(void);
  size_t (*getSize)(void);

  // Return the next pseudo random number as a double using the Downey
  // algorithm
  // Output:
  //   Return the random number as a double in [0.0, 1.0]
  double (*getDouble)(void);

  // Return the next pseudo random number as a double using integer
  // division
  // Output:
  //   Return the random number as a double in [0.0, 1.0]. It is around
  //   two times faster than the Downey's algorithm but generates random
  //   floating-point number less efficiently. See
  //   https://baillehachepascal.dev/2021/random_float.php
  double (*getDoubleFast)(void);

  // Return the next pseudo random number as an integer mapped into a range
  // Input:
  //   range: the range to map the random number to
  // Output:
  //   Return the random number as an integer in the given range, bounds are
  //   inclusive
  uint8_t (*getUInt8Range)(CapyRangeUInt8 const* const range);
  int8_t (*getInt8Range)(CapyRangeInt8 const* const range);
  uint16_t (*getUInt16Range)(CapyRangeUInt16 const* const range);
  int16_t (*getInt16Range)(CapyRangeInt16 const* const range);
  uint32_t (*getUInt32Range)(CapyRangeUInt32 const* const range);
  int32_t (*getInt32Range)(CapyRangeInt32 const* const range);
  uint64_t (*getUInt64Range)(CapyRangeUInt64 const* const range);
  int64_t (*getInt64Range)(CapyRangeInt64 const* const range);
  size_t (*getSizeRange)(CapyRangeSize const* const range);

  // Return the next pseudo random number as an integer mapped into a range
  // using Lemire's algorithm
  // Input:
  //   range: the range to map the random number to
  // Output:
  //   Return the random number as an integer in the given range, bounds are
  //   inclusive
  uint32_t (*getUInt32RangeLemire)(CapyRangeUInt32 const* const range);

  // Return the next pseudo random number as a double mapped into a range
  // Input:
  //   range: the range to map the random number to
  // Output:
  //   Return the random number as a double in the given range, bounds are
  //   inclusive
  double (*getDoubleRange)(CapyRangeDouble const* const range);
  double (*getDoubleRangeFast)(CapyRangeDouble const* const range);

  // Return the next pseudo random number as a CapyRatio
  // Output:
  //   Return a random CapyRatio in [0.0, 1.0]
  CapyRatio (*getRatio)(void);

  // Get a pseudo-random uint32_t using the Squirrel3 algorithm.
  // Input:
  //   position: the position in the pseudo-random sequence
  // Output:
  //   Return the pseudo-random uint32_t at the given position in the
  //   sequence. Note that this sequence is different from the one
  //   generated by the other methods of CapyRandom. Note also that the
  //   sequence is completely determined by the seed of the CapyRandom.
  uint32_t (*getSquirrel3)(uint32_t const position);

  // Get a random event in a statistical distribution
  // Input:
  //   dist: the distribution
  // Output:
  //   Return a distribution event
  CapyDistEvt (*getDistEvt)(CapyDist const* const dist);

  // Get a random index according to a vector of probability per index
  // Input:
  //   prob: the vector of probabilities
  // Output:
  //   Return the index
  size_t (*getIdxGivenProbVec)(CapyVec const* const prob);
} CapyRandom;

// Create a CapyRandom
// Input:
//   seed: seed of the generator
// Output:
//   Return a CapyRandom
CapyRandom CapyRandomCreate(CapyRandomSeed_t const seed);

// Allocate memory for a new CapyRandom and create it
// Input:
//   seed: seed of the generator
// Output:
//   Return a CapyRandom
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRandom* CapyRandomAlloc(CapyRandomSeed_t const seed);

// Free the memory used by a CapyRandom* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRandom to free
void CapyRandomFree(CapyRandom** const that);
#if CAPY_ARCH_X86 == 1

// Get a truly random value to seed a random generator
// Output:
//   Return the truly random value.
uint32_t CapyGetTrulyRandomValue(void);
#endif
#endif
