// --------------------------------- range.h --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_RANGE_H
#define CAPY_RANGE_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Class to implement range of value.

// Range object. Declaration macro for a range of values of type 'type' and
// name 'name'
#define CapyDecRange(name, type)                                      \
typedef struct CapyRange ## name CapyRange ## name;                   \
struct CapyRange ## name {                                            \
  union {                                                             \
    type vals[2];                                                     \
    struct __attribute__((packed)) {type min, max;};                  \
  };                                                                  \
  CapyPad(type[2], 0);                                                \
  void (*destruct)(void);                                             \
  type (*trim)(type const val);                                       \
  void (*clip)(CapyRange ## name const* const range);                 \
};                                                                    \
CapyRange ## name CapyRange ## name ## Create(                        \
  type const min,                                                     \
  type const max);                                                    \
CapyRange ## name* CapyRange ## name ## Alloc(                        \
  type const min,                                                     \
  type const max);                                                    \
void CapyRange ## name ## Destruct(void);                             \
void CapyRange ## name ## Free(CapyRange ## name** const that);       \
type CapyRange ## name ## Trim(type const val);                       \
void CapyRange ## name ## Clip(CapyRange ## name const* const range);

// Create a new CapyRange
// Input:
//   min: the lower value of the range
//   max: the upper value of the range
// Output:
//   Return a new CapyRange
#define CapyDefRangeCreate(name, type)           \
CapyRange ## name CapyRange ## name ## Create(   \
  type const min,                                \
  type const max) {                              \
  CapyRange ## name that = {0};                  \
  that.vals[0] = min;                            \
  that.vals[1] = max;                            \
  that.destruct = CapyRange ## name ## Destruct; \
  that.trim = CapyRange ## name ## Trim;         \
  that.clip = CapyRange ## name ## Clip;         \
  return that;                                   \
}

// Create a newly allocated CapyRange
// Input:
//   min: the lower value of the range
//   max: the upper value of the range
// Output:
//   Return a newly allocated CapyRange
// Exception:
//   May raise CapyExc_MallocFailed
#define CapyDefRangeAlloc(name, type)            \
CapyRange ## name* CapyRange ## name ## Alloc(   \
  type const min,                                \
  type const max) {                              \
  CapyRange ## name* that = NULL;                \
  safeMalloc(that, 1);                           \
  if(!that) return NULL;                         \
  *that = CapyRange ## name ## Create(min, max); \
  return that;                                   \
}

// Free the memory used by a CapyRange
// Input:
//   that: the CapyRange to free
#define CapyDefRangeDestruct(name, type)   \
void CapyRange ## name ## Destruct(void) { \
  return;                                  \
}

// Free the memory used by a pointer to a CapyRange and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRange to free
#define CapyDefRangeFree(name, type)                             \
void CapyRange ## name ## Free(CapyRange ## name** const that) { \
  if(that == NULL || *that == NULL) return;                      \
  $(*that, destruct)();                                          \
  free(*that);                                                   \
  *that = NULL;                                                  \
}

// Trim a value to a CapyRange interval
// Input:
//   val: the value to trim
// Output:
//   Returned the value trimmed to the interval of the range
#define CapyDefRangeTrim(name, type)                      \
type CapyRange ## name ## Trim(type const val) {          \
  CapyRange ## name* that = (CapyRange ## name*)capyThat; \
  return (val < that->vals[0] ? that->vals[0] :           \
          val > that->vals[1] ? that->vals[1] : val);     \
}

// Clip the range to with another
// Input:
//   range: the other range
// Output:
//   The range is modified to the intersection of the range and the other
//   range. max < min means the intersection is empty 
#define CapyDefRangeClip(name, type)                                   \
void CapyRange ## name ## Clip(CapyRange ## name const* const range) { \
  CapyRange ## name* that = (CapyRange ## name*)capyThat;              \
  if(range != NULL) {                                                  \
    if(that->min < range->min) that->min = range->min;                 \
    if(that->max > range->max) that->max = range->max;                 \
  }                                                                    \
}

// Definition macro calling all the submacros at once for a range object
#define CapyDefRange(name, type)   \
  CapyDefRangeCreate(name, type)   \
  CapyDefRangeAlloc(name, type)    \
  CapyDefRangeDestruct(name, type) \
  CapyDefRangeFree(name, type)     \
  CapyDefRangeTrim(name, type)     \
  CapyDefRangeClip(name, type)

// Declaration of ranges for basic types
CapyDecRange(UInt8, uint8_t)
CapyDecRange(Int8, int8_t)
CapyDecRange(UInt16, uint16_t)
CapyDecRange(Int16, int16_t)
CapyDecRange(UInt32, uint32_t)
CapyDecRange(Int32, int32_t)
CapyDecRange(UInt64, uint64_t)
CapyDecRange(Int64, int64_t)
CapyDecRange(Size, size_t)
CapyDecRange(Float, float)
CapyDecRange(Double, double)
#endif
