// ------------------------------ ratio.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "ratio.h"

// CapyRatio value for NaN
CapyRatio const capyRatioNaN = {.base = 0, .num = 0, .den = 0};

// Predefined constants
CapyRatio const capyRatioZero = {.base = 0, .num = 0, .den = 1};
CapyRatio const capyRatioPi =
  {.base = 3, .num = 3612111, .den = 25510582};

// Create a CapyRatio equal to base+num/den
// Input:
//   base: the base
//   num: the numerator
//   den: the denominator
// Output:
//   Return a CapyRatio
CapyRatio CapyRatioCreate(
   CapyRatioBase_t const base,
   CapyRatioFrac_t const num,
   CapyRatioFrac_t const den) {
  return (CapyRatio){
    .base = base,
    .num = num,
    .den = den,
  };
}

// Allocate memory for a new CapyRatio and create it
// Input:
//   base: the base
//   num: the numerator
//   den: the denominator
// Output:
//   Return a CapyRatio
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRatio* CapyRatioAlloc(
   CapyRatioBase_t const base,
   CapyRatioFrac_t const num,
   CapyRatioFrac_t const den) {
  CapyRatio* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyRatioCreate(base, num, den);
  return that;
}

// Free the memory used by a CapyRatio
void CapyRatioDestruct(CapyRatio* const that) {
  (void)that;
}

// Free the memory used by a CapyRatio* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRatio to free
void CapyRatioFree(CapyRatio** const that) {
  if(that == NULL || *that == NULL) return;
  CapyRatioDestruct(*that);
  free(*that);
  *that = NULL;
}

// Check if a rational is equal to NaN
// Input:
//   r: the CapyRatio to check
// Output:
//   Return true if the rational is equal to NaN (denominator equals to
//   0)
bool CapyRatioIsNaN(CapyRatio const r) {
  return (r.den == 0);
}

// Create a CapyRatio from a double
// Input:
//   a: the double
// Output:
//   Return a new CapyRatio representing the nearest possible value to
//   the input double. If the input value is out of range, return
//   capyRatioNaN.
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioFromDouble(double const a) {

  // Get the integral and fractional values of the input value
  double base;
  double dec = modf(a, &base);

  // Ensure the decimal part is positive according to CapyRatio format
  if(dec < 0.0) {
    dec += 1.0;
    base -= 1.0;
  }

  // Create the bounding CapyRatios
  CapyRatio low = CapyRatioCreate((CapyRatioBase_t)base, 0, 1);
  CapyRatio high = CapyRatioCreate(low.base, 1, 1);

  // If the input value isn't in the supported range
  if(!equald(base, (double)(low.base))) {

    // Raise an exception or return NaN in case the exception isn't caught
    raiseExc(CapyExc_NumericalOverflow);
    return capyRatioNaN;
  }

  // If the decimal are equal to zero, nothing to do
  if(equald(dec, 0.0)) return low;

  // Use a binary search with the Farey sequence to find the nearest
  // ratio to the decimals
  CapyRatio med = low;
  while(
    high.den <= UINT64_MAX - high.den &&
    (low.num != high.num || low.den != high.den)
  ) {

    // Calculate the number of steps we can safely jump over to speed
    // up the conversion
    CapyRatioFrac_t k =
      (CapyRatioFrac_t)floor(
        (dec * (double)(low.den) - (double)(low.num)) /
        ((double)(high.num) - dec * (double)(high.den)));
    if(k == 0) k = 1;
    med.num = low.num + k * high.num;
    med.den = low.den + k * high.den;

    // Update the bounding fraction
    double dblMed = (double)(med.num) / (double)(med.den);
    if(equald(dec, dblMed)) low = high = med;
    else if(dblMed > dec) high = med;
    else low = med;
  }

  // Return the reduced result
  return CapyRatioReduce(low);
}

// Convert a CapyRatio to a double
// Input:
//   that: the CapyRatio to convert
// Output:
//   Return a double representing approximating the CapyRatio.
double CapyRatioToDouble(CapyRatio const that) {
  return (double)(that.base) + (double)(that.num) / (double)(that.den);
}

// Compare two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return -1 if x<y, else 0 if x==y, else 1 if x>y. Return 2 if an
//   exception occured and wasn't caught.
// Exception:
//   May raise CapyExc_NumericalOverflow
int8_t CapyRatioCmp(
  CapyRatio const x,
  CapyRatio const y) {
  if(CapyRatioIsNaN(x) || CapyRatioIsNaN(y)) {

    // Raise an exception, or return 2 in case the exception isn't caught
    raiseExc(CapyExc_NumericalOverflow);
    return 2;
  }

  // Compare the base to eliminate trivial cases
  if(x.base < y.base) return -1;
  else if(x.base > y.base) return 1;
  else if(x.num == y.num && x.den == y.den) return 0;

  // Else, the bases are equals
  else {

    // Calculate the difference of the fractional parts
    CapyRatio diff = CapyRatioSub(
      (CapyRatio){.base = 0, .num = x.num, .den = x.den},
      (CapyRatio){.base = 0, .num = y.num, .den = y.den});

    // Return the result according to the sign of the difference
    if(diff.base == 0 && diff.num == 0) return 0;
    else if(diff.base < 0) return -1;
    else return 1;
  }
}

// Reduce a CapyRatio
// Input:
//   that: the CapyRatio to reduce
// Output:
//   Return a new CapyRatio equals to the reduced CapyRatio
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioReduce(CapyRatio const that) {

  // Trying to reduce NaN gives NaN
  if(CapyRatioIsNaN(that)) return capyRatioNaN;

  // Declare the variable to memorise the result
  CapyRatio res = that;

  // If the numerator is greater than the denominator
  if(res.num >= res.den) {

    // If the reduction triggers an overflow of the base
    CapyRatioFrac_t inc = res.num / res.den;
    CapyRatioFrac_t threshold = 0;
    if(res.base < 0) {
      threshold = (CapyRatioFrac_t)INT64_MAX + (CapyRatioFrac_t)(-res.base);
    } else {
      threshold = (CapyRatioFrac_t)INT64_MAX - (CapyRatioFrac_t)(res.base);
    }
    if(inc > threshold) {

      // Raise an exception, or return NaN if the exception is not
      // caught
      raiseExc(CapyExc_NumericalOverflow);
      return capyRatioNaN;
    }

    // Update the components to keep the numerator smaller than the
    // denominator
    res.base += (CapyRatioBase_t)inc;
    res.num -= inc * res.den;
  }

  // If the numerator is not null
  if(res.num != 0) {

    // Divide the numerator and denominator by their gcd
    CapyRatioFrac_t gcd = CapyGcd(res.num, res.den);
    res.num /= gcd;
    res.den /= gcd;

  // Else, the numerator is null, ensure the denominator is equal to 1
  // by convention
  } else res.den = 1;

  // Return the result
  return res;
}

// Add two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x+y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioAdd(
  CapyRatio const x,
  CapyRatio const y) {

  // Trying to add NaN gives NaN
  if(CapyRatioIsNaN(x) || CapyRatioIsNaN(y)) return capyRatioNaN;

  // Declare the variable to memorise the result
  CapyRatio res = capyRatioZero;

  // Declare a variable to manage a residual part when adding fraction
  CapyRatio residual = capyRatioZero;

  // Variable to memorise the eventual increment of the base
  CapyRatioBase_t inc = 0;

  // If the multiplication of denominators doesn't overflow
  if(x.den <= (UINT64_MAX / y.den)) {

    // Calculate the result denominator
    res.den = x.den * y.den;

    // Calculate the two halves of the numerator (they can't overflow)
    CapyRatioFrac_t firstHalf = x.num * y.den;
    CapyRatioFrac_t sndHalf = y.num * x.den;

    // If the sum of the two halves doesn't overflow, calculate the
    // result numerator at once
    if(firstHalf <= (UINT64_MAX - sndHalf)) res.num = firstHalf + sndHalf;

    // Else, the sum of the two halves overflows
    else {

      // If the base cannot be incremented
      if(res.base == INT64_MAX) {

        // Raise an exception or return NaN in case the exception isn't
        // caught
        raiseExc(CapyExc_NumericalOverflow);
        return capyRatioNaN;
      }

      // Bring back the numerator in range by increasing the base
      // and correcting the numerator accordingly
      inc += 1;
      res.num = firstHalf - (res.den - sndHalf);
    }

  // Else, the multiplication of denominators overflows
  } else {

    // Variables to commonalise code
    CapyRatioFrac_t a, b, c, d;

    // Set the commonalised variables according to the smallest
    // component of the denominator
    if(x.den > y.den) {
      a = x.num;
      b = x.den;
      c = y.num;
      d = y.den;
      res.den = x.den;
    } else {
      a = y.num;
      b = y.den;
      c = x.num;
      d = x.den;
      res.den = y.den;
    }

    // Calculate bc/d first using the modified Peasant multiplication
    // algorithm. Here the fractional part is below the available
    // precision level, it is then ignored.
    CapyPeasantMulDivRes pmd = CapyPeasantMulDiv(b, c, d);
    if(CapyRatioIsNaN(pmd.frac)) {

      // Raise an exception or return NaN in case the exception isn't
      // caught
      raiseExc(CapyExc_NumericalOverflow);
      return capyRatioNaN;
    }

    // The base is normally in pmd.base, but for the trivial case it
    // may be in pmd.frac.base. Adding both cover all cases at once.
    // Here pmd.frac.base is guaranteed to be positive so it is safe to cast
    res.num = pmd.base + (CapyRatioFrac_t)(pmd.frac.base);

    // Avoid doing extra useless calculation if the residual is null
    if(pmd.frac.num > 0) {

      // The numerator of the residual may be reducable by the result
      // denominator. If it is, do it right now as it can help to avoid
      // overflow in the next step.
      CapyRatioFrac_t gcd = CapyGcd(pmd.frac.num, res.den);
      if(gcd > 1) {
        pmd.frac.num /= gcd;
        residual.den = res.den / gcd;

      // Else it wasn't reducable, leave it as it is
      } else residual.den = res.den;

      // If the residual of the Peasant multiplication can be multiplied
      // by the denominator
      CapyRatioFrac_t r = UINT64_MAX / pmd.frac.den;
      if(residual.den <= r) {

        // Memorise the residual to add it later
        residual.num = pmd.frac.num;
        residual.den *= pmd.frac.den;

      // Else, the residual of the Peasant multiplication can't be
      // multiplied by the denominator
      } else {

        // Approximate the residual to the highest possible precision
        residual.num = pmd.frac.num / (res.den / r);
        residual.den = UINT64_MAX;
      }

      // Reduce the residual
      residual = CapyRatioReduce(residual);
    }

    // If the remaining addition does not overflow, do it
    if(res.num <= UINT64_MAX - a) res.num += a;

    // Else, the remaining addition overflows, increment the base
    // and perform the corrected addition in the numerator
    else {
      inc += 1;
      res.num = a - (res.den - res.num);
    }
  }

  // Check if the reduction would increase the base, if it does do
  // it right now to be able to check for overflow when performing
  // the addition of the two bases
  while(res.num >= res.den) {
    inc += 1;
    res.num -= res.den;
  }

  // Check for overflow on the addition of the bases
  if(
    (
      x.base >= 0 && y.base >= 0 &&
      x.base > (INT64_MAX - y.base - inc)
    ) || (
      x.base < 0 && y.base < 0 &&
      (x.base + inc) < (INT64_MIN - y.base)
    )
  ) {

    // Raise an exception or return NaN in case the exception isn't caught
    raiseExc(CapyExc_NumericalOverflow);
    return capyRatioNaN;
  }

  // Add the bases and the eventual increment in the proper order to
  // avoid overflow
  if(x.base < y.base) res.base = (x.base + inc) + y.base;
  else res.base = (y.base + inc) + x.base;

  // Reduce the result
  res = CapyRatioReduce(res);

  // If the residual is not null, add it to the result
  if(residual.num != 0) res = CapyRatioAdd(res, residual);

  // Return the result
  return res;
}

// Get the negative of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to -x
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioNeg(CapyRatio const x) {

  // Negative of NaN gives NaN
  if(CapyRatioIsNaN(x)) return capyRatioNaN;

  // Check for overflow
  if(x.base == INT64_MIN && x.num == 0) {

    // Raise an exception or return NaN in case the exception isn't caught
    raiseExc(CapyExc_NumericalOverflow);
    return capyRatioNaN;
  }

  // Calculate the negative
  CapyRatio res;
  if(x.base > 0) res.base = -x.base - 1;
  else res.base = -(x.base + 1);
  res.num = x.den - x.num;
  res.den = x.den;

  // Reduce the result
  res = CapyRatioReduce(res);

  // Return the result
  return res;
}

// Substract two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x-y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioSub(
  CapyRatio const x,
  CapyRatio const y) {
  return CapyRatioAdd(x, CapyRatioNeg(y));
}

// Multiply two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x*y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioMul(
  CapyRatio const x,
  CapyRatio const y) {

  // Trying to multiply NaN gives NaN
  if(CapyRatioIsNaN(x) || CapyRatioIsNaN(y)) return capyRatioNaN;

  // Declare the variable to memorise the result
  CapyRatio res = capyRatioZero;

  // Intermediate CapyRatio-s to memorise each component of the
  // multiplication
  #define nbComps 8
  CapyRatio comps[nbComps] = {
    capyRatioZero, capyRatioZero, capyRatioZero, capyRatioZero,
    capyRatioZero, capyRatioZero, capyRatioZero, capyRatioZero
  };

  // If the base product is not null
  if(x.base != 0 && y.base != 0) {

    // Variables to memorise the corrected bases
    CapyRatioBase_t a = x.base;
    CapyRatioBase_t b = y.base;

    // Check for overflow from the multplication of the bases
    bool overflow = false;
    if(a > 0 && b > 0) {
      if(a > INT64_MAX / b) overflow = true;
    } else if(a < 0 && b < 0) {
      a += 1;
      b += 1;
      comps[5].base = -a;
      comps[6].base = -b;
      comps[7].base = 1;
      if(b != 0 && -a > INT64_MAX / -b) overflow = true;
    } else if(a < 0 && b > 0) {
      a += 1;
      comps[6].base = -b;
      if(a < INT64_MIN / b) overflow = true;
    } else if(a > 0 && b < 0) {
      b += 1;
      comps[5].base = -a;
      if(b < INT64_MIN / a) overflow = true;
    }
    if(overflow) {

      // Raise an exception or return NaN in case the exception isn't
      // caught
      raiseExc(CapyExc_NumericalOverflow);
      return capyRatioNaN;
    }

    // Calculate the first component with the eventually corrected bases
    comps[0].base = a * b;
  }

  // Variables to commonalise code in the calculation of the second
  // and third components
  struct {
    CapyRatioFrac_t b;
    CapyRatioFrac_t n;
    CapyRatioFrac_t d;
    int s;
    CapyPad(int, s);
  } v[2];
  v[0].b = (CapyRatioFrac_t)(y.base < 0 ? -(y.base) : y.base);
  v[0].n = x.num;
  v[0].d = x.den;
  v[0].s = (y.base < 0 ? -1 : 1);
  v[1].b = (CapyRatioFrac_t)(y.base < 0 ? -(x.base) : x.base);
  v[1].n = y.num;
  v[1].d = y.den;
  v[1].s = (x.base < 0 ? -1 : 1);

  // Loop to commonalise code of the calculation of the second and third
  // components
  loop(i, 2) {

    // If the numerator is not null
    if(v[i].b != 0 && v[i].n != 0) {

      // Calculate the intermediate fraction using the modified Peasant
      // algorithm.
      CapyPeasantMulDivRes pmd =
        CapyPeasantMulDiv(v[i].b, v[i].n, v[i].d);

      // Memorise the result. pmd.base + pmd.frac.base is guaranteed
      // to not overflow: it's the multiplication of an integer by a
      // fraction less than 1.
      comps[1 + i] = pmd.frac;
      comps[1 + i].base += (CapyRatioBase_t)(pmd.base);

      // Correct the sign if the base was negative.
      if(v[i].s == -1) {
        comps[1 + i].base = -(comps[1 + i].base + 1);
        comps[1 + i].num = comps[1 + i].den - comps[1 + i].num;
        comps[1 + i] = CapyRatioReduce(comps[1 + i]);
      }
    }
  }

  // If the denominator of the third fraction doesn't overflow
  if(x.den <= UINT64_MAX / y.den) {

    // Calculate the denominator of the third fraction
    CapyRatioFrac_t den = x.den * y.den;

    // Now that the denominator is calculated we can reuse the modified
    // Peasant multiplication to calculate the third fraction
    CapyPeasantMulDivRes pmd =
      CapyPeasantMulDiv(x.num, y.num, den);

    // Memorise the result. pmd.base + pmd.frac.base is guaranteed
    // to be null: it's the multiplication of two fractions less
    // than 1.
    comps[3] = pmd.frac;

  // Else, the denominator of the third fraction overflows
  } else {

    // Variables to commonalise code
    CapyRatioFrac_t a, b;

    // Set the commonalised variables according to the smallest
    // component of the denominator
    if(x.den > y.den) {
      a = y.den;
      b = x.den;
    } else {
      a = x.den;
      b = y.den;
    }

    // Calculate x_n*y_n/a first using the modified Peasant
    // multiplication algorithm. Here the fractional part is below the
    // available precision level, it is then ignored.
    CapyPeasantMulDivRes pmd = CapyPeasantMulDiv(x.num, y.num, a);
    if(CapyRatioIsNaN(pmd.frac)) {

      // Raise an exception or return NaN in case the exception isn't
      // caught.
      raiseExc(CapyExc_NumericalOverflow);
      return capyRatioNaN;
    }

    // The base is normally in pmd.base, but for the trivial case it
    // may be in pmd.frac.base. Adding both cover all cases at once.
    comps[3].num = pmd.base + (CapyRatioFrac_t)(pmd.frac.base);
    comps[3].den = b;
    comps[3] = CapyRatioReduce(comps[3]);

    // Avoid doing extra useless calculation if the residual is null
    if(pmd.frac.num > 0) {

      // Initialise the residual
      comps[4] = pmd.frac;
      comps[4].base = 0;

      // The numerator of the residual may be reducable by the result
      // denominator. If it is, do it right now as it can help to avoid
      // overflow in the next step.
      CapyRatioFrac_t gcd = CapyGcd(comps[4].num, b);
      if(gcd > 1) {
        comps[4].num /= gcd;
        b /= gcd;
      }

      // If the residual of the Peasant multiplication can be multiplied
      // by the denominator
      CapyRatioFrac_t r = UINT64_MAX / b;
      if(comps[4].den <= r) {

        // Multiply by the denominator
        comps[4].den *= b;

      // Else, the residual of the Peasant multiplication can't be
      // multiplied by the denominator
      } else {

        // Approximate the residual to the highest possible precision
        comps[4].num /= comps[4].den / r;
        comps[4].den = UINT64_MAX;
      }

      // Reduce the residual
      comps[4] = CapyRatioReduce(comps[4]);
    }
  }

  // Variables to memorise the components gathered by signedness
  CapyRatio posComps[nbComps];
  CapyRatio negComps[nbComps];
  int nbPos = 0;
  int nbNeg = 0;

  // Gather the components by signedness
  loop(i, nbComps) {
    if(comps[i].base >= 0) posComps[nbPos++] = comps[i];
    else negComps[nbNeg++] = comps[i];
  }

  // Add the components by pair of positive/negative values
  int iPos = 0;
  int iNeg = 0;
  while(iPos < nbPos && iNeg < nbNeg) {
    CapyRatio p = CapyRatioAdd(posComps[iPos], negComps[iNeg]);
    if(p.base >= 0) {
      posComps[iPos] = p;
      ++iNeg;
    } else {
      negComps[iNeg] = p;
      ++iPos;
    }
  }

  // Here there is no more pair, just sum up the remaining values
  CapyRatio* remainVals;
  int iRemain;
  int nbRemain;
  if(iPos < nbPos) {
    remainVals = posComps;
    iRemain = iPos;
    nbRemain = nbPos;
  } else {
    remainVals = negComps;
    iRemain = iNeg;
    nbRemain = nbNeg;
  }
  while(iRemain < nbRemain) res = CapyRatioAdd(res, remainVals[iRemain++]);

  // Return the result
  return res;
}

// Get the inverse of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to 1/that
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioInv(CapyRatio const x) {

  // Eliminate the trivial case of x equal to 0
  if(x.base == 0 && x.num == 0) return capyRatioNaN;

  // Variable to memorise the input eventually corrected to be positive
  CapyRatio y;
  if(x.base < 0) y = CapyRatioNeg(x);
  else y = x;

  // Trying to inverse NaN gives NaN
  if(CapyRatioIsNaN(y)) return capyRatioNaN;

  // Check for overflow
  CapyRatio threshold = {.base = 0, .num = 1, .den = INT64_MAX};
  if(CapyRatioCmp(y, threshold) < 0) return capyRatioNaN;

  // Variable to memorise the result
  CapyRatio res = capyRatioZero;

  // Variable to check for overflow of the denominator
  // Here y.base is guaranteed to be positive
  CapyPeasantMulDivRes pmd =
    CapyPeasantMulDiv((CapyRatioFrac_t)(y.base), y.den, UINT64_MAX);
  CapyRatio z = {.base = 0, .num = y.num, .den = UINT64_MAX};
  pmd.frac = CapyRatioAdd(pmd.frac, z);

  // If the result can be calculated directly
  if(pmd.base == 0 && pmd.frac.base == 0) {

    // Calculate the inverse
    res.num = y.den;
    res.den = (CapyRatioFrac_t)(y.base) * y.den + y.num;

  // Else the result can't be calculated directly, approximate it by
  // dichotomic search.
  } else {

    // Bounds of the dichotomic search.
    CapyRatio low = {.base = 0, .num = 0, .den = 1};
    CapyRatio high = {.base = 0, .num = 1, .den = 1};

    // Loop until convergence.
    while(
      low.den <= UINT64_MAX / 2 && high.den <= UINT64_MAX &&
      (low.num != high.num || low.den != high.den)
    ) {

      // Calculate the median
      CapyRatio a = low;
      a.den *= 2;
      CapyRatio b = high;
      b.den *= 2;
      CapyRatio ab = CapyRatioAdd(a, b);

      // If the median could not be calculated, end the search
      if(CapyRatioIsNaN(ab)) low = high;

      // Else the median could be calculated
      else {

        // Update the result
        res = ab;

        // Update the bounds. We can't calculate the value we are
        // approximating, but we have a comparison condition on the
        // median by multiplying it with the input. If the result is
        // greater than 1 the median is greater than the approximated
        // value, if it's lower than 1 it is lower than the approximated
        // value.
        CapyRatio mul = CapyRatioMul(y, res);
        if(res.num == low.num && res.den == low.den) low = high;
        else if(mul.base == 1 && mul.num == 0) low = high;
        else if(mul.base == 0) low = res;
        else high = res;
      }
    }
  }

  // Reduce the result
  res = CapyRatioReduce(res);

  // If the input was negative, correct the result
  if(x.base < 0) res = CapyRatioNeg(res);

  // Return the result
  return res;
}

// Divide two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x/y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioDiv(
  CapyRatio const x,
  CapyRatio const y) {

  // Reuse the inversion and multiplication
  return CapyRatioMul(x, CapyRatioInv(y));
}

// Get the absolute value of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to abs(x)
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioAbs(CapyRatio const x) {

  // Absolute value of NaN gives NaN
  if(CapyRatioIsNaN(x)) return capyRatioNaN;

  // Return the absolute value
  if(x.base < 0) return CapyRatioNeg(x);
  else return x;
}

// Raise a CapyRatio to an integer power
// Input:
//   x: the CapyRatio
//   n: the power
// Output:
//   Return a new CapyRatio (in reduced form) equal to x^n
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioPowi(
        CapyRatio const x,
  CapyRatioBase_t const n) {

  // NaN raised to a power is NaN
  if(CapyRatioIsNaN(x)) return capyRatioNaN;

  // Variable to memorise the result
  CapyRatio res;

  // Temporary variable for calculation
  CapyRatio y = (CapyRatio){1, 0, 1};
  CapyRatioBase_t m = n;

  // Trivial case
  if(n == 0) return y;

  // If the exponent is negative, correct the base and exponent and
  // use the same calculation as if the exponent was positive
  if(n < 0) {
    res = CapyRatioDiv(y, x);
    m *= -1;
  } else res = x;

  // Calculate the result
  while(m > 1) {
    if(m & 1) y = CapyRatioMul(y, res);
    res = CapyRatioMul(res, res);
    m /= 2;
  }
  res = CapyRatioMul(res, y);

  // Return the result
  return res;
}

// Get the square root of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to sqrt(x)
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioSqrt(CapyRatio const x) {

  // The square root of NaN is NaN
  if(CapyRatioIsNaN(x)) return capyRatioNaN;

  // The square root of a negative number is NaN
  if(x.base < 0) return capyRatioNaN;

  // Variable to memorise the result
  CapyRatio res = x;

  // We can estimate the square root of the input with the rounded
  // square root of the base
  res.base = (CapyRatioBase_t)sqrt((double)(res.base));

  // Heron's method
  CapyRatio diff = {INT64_MAX - 1, 0, 1};
  CapyRatio prevDiff = {INT64_MAX, 0, 1};
  int improvment = -1;
  while(CapyRatioIsNaN(res) == false && improvment < 0) {
    CapyRatio y = CapyRatioAdd(res, CapyRatioDiv(x, res));
    y = CapyRatioMul(y, (CapyRatio){0, 1, 2});
    prevDiff = diff;
    diff = CapyRatioAbs(CapyRatioSub(res, y));
    improvment = CapyRatioCmp(diff, prevDiff);
    if(improvment < 0) res = y;
  }

  // If the input is less than 1 and the result is NaN it means the
  // input was so small that the division in the Heron's method led to
  // NaN. In this case we can safely correct the result to zero
  if(CapyRatioIsNaN(res) && x.base == 0) res = capyRatioZero;

  // Return the result
  return res;
}
