// ---------------------------------- ratio.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_RATIO_H
#define CAPY_RATIO_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Rational number class.

// Format for print instructions
#define PRIratiofmt "%ld+%lu/%lu"
#define PRIratioval(a) (a).base, (a).num, (a).den

// Type for the base of a CapyRatio
typedef int64_t CapyRatioBase_t;

// Type for the fractional parts of a CapyRatio
typedef uint64_t CapyRatioFrac_t;

// Ratio object
typedef struct CapyRatio {

  // Components of the ratio
  CapyRatioBase_t base;
  CapyRatioFrac_t num;
  CapyRatioFrac_t den;
} CapyRatio;

#include "capymath.h"

// CapyRatio value for NaN
extern CapyRatio const capyRatioNaN;

// Predefined constants
extern CapyRatio const capyRatioZero;
extern CapyRatio const capyRatioPi;

// Create a CapyRatio equal to base+num/den
// Input:
//   base: the base
//   num: the numerator
//   den: the denominator
// Output:
//   Return a CapyRatio
CapyRatio CapyRatioCreate(
   CapyRatioBase_t const base,
   CapyRatioFrac_t const num,
   CapyRatioFrac_t const den);

// Allocate memory for a new CapyRatio and create it
// Input:
//   base: the base
//   num: the numerator
//   den: the denominator
// Output:
//   Return a CapyRatio
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRatio* CapyRatioAlloc(
   CapyRatioBase_t const base,
   CapyRatioFrac_t const num,
   CapyRatioFrac_t const den);

// Free the memory used by a CapyRatio
// Input:
//   that: the CapyRatio to free
void CapyRatioDestruct(CapyRatio* const that);

// Free the memory used by a CapyRatio* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRatio to free
void CapyRatioFree(CapyRatio** const that);

// Check if a rational is equal to NaN
// Input:
//   r: the CapyRatio to check
// Output:
//   Return true if the rational is equal to NaN (denominator equals to
//   0)
bool CapyRatioIsNaN(CapyRatio const r);

// Create a CapyRatio from a double
// Input:
//   a: the double
// Output:
//   Return a new CapyRatio representing the nearest possible value to
//   the input double.
CapyRatio CapyRatioFromDouble(double const a);

// Convert a CapyRatio to a double
// Input:
//   that: the CapyRatio to convert
// Output:
//   Return a double representing approximating the CapyRatio.
double CapyRatioToDouble(CapyRatio const that);

// Reduce a CapyRatio
// Input:
//   that: the CapyRatio to reduce
// Output:
//   Return a new CapyRatio equals to the reduced CapyRatio
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioReduce(CapyRatio const that);

// Add two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x+y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioAdd(
  CapyRatio const x,
  CapyRatio const y);

// Get the negative of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to -x
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioNeg(CapyRatio const x);

// Substract two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x-y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioSub(
  CapyRatio const x,
  CapyRatio const y);

// Compare two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return -1 if x<y, else 0 if x==y, else 1 if x>y
int8_t CapyRatioCmp(
  CapyRatio const x,
  CapyRatio const y);

// Multiply two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x*y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioMul(
  CapyRatio const x,
  CapyRatio const y);

// Get the inverse of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to 1/that
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioInv(CapyRatio const x);

// Divide two CapyRatios
// Input:
//   x: the first CapyRatio (must be in reduced form)
//   y: the second CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to x/y
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioDiv(
  CapyRatio const x,
  CapyRatio const y);

// Get the absolute value of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to abs(x)
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioAbs(CapyRatio const x);

// Raise a CapyRatio to an integer power
// Input:
//   x: the CapyRatio
//   n: the power
// Output:
//   Return a new CapyRatio (in reduced form) equal to x^n
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioPowi(
        CapyRatio const x,
  CapyRatioBase_t const n);

// Get the square root of a CapyRatio
// Input:
//   x: the CapyRatio (must be in reduced form)
// Output:
//   Return a new CapyRatio (in reduced form) equal to sqrt(x)
// Exception:
//   May raise CapyExc_NumericalOverflow
CapyRatio CapyRatioSqrt(CapyRatio const x);
#endif
