// ---------------------------------- rsacipher.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_RSACIPHER_H
#define CAPY_RSACIPHER_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// RSACipher class.

// RSACipher keys
typedef struct CapyRSACipherKey {
  uint64_t n;
  uint64_t e;
  uint64_t d;
} CapyRSACipherKey;

// RSA ciphered/deciphered data
typedef struct CapyRSACipherData {
  size_t size;
  uint8_t* data;
} CapyRSACipherData;

// RSACipher object
typedef struct CapyRSACipher {

  // Keys
  CapyRSACipherKey keys;

  // Destructor
  void (*destruct)(void);

  // Generate keys
  // Input:
  //   p: the first prime number to be used for key generation
  //   q: the second prime number to be used for key generation
  // Output:
  //   'that->keys' is updated.
  void (*generateKeys)(
    uint64_t const p,
    uint64_t const q);

  // Cipher a message
  // Input:
  //   message: the message to cipher
  // Output:
  //   Return the ciphered message.
  uint64_t (*cipher)(uint64_t const message);

  // Decipher a message
  // Input:
  //   message: the message to decipher
  // Output:
  //   Return the deciphered message.
  uint64_t (*decipher)(uint64_t const message);

  // Cipher a message divided into blocks of appropriate sizes
  // Input:
  //   message: the message to cipher
  // Output:
  //   Return the ciphered message.
  CapyRSACipherData (*cipherBlock)(CapyRSACipherData const message);

  // Decipher a message divided into blocks of appropriate sizes
  // Input:
  //   message: the message to decipher
  // Output:
  //   Return the deciphered message.
  CapyRSACipherData (*decipherBlock)(CapyRSACipherData const message);
} CapyRSACipher;

// Create a CapyRSACipher
// Output:
//   Return a CapyRSACipher
CapyRSACipher CapyRSACipherCreate(void);

// Allocate memory for a new CapyRSACipher and create it
// Output:
//   Return a CapyRSACipher
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRSACipher* CapyRSACipherAlloc(void);

// Free the memory used by a CapyRSACipher* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRSACipher to free
void CapyRSACipherFree(CapyRSACipher** const that);
#endif
