// ---------------------------- rulerandcompass.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2024 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "rulerandcompass.h"

// Definition of a dictionary of points
CapyDefDict(CapyRulerAndCompassPoints, CapyPoint2D)

// Definition of a dictionary of segments
CapyDefDict(CapyRulerAndCompassSegments, CapySegment)

// Definition of a dictionary of circles
CapyDefDict(CapyRulerAndCompassCircles, CapyCircle)

// Add a segment to the geometry
// Input:
//   label: label of the segment
//   pointA: the label of the first extremity of the segment
//   pointB: the label of the second extremity of the segment
// Output:
//   Add the segment to the geometry
static void AddSegment(
  char const* const label,
  char const* const pointA,
  char const* const pointB) {
  methodOf(CapyRulerAndCompass);

  // Get the two points
  CapyPoint2D a = $(that->points, get)(pointA);
  CapyPoint2D b = $(that->points, get)(pointB);

  // Create the segment
  CapySegment seg = CapySegmentCreate();
  loop(i, 2) {
    seg.points[0].coords[i] = a.coords[i];
    seg.points[1].coords[i] = b.coords[i];
  }

  // Add the segment to the dictionary
  $(that->segments, set)(label, seg);
}

// Add a circle to the geometry given two points
// Input:
//   label: label of the circle
//   center: the label of the center of the circle
//   point: the label of a point on the circle
// Output:
//   Add the circle to the geometry
static void AddCircle(
  char const* const label,
  char const* const center,
  char const* const point) {
  methodOf(CapyRulerAndCompass);

  // Get the two points
  CapyPoint2D a = $(that->points, get)(center);
  CapyPoint2D b = $(that->points, get)(point);

  // Create the circle
  CapyCircle circle = CapyCircleCreate();
  loop(i, 2) circle.center.coords[i] = a.coords[i];
  circle.radius =
    sqrt((a.x - b.x) * (a.x - b.x) + (a.y - b.y) * (a.y - b.y));

  // Add the circle to the dictionary
  $(that->circles, set)(label, circle);
}

// Add a circle to the geometry given three points
// Input:
//   label: label of the circle
//   center: the label of the center of the circle
//   pointA: the label of the first point defining the radius
//   pointB: the label of the second point defining the radius
// Output:
//   Add the circle to the geometry
static void AddCircleRadius(
  char const* const label,
  char const* const center,
  char const* const pointA,
  char const* const pointB) {
  methodOf(CapyRulerAndCompass);

  // Get the three points
  CapyPoint2D a = $(that->points, get)(center);
  CapyPoint2D b = $(that->points, get)(pointA);
  CapyPoint2D c = $(that->points, get)(pointB);

  // Create the circle
  CapyCircle circle = CapyCircleCreate();
  loop(i, 2) circle.center.coords[i] = a.coords[i];
  circle.radius =
    sqrt((c.x - b.x) * (c.x - b.x) + (c.y - b.y) * (c.y - b.y));

  // Add the circle to the dictionary
  $(that->circles, set)(label, circle);
}

// Add the point(s) of the intersection of two geometries
// Input:
//   labelA: label of the first intersection
//   labelB: label of the second intersection
//   geomA: the label of the first geometry
//   geomB: the label of the second geometry
// Output:
//   The point(s) at the intersection of the geometries are added. If there
//   is no intersection nothing happens. If there is only one intersection
//   'labelB' is ignored. Return the number of intersection.
static size_t AddPoint(
  char const* const labelA,
  char const* const labelB,
  char const* const geomA,
  char const* const geomB) {
  methodOf(CapyRulerAndCompass);

  // Create the intersection calculator
  CapyGeometry2DIntersection inter = CapyGeometry2DIntersectionCreate();

  // If the first geometry is a segment
  bool isSegmentA = $(that->segments, hasKey)(geomA);
  if(isSegmentA) {
    CapySegment segA = $(that->segments, get)(geomA);

    // If the second geometry is a segment
    bool isSegmentB = $(that->segments, hasKey)(geomB);
    if(isSegmentB) {
      CapySegment segB = $(that->segments, get)(geomB);

      // Get the intersection
      $(&inter, getIntersectSegments)(&segA, &segB, false);

    // Else, the second geometry must be a circle
    } else {
      CapyCircle circleB = $(that->circles, get)(geomB);

      // Get the intersection
      $(&inter, getIntersectSegmentCircle)(&segA, &circleB, false);
    }

  // Else, the first geometry must be a circle
  } else {
    CapyCircle circleA = $(that->circles, get)(geomA);

    // If the second geometry is a segment
    bool isSegmentB = $(that->segments, hasKey)(geomB);
    if(isSegmentB) {
      CapySegment segB = $(that->segments, get)(geomB);

      // Get the intersection
      $(&inter, getIntersectSegmentCircle)(&segB, &circleA, false);

    // Else, the second geometry must be a circle
    } else {
      CapyCircle circleB = $(that->circles, get)(geomB);

      // Get the intersection
      $(&inter, getIntersectCircles)(&circleA, &circleB);
    }
  }

  // Add the intersection to the dictionary of points
  if(inter.nbPoint > 0) {
    CapyPoint2D pointA = {.coords = {inter.points[0].x, inter.points[0].y}};
    $(that->points, set)(labelA, pointA);
    if(inter.nbPoint > 1) {
      CapyPoint2D pointB = {.coords = {inter.points[1].x, inter.points[1].y}};
      $(that->points, set)(labelB, pointB);
    }
  }

  // Memorise the number of points to be able to return it after the
  // intersection is destructed
  size_t const nbPoint = inter.nbPoint;

  // Free memory
  $(&inter, destruct)();

  // Return the number of intersection
  return nbPoint;
}

// Draw all the geometries
// Input:
//   img: the image on which to draw
// Output:
//   The geometries are drawn on 'img' using 'that->pen'.
static void DrawAll(CapyImg* const img) {
  methodOf(CapyRulerAndCompass);

  // If the point radius is not null
  if(that->radiusPoint >= 1.0) {

    // Circle for drawing the points
    CapyCircle circle = CapyCircleCreate();
    circle.radius = that->radiusPoint;

    // Loop on the points
    forEach(entry, that->points->iter) {

      // Draw the point
      loop(i, 2) circle.center.coords[i] = entry.val.coords[i];
      if(that->spline.destruct) $(&(that->spline), destruct)();
      that->spline = $((CapyGeometry2D*)(&circle), getBezierSpline)();
      $(&(that->pen), drawBezierSpline)(&(that->spline), img);
    }

    // Free memory
    $(&circle, destruct)();
  }

  // Loop on the segments
  forEach(entry, that->segments->iter) {

    // Draw the segment
    if(that->spline.destruct) $(&(that->spline), destruct)();
    that->spline = $((CapyGeometry2D*)(&(entry.val)), getBezierSpline)();
    $(&(that->pen), drawBezierSpline)(&(that->spline), img);
  }

  // Loop on the circles
  forEach(entry, that->circles->iter) {

    // Draw the circle
    if(that->spline.destruct) $(&(that->spline), destruct)();
    that->spline = $((CapyGeometry2D*)(&(entry.val)), getBezierSpline)();
    $(&(that->pen), drawBezierSpline)(&(that->spline), img);
  }
}

// Draw the requested geometry
// Input:
//   img: the image on which to draw
//   lbl: the label of the geometry
// Output:
//   The geometry 'lbl' are drawn on 'img' using 'that->pen'. If the label
//   is not found nothing happens.
static void DrawGeometry(
     CapyImg* const img,
  char const* const lbl) {
  methodOf(CapyRulerAndCompass);

  // If the point radius is not null
  if(that->radiusPoint >= 1.0) {

    // Circle for drawing the points
    CapyCircle circle = CapyCircleCreate();
    circle.radius = that->radiusPoint;

    // Loop on the points
    forEach(entry, that->points->iter) {

      // If it is the requested geometry
      if(strcmp(entry.key, lbl) == 0) {

        // Draw the point
        loop(i, 2) circle.center.coords[i] = entry.val.coords[i];
        if(that->spline.destruct) $(&(that->spline), destruct)();
        that->spline = $((CapyGeometry2D*)(&circle), getBezierSpline)();
        $(&(that->pen), drawBezierSpline)(&(that->spline), img);
      }
    }

    // Free memory
    $(&circle, destruct)();
  }

  // Loop on the segments
  forEach(entry, that->segments->iter) {

    // If it is the requested geometry
    if(strcmp(entry.key, lbl) == 0) {

      // Draw the segment
      if(that->spline.destruct) $(&(that->spline), destruct)();
      that->spline = $((CapyGeometry2D*)(&(entry.val)), getBezierSpline)();
      $(&(that->pen), drawBezierSpline)(&(that->spline), img);
    }
  }

  // Loop on the circles
  forEach(entry, that->circles->iter) {

    // If it is the requested geometry
    if(strcmp(entry.key, lbl) == 0) {

      // Draw the circle
      if(that->spline.destruct) $(&(that->spline), destruct)();
      that->spline = $((CapyGeometry2D*)(&(entry.val)), getBezierSpline)();
      $(&(that->pen), drawBezierSpline)(&(that->spline), img);
    }
  }
}

// Draw an arc of a circle between two given points
// Input:
//   img: the image on which to draw
//   lblCircle: the label of the circle
//   lblPointA: the label of the origin point
//   lblPointB: the label of the destination point
// Output:
//   Draw the arc of the circle between the projection of the points on the
//   circle, counter-clockwise. If the label are not found nothing happens.
static void DrawArc(
     CapyImg* const img,
  char const* const lblCircle,
  char const* const lblPointA,
  char const* const lblPointB) {
  methodOf(CapyRulerAndCompass);

  // If the three geometries exist
  bool const hasCircle = $(that->circles, hasKey)(lblCircle);
  bool const hasPointA = $(that->points, hasKey)(lblPointA);
  bool const hasPointB = $(that->points, hasKey)(lblPointB);
  if(hasCircle && hasPointA && hasPointB) {

    // Get the arc
    CapyCircle circle = $(that->circles, get)(lblCircle);
    CapyPoint2D pointA =$(that->points, get)(lblPointA);
    CapyPoint2D pointB =$(that->points, get)(lblPointB);
    if(that->spline.destruct) $(&(that->spline), destruct)();
    that->spline = $(&circle, getArcAsBezierSpline)(&pointA, &pointB);

    // Draw the arc
    $(&(that->pen), drawBezierSpline)(&(that->spline), img);
  }
}

// Free the memory used by a CapyRulerAndCompass
static void Destruct(void) {
  methodOf(CapyRulerAndCompass);
  $(&(that->pen), destruct)();
  CapyRulerAndCompassPointsFree(&(that->points));
  CapyRulerAndCompassCirclesFree(&(that->circles));
  CapyRulerAndCompassSegmentsFree(&(that->segments));
}

// Create a CapyRulerAndCompass
// Input:
//   pointA: the first initial point
//   pointB: the second initial point
// Output:
//   Return a CapyRulerAndCompass with 'pointA' and 'pointB' as initial
//   points for the geometry, respectively labeled 'a' and 'b'
CapyRulerAndCompass CapyRulerAndCompassCreate(
  CapyPoint2D const* const pointA,
  CapyPoint2D const* const pointB) {
  CapyPoint2D defaultPoint = {.coords = {0.0, 0.0}};
  CapySegment defaultSegment = CapySegmentCreate();
  CapyCircle defaultCircle = CapyCircleCreate();
  CapyRulerAndCompass that = {
    .pen = CapyPenCreate(),
    .radiusPoint = 0.0,
    .points = CapyRulerAndCompassPointsAlloc(
      CAPY_RULERANDCOMPASS_SIZE_HASH, defaultPoint),
    .segments = CapyRulerAndCompassSegmentsAlloc(
      CAPY_RULERANDCOMPASS_SIZE_HASH, defaultSegment),
    .circles = CapyRulerAndCompassCirclesAlloc(
      CAPY_RULERANDCOMPASS_SIZE_HASH, defaultCircle),
    .destruct = Destruct,
    .addSegment = AddSegment,
    .addCircle = AddCircle,
    .addCircleRadius = AddCircleRadius,
    .addPoint = AddPoint,
    .drawAll = DrawAll,
    .drawGeometry = DrawGeometry,
    .drawArc = DrawArc,
  };
  $(that.points, set)("A", *pointA);
  $(that.points, set)("B", *pointB);
  that.spline = (CapyBezierSpline){0};
  return that;
}

// Allocate memory for a new CapyRulerAndCompass and create it
// Input:
//   pointA: the first initial point
//   pointB: the second initial point
// Output:
//   Return a CapyRulerAndCompass with 'pointA' and 'pointB' as initial
//   points for the geometry, respectively labeled 'a' and 'b'
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRulerAndCompass* CapyRulerAndCompassAlloc(
  CapyPoint2D const* const pointA,
  CapyPoint2D const* const pointB) {
  CapyRulerAndCompass* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyRulerAndCompassCreate(pointA, pointB);
  return that;
}

// Free the memory used by a CapyRulerAndCompass* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRulerAndCompass to free
void CapyRulerAndCompassFree(CapyRulerAndCompass** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
