// ---------------------------------- rulerandcompass.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2024 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_RULERANDCOMPASS_H
#define CAPY_RULERANDCOMPASS_H
#include "externalHeaders.h"
#include "cext.h"
#include "geometricShape.h"
#include "dict.h"
#include "pen.h"
#include "image.h"

// Description:
// RulerAndCompass class.

// Declaration of a dictionary of points
CapyDecDict(CapyRulerAndCompassPoints, CapyPoint2D)

// Declaration of a dictionary of segments
CapyDecDict(CapyRulerAndCompassSegments, CapySegment)

// Declaration of a dictionary of circles
CapyDecDict(CapyRulerAndCompassCircles, CapyCircle)

// Size of the hash table in dictionaries of points, circles, and segments
#define CAPY_RULERANDCOMPASS_SIZE_HASH 100

// RulerAndCompass object
typedef struct CapyRulerAndCompass {

  // Pen for drawing
  CapyPen pen;

  // Size for drawing points (default: 0.0, points are not drawn if the radius
  // is less than 1.0)
  double radiusPoint;

  // Dictionary of points
  CapyRulerAndCompassPoints* points;

  // Dictionary of segments
  CapyRulerAndCompassSegments* segments;

  // Dictionary of circles
  CapyRulerAndCompassCircles* circles;

  // Last drawn geometry as a bezier spline
  CapyBezierSpline spline;

  // Destructor
  void (*destruct)(void);

  // Add a segment to the geometry
  // Input:
  //   label: label of the segment
  //   pointA: the label of the first extremity of the segment
  //   pointB: the label of the second extremity of the segment
  // Output:
  //   Add the segment to the geometry
  void (*addSegment)(
    char const* const label,
    char const* const pointA,
    char const* const pointB);

  // Add a circle to the geometry given two points
  // Input:
  //   label: label of the circle
  //   center: the label of the center of the circle
  //   point: the label of a point on the circle
  // Output:
  //   Add the circle to the geometry
  void (*addCircle)(
    char const* const label,
    char const* const center,
    char const* const point);

  // Add a circle to the geometry given three points
  // Input:
  //   label: label of the circle
  //   center: the label of the center of the circle
  //   pointA: the label of the first point defining the radius
  //   pointB: the label of the second point defining the radius
  // Output:
  //   Add the circle to the geometry
  void (*addCircleRadius)(
    char const* const label,
    char const* const center,
    char const* const pointA,
    char const* const pointB);

  // Add the point(s) of the intersection of two geometries
  // Input:
  //   labelA: label of the first intersection
  //   labelB: label of the second intersection
  //   geomA: the label of the first geometry
  //   geomB: the label of the second geometry
  // Output:
  //   The point(s) at the intersection of the geometries are added. If there
  //   is no intersection nothing happens. If there is only one intersection
  //   'labelB' is ignored. Return the number of intersection
  size_t (*addPoint)(
    char const* const labelA,
    char const* const labelB,
    char const* const geomA,
    char const* const geomB);

  // Draw all the geometries
  // Input:
  //   img: the image on which to draw
  // Output:
  //   The geometries are drawn on 'img' using 'that->pen'.
  void (*drawAll)(CapyImg* const img);

  // Draw the requested geometry
  // Input:
  //   img: the image on which to draw
  //   lbl: the label of the geometry
  // Output:
  //   The geometry 'lbl' are drawn on 'img' using 'that->pen'. If the label
  //   is not found nothing happens.
  void (*drawGeometry)(
       CapyImg* const img,
    char const* const lbl);

  // Draw an arc of a circle between two given points
  // Input:
  //   img: the image on which to draw
  //   lblCircle: the label of the circle
  //   lblPointA: the label of the origin point
  //   lblPointB: the label of the destination point
  // Output:
  //   Draw the arc of the circle between the projection of the points on the
  //   circle, counter-clockwise. If the label are not found nothing happens.
  void (*drawArc)(
       CapyImg* const img,
    char const* const lblCircle,
    char const* const lblPointA,
    char const* const lblPointB);
} CapyRulerAndCompass;

// Create a CapyRulerAndCompass
// Input:
//   pointA: the first initial point
//   pointB: the second initial point
// Output:
//   Return a CapyRulerAndCompass with 'pointA' and 'pointB' as initial
//   points for the geometry, respectively labeled 'a' and 'b'
CapyRulerAndCompass CapyRulerAndCompassCreate(
  CapyPoint2D const* const pointA,
  CapyPoint2D const* const pointB);

// Allocate memory for a new CapyRulerAndCompass and create it
// Input:
//   pointA: the first initial point
//   pointB: the second initial point
// Output:
//   Return a CapyRulerAndCompass with 'pointA' and 'pointB' as initial
//   points for the geometry, respectively labeled 'a' and 'b'
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRulerAndCompass* CapyRulerAndCompassAlloc(
  CapyPoint2D const* const pointA,
  CapyPoint2D const* const pointB);

// Free the memory used by a CapyRulerAndCompass* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRulerAndCompass to free
void CapyRulerAndCompassFree(CapyRulerAndCompass** const that);
#endif
