// ---------------------------------- rungekutta.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_RUNGEKUTTA_H
#define CAPY_RUNGEKUTTA_H
#include "externalHeaders.h"
#include "cext.h"
#include "mathfun.h"

// Description:
// RungeKutta implementation.

// RungeKutta object
typedef struct CapyRungeKutta {

  // Derivative function F().
  // For first order: F(t, X) = X'
  // For second order: F(t, X, X') = X''
  CapyMathFun* derivative;

  // Order of the derivative (in {1,2}), used to interpret the shape of X in the
  // derivative function
  size_t order;

  // Number of variables (automatically set based on derivation.nbInput and
  // order)
  size_t dimVar;

  // Initial values [t0, X0] or [t0, X0, X'0] (default: all 0)
  CapyVec initVal;

  // Temporary vectors for calculation
  CapyVec vecs[5];

  // Discretise step delta t (default: 1e-3)
  double deltaT;

  // Approximate X(t) using RK4
  // Input:
  //   t: value for which we want to approximate X(t)
  //   vals: values [t0, X(t0)] or [t0, X(t0), X'(t0)]
  // Output:
  //   Update 'vals' with the approximated values at 't'
  void (*eval)(
     double const t,
    double* const vals);

  // Run one step
  // Input:
  //   vals: current values [t0, X(t0)] or [t0, X(t0), X'(t0)]
  // Output:
  //   'vals' is updated with the result of one step of RK4
  void (*step)(double* const vals);

  // Destructor
  void (*destruct)(void);
} CapyRungeKutta;

// Create a CapyRungeKutta
// Input:
//   derivative: the derivative function
//   order: order of the derivative (in {1,2})
// Output:
//   Return a CapyRungeKutta
CapyRungeKutta CapyRungeKuttaCreate(
  CapyMathFun* const derivative,
        size_t const order);

// Allocate memory for a new CapyRungeKutta and create it
// Input:
//   derivative: the derivative function
//   order: order of the derivative (in {1,2})
// Output:
//   Return a CapyRungeKutta
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRungeKutta* CapyRungeKuttaAlloc(
  CapyMathFun* const derivative,
        size_t const order);

// Free the memory used by a CapyRungeKutta* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRungeKutta to free
void CapyRungeKuttaFree(CapyRungeKutta** const that);
#endif
