// --------------------------------- sampling.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "sampling.h"

// Array of samples
CapyDefArray(CapySamples, CapySample)

// Free the memory used by a CapySamplw
static void DestructSample(void) {
  methodOf(CapySample);
  free(that->vals);
  that->vals = NULL;
}

// Create a CapySample
// Input:
//    dim: dimension of the sample
// Output:
//   Return a CapySample
CapySample CapySampleCreate(size_t const dim) {
  CapySample that = {
    .destruct = DestructSample
  };
  safeMalloc(that.vals, dim);
  loop(i, dim) that.vals[i] = 0.0;
  return that;
}

// Get samples
// Output:
//   Return a new array of samples
static CapySamples* GetSamples(void) {
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Sampling.getSamples is undefined.");
  return NULL;
}

// Free the memory used by a CapySampling
static void Destruct(void) {
  methodOf(CapySampling);
  $(&(that->rng), destruct)();
  loop(i, that->dim) $(that->domains + i, destruct)();
  free(that->domains);
}

// Create a CapySampling
// Input:
//   seed: seed for the random number generator
//    dim: dimension of the samples
// Output:
//   Return a CapySampling
CapySampling CapySamplingCreate(
  CapyRandomSeed_t const seed,
            size_t const dim) {

  // Create the Sampling
  CapySampling that = {
    .rng = CapyRandomCreate(seed),
    .dim = dim,
  };
  safeMalloc(that.domains, dim);
  loop(i, that.dim) that.domains[i] = CapyRangeDoubleCreate(0.0, 1.0);
  that.destruct = Destruct;
  that.getSamples = GetSamples;
  return that;
}

// Allocate memory for a new CapySampling and create it
// Input:
//   seed: seed for the random number generator
//   dim: dimension of the samples
// Output:
//   Return a CapySampling
// Exception:
//   May raise CapyExc_MallocFailed.
CapySampling* CapySamplingAlloc(
  CapyRandomSeed_t const seed,
            size_t const dim) {

  // Allocate memory and create the new CapySampling
  CapySampling* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  CapySampling b = CapySamplingCreate(seed, dim);
  memcpy(that, &b, sizeof(CapySampling));

  // Return the new CapySampling
  return that;
}

// Free the memory used by a CapySampling* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySampling to free
void CapySamplingFree(CapySampling** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
