// --------------------------------- sampling.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_SAMPLING_H
#define CAPY_SAMPLING_H
#include "externalHeaders.h"
#include "cext.h"
#include "array.h"
#include "random.h"
#include "range.h"

// Description:
// Sampling function parent class.

// Sample structure
typedef struct CapySample {

  // Values of the sample
  double* vals;

  // Destructor
  void (*destruct)(void);
} CapySample;

// Create a CapySample
// Input:
//    dim: dimension of the sample
// Output:
//   Return a CapySample
CapySample CapySampleCreate(size_t const dim);

// Array of samples
CapyDecArray(CapySamples, CapySample)

// CapySampling object definition macro
#define CapySamplingDef struct {    \
  CapyRandom rng;                   \
  size_t dim;                       \
  CapyRangeDouble* domains;         \
  void (*destruct)(void);           \
  CapySamples* (*getSamples)(void); \
}

// CapySampling object
typedef CapySamplingDef CapySampling;

// Create a CapySampling
// Input:
//   seed: seed for the random number generator
//    dim: dimension of the samples
// Output:
//   Return a CapySampling
CapySampling CapySamplingCreate(
  CapyRandomSeed_t const seed,
            size_t const dim);

// Allocate memory for a new CapySampling and create it
// Input:
//   seed: seed for the random number generator
//   dim: dimension of the samples
// Output:
//   Return a CapySampling
// Exception:
//   May raise CapyExc_MallocFailed.
CapySampling* CapySamplingAlloc(
  CapyRandomSeed_t const seed,
            size_t const dim);

// Free the memory used by a CapySampling* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySampling to free
void CapySamplingFree(CapySampling** const that);
#endif
