// ---------------------------- SdfCsg.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "sdfcsg.h"

// Epsilon value for raymarching and normal calculation
static double sdfEpsilon = 1e-6;

// Getter and setter for the epsilon value
double CapySdfGetEpsilon(void) {
  return sdfEpsilon;
}

void CapySdfSetEpsilon(double epsilon) {
  sdfEpsilon = epsilon;
}

// Maximum number of step (to avoid infinite loop) during ray marching.
static size_t sdfRayMarchNbMaxStep = 1000;
size_t CapySdfGetRayMarchNbMaxStep(void) {return sdfRayMarchNbMaxStep;}
void CapySdfSetRayMarchNbMaxStep(size_t const nb) {
  sdfRayMarchNbMaxStep = nb;
}

// Helper functions for the signed distance field definitions
static inline double SdfLength(double const c[3]) {
  return sqrt(c[0] * c[0] + c[1] * c[1] + c[2] * c[2]);
}

static inline void SdfAbs3(
  double const c[3],
        double r[3]) {
  loop(i, 3) r[i] = fabs(c[i]);
}

static inline void SdfMin3(
  double const ca[3],
  double const cb[3],
        double r[3]) {
  loop(i, 3) r[i] = (ca[i] < cb[i] ? ca[i] : cb[i]);
}

static inline void SdfMax3(
  double const ca[3],
  double const cb[3],
        double r[3]) {
  loop(i, 3) r[i] = (ca[i] > cb[i] ? ca[i] : cb[i]);
}

static inline double SdfMin(
  double const a,
  double const b) {
  return (a < b ? a : b);
}

static inline double SdfMax(
  double const a,
  double const b) {
  return (a > b ? a : b);
}

// Apply the inverse of a quaternion to a vector
// Inputs:
//   quat: the quaternion
//      u: the input vector
//      v: the output vector (can be same vector as u)
// Output:
//   v is updated.
static void SdfApplyRotationInverse(
  double const quat[4],
  double const u[3],
        double v[3]) {
  double p[4];
  loop(i, 3) p[i] = u[i];
  p[3] = 0.0;
  double w[4];
  w[0] = p[3] * -quat[0] + p[0] * quat[3] - p[1] * -quat[2] + p[2] * -quat[1];
  w[1] = p[3] * -quat[1] + p[0] * -quat[2] + p[1] * quat[3] - p[2] * -quat[0];
  w[2] = p[3] * -quat[2] - p[0] * -quat[1] + p[1] * -quat[0] + p[2] * quat[3];
  w[3] = p[3] * quat[3] - p[0] * -quat[0] - p[1] * -quat[1] - p[2] * -quat[2];
  loop(i, 3) p[i] = quat[i];
  p[3] = quat[3];
  v[0] = p[3] * w[0] + p[0] * w[3] - p[1] * w[2] + p[2] * w[1];
  v[1] = p[3] * w[1] + p[0] * w[2] + p[1] * w[3] - p[2] * w[0];
  v[2] = p[3] * w[2] - p[0] * w[1] + p[1] * w[0] + p[2] * w[3];
}

// Apply the inverse transformation of a shape to a position
// Inputs:
//   that: the shape
//   pos: the position
//   res: the result
// Output:
//   res is updated
static void SdfUntransform(
  CapySdf const* const that,
          double const pos[3],
                double res[3]) {
  loop(i, 3) res[i] = pos[i];
  for(ssize_t iTrans = CAPY_MAX_NB_TRANSFORM_SDF - 1; iTrans >= 0; --iTrans) {
    CapySdfTransform const* trans = that->transforms + iTrans;
    double theta;
    double quat[4];
    switch(trans->type) {
      case capySdfTransformType_translate:
        loop(i, 3) res[i] -= trans->vals[i];
        break;
      case capySdfTransformType_scale:
        loop(i, 3) res[i] /= trans->vals[0];
        break;
      case capySdfTransformType_rotate:
        SdfApplyRotationInverse(trans->vals, res, res);
        break;
      case capySdfTransformType_extrusion:
        if(res[2] < 0) res[2] = SdfMin(0, res[2] + trans->vals[0]);
        else res[2] = SdfMax(0, res[2] - trans->vals[0]);
        break;
      case capySdfTransformType_symmetry:
        res[0] = fabs(res[0]);
        break;
      case capySdfTransformType_revolution:
        res[0] = sqrt(res[0] * res[0] + res[2] * res[2]);
        res[2] = 0.0;
        break;
      case capySdfTransformType_twist:
        theta = 2.0 * M_PI * res[1] / trans->vals[0];
        quat[0] = 0.0;
        quat[1] = sin(0.5 * theta);
        quat[2] = 0.0;
        quat[3] = cos(0.5 * theta);
        SdfApplyRotationInverse(quat, res, res);
        break;
      default:
        break;
    }
  }
}

// Apply a translation to the shape
// Inputs:
//   iTrans: the index of the transformation
//   u: the translation
static void SdfTranslate(
  size_t const iTrans,
  double const u[3]) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_translate;
    loop(i, 3) that->transforms[iTrans].vals[i] = u[i];
  }
}

// Apply a rotation to the shape
// Inputs:
//   iTrans: the index of the transformation
//   u: the rotation axis
//   theta: the rotation angle in radians
static void SdfRotate(
  size_t const iTrans,
  double const u[3],
  double const theta) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_rotate;
    double n = SdfLength(u);
    double s = sin(0.5 * theta);
    loop(i, 3) that->transforms[iTrans].vals[i] = u[i] / n * s;
    that->transforms[iTrans].vals[3] = cos(0.5 * theta);
  }
}

// Apply a scaling to the shape
// Inputs:
//   iTrans: the index of the transformation
//   s: the scaling factor
static void SdfScale(
  size_t const iTrans,
  double const s) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_scale;
    that->transforms[iTrans].vals[0] = s;
  }
}

// Apply an extrusion (along the z axis, relative to the origin) to the shape
// Inputs:
//   iTrans: the index of the transformation
//   l: the extrusion length
static void SdfExtrude(
  size_t const iTrans,
  double const l) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_extrusion;
    that->transforms[iTrans].vals[0] = l * 0.5;
  }
}

// Apply a symmetry (duplicate the x>=0 half-space relative to the yz plane)
// to the shape
// Inputs:
//   iTrans: the index of the transformation
static void SdfSymmetry(size_t const iTrans) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_symmetry;
  }
}

// Apply a revolution (360 degrees rotation around y of the profile in the
// xy x>=0 half-plane) to the shape
// Inputs:
//   iTrans: the index of the transformation
static void SdfRevolution(size_t const iTrans) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_revolution;
  }
}

// Apply a twist (around y) to the shape
// Inputs:
//   iTrans: the index of the transformation
//   freq: the frequency of the twist (one rotation per 'freq' unit along y,
//         twist clockwise with +y)
// Output:
//   The transformation is set and will be applied when using the 'dist'
//   method.
static void SdfTwist(
  size_t const iTrans,
  double const freq) {
  methodOf(CapySdf);
  if(iTrans < CAPY_MAX_NB_TRANSFORM_SDF) {
    that->transforms[iTrans].type = capySdfTransformType_twist;
    that->transforms[iTrans].vals[0] = freq;
  }
}

// Get the value at a given position in a signed distance field for a shape
// Inputs:
//   pos: the position
// Output:
//   Return the distance, less or equal to the exact distance.
//   A positive distance indicates a posiiton outside the shape. A negative
//   distance indicates a position inside the shape.
static double SdfDist(double const pos[3]) {
  methodOf(CapySdf);

  // Get the position in the untransformed shape
  double posUntransformed[3];
  SdfUntransform((CapySdf*)that, pos, posUntransformed);

  // Get the distance
  double dist = $(that, distCanonical)(that, posUntransformed);

  // Account for the scaling
  loop(i, CAPY_MAX_NB_TRANSFORM_SDF) {
    if(that->transforms[i].type == capySdfTransformType_scale) {
      dist *= that->transforms[i].vals[0];
    }
  }

  // Apply thickness and roundness
  dist -= that->roundness;
  if(that->thickness > 1e-9) dist = fabs(dist) - that->thickness;

  // Return the distance
  return dist;
}

// Get the value at a given position in a signed distance field for a box
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistBox(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double q[3];
  SdfAbs3(pos, q);
  loop(i, 3) q[i] -= that->params[capySdfParam_halfWidth + i];
  double maxQ[3];
  double const sdfOrig[3] = {0.0, 0.0, 0.0};
  SdfMax3(q, sdfOrig, maxQ);
  double dist = SdfLength(maxQ) + SdfMin(SdfMax(q[0], SdfMax(q[1], q[2])), 0.0);

  // Return the distance
  return dist;
}

// Setter and getter for the user data
static void* SdfGetData(void) {
  methodOf(CapySdf);
  return that->data;
}

static void SdfSetData(void* const data) {
  methodOf(CapySdf);
  that->data = data;
}

// Normal at a given position (as the gradient in the field at that position)
// Input:
//   pos: the given position
//   normal: the result normal
// Output:
//   'normal' is updated, normalised, oriented away from the shape.
static void SdfNormal(
  double const pos[3],
        double normal[3]) {
  methodOf(CapySdf);
  loop(i, 3) {
    double q[3];
    loop(j, 3) q[j] = pos[j];
    q[i] = pos[i] + sdfEpsilon;
    normal[i] = $(that, dist)(q);
    q[i] = pos[i] - sdfEpsilon;
    normal[i] -= $(that, dist)(q);
  }
  double n = 1.0 / SdfLength(normal);
  loop(i, 3) normal[i] *= n;
}

// Setter and getter for the shape parameters
static void SdfSetParam(
  CapySdfParam const iParam,
        double const val) {
  methodOf(CapySdf);
  that->params[iParam] = val;
}

static double SdfGetParam(CapySdfParam const iParam) {
  methodOf(CapySdf);
  return that->params[iParam];
}

// Search the intersection between a ray and the shape
// Inputs:
//   from: origin of the ray
//   dir: direction of the ray, normalised
//   res: result of the search
// Output:
//   If there is an intersection nearer than the current one in 'res', 'res'
//   is updated, else it is left unchanged.
static void SdfRayMarching(
               double const from[3],
               double const dir[3],
  CapyRayMarchingRes* const res) {
  methodOf(CapySdf);
  if(that->type == capySdf_union) {
    loop(i, 2) $(that->comps[i], rayMarching)(from, dir, res);
  } else {
    double pos[3];
    loop(i, 3) pos[i] = from[i];
    double dist = dist = $(that, dist)(pos);
    double sumDist = dist;
    bool flagHit = false;
    res->hasGaveUp = false;
    size_t nbStep = 0;
    while(
      flagHit == false &&
      nbStep < sdfRayMarchNbMaxStep &&
      fabs(sumDist) < fabs(res->dist)
    ) {
      if(fabs(dist) > sdfEpsilon) {
        ++nbStep;
        if(fabs(res->distMin) > fabs(dist)) res->distMin = dist;
        loop(i, 3) pos[i] += dist * dir[i];
        dist = $(that, dist)(pos);
        sumDist += dist;
      } else {
        flagHit = true;
        res->flagIntersect = true;
        res->dist = sumDist;
        if(fabs(res->distMin) > fabs(dist)) res->distMin = dist;
        res->shape = that;
        loop(i, 3) res->pos[i] = pos[i] + dist * dir[i];
        SdfUntransform(that, pos, res->posLocal);
        $(that, normal)(pos, res->normal);
      }
    }
    res->hasGaveUp = (nbStep >= sdfRayMarchNbMaxStep);
  }
}

// Free the memory used by a CapySdf
static void SdfDestruct(void) {
  methodOf(CapySdf);
  loop(i, 2) if(that->comps[i] != NULL) CapySdfFree(that->comps + i);
}

// Create a CapySdf (by defaut a unit box)
// Output:
//   Return a CapySdf
CapySdf CapySdfCreate(void) {
  CapySdf that = {
    .type = capySdf_box,
    .thickness = 0.0,
    .roundness = 0.0,
    .data = NULL,
    .comps = {NULL, NULL},
    .parent = NULL,
    .smoothness = 0.0,
    .destruct = SdfDestruct,
    .dist = SdfDist,
    .distCanonical = SdfDistBox,
    .translate = SdfTranslate,
    .scale = SdfScale,
    .extrude = SdfExtrude,
    .symmetry = SdfSymmetry,
    .revolution = SdfRevolution,
    .twist = SdfTwist,
    .rotate = SdfRotate,
    .setData = SdfSetData,
    .getData = SdfGetData,
    .normal = SdfNormal,
    .getParam = SdfGetParam,
    .setParam = SdfSetParam,
    .rayMarching = SdfRayMarching,
  };
  loop(i, CAPY_MAX_NB_TRANSFORM_SDF) {
    that.transforms[i] =
      (CapySdfTransform){ .type = capySdfTransformType_none };
  }
  loop(i, capySdfParam_nbParam) that.params[i] = 0.0;
  that.params[capySdfParam_halfWidth] = 0.5;
  that.params[capySdfParam_halfHeight] = 0.5;
  that.params[capySdfParam_halfDepth] = 0.5;
  return that;
}

// Allocate memory for a new CapySdf and create it (by defaut a unit box)
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAlloc(void) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreate();
  return that;
}

// Free the memory used by a CapySdf* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySdf to free
void CapySdfFree(CapySdf** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Create a CapySdf for a box of given dimensions
// Inputs:
//   width: width of the box (x axis)
//   height: height of the box (y axis)
//   depth: depth of the box (z axis)
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateBox(
  double const width,
  double const height,
  double const depth) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_box;
  that.params[capySdfParam_halfWidth] = width * 0.5;
  that.params[capySdfParam_halfHeight] = height * 0.5;
  that.params[capySdfParam_halfDepth] = depth * 0.5;
  that.distCanonical = SdfDistBox;
  return that;
}

// Allocate memory for a new CapySdf for a box of given dimensions and create it
// Inputs:
//   width: width of the box (x axis)
//   height: height of the box (y axis)
//   depth: depth of the box (z axis)
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocBox(
  double const width,
  double const height,
  double const depth) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateBox(width, height, depth);
  return that;
}

// Get the value at a given position in a signed distance field for a plane
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistPlane(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double dist =
    pos[0] * that->params[capySdfParam_normX] +
    pos[1] * that->params[capySdfParam_normY] +
    pos[2] * that->params[capySdfParam_normZ] -
    that->params[capySdfParam_dist];

  // Return the distance
  return dist;
}

// Create a CapySdf for a plane
// Inputs:
//   norm: normal of the plane (toward the 'out' side of the plane)
//   dist: distance from the origin in the direction of the normal
// Output:
//   Return a CapySdf
CapySdf CapySdfCreatePlane(
  double const norm[3],
  double const dist) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_plane;
  double l = 1.0 / SdfLength(norm);
  loop(i, 3) that.params[capySdfParam_normX + i] = norm[i] * l;
  that.params[capySdfParam_dist] = dist;
  that.distCanonical = SdfDistPlane;
  return that;
}

// Allocate memory for a new CapySdf for a plane and create it
// Inputs:
//   norm: normal of the plane (toward the 'out' side of the plane)
//   dist: distance from the origin in the direction of the normal
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocPlane(
  double const norm[3],
  double const dist) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreatePlane(norm, dist);
  return that;
}

// Get the value at a given position in a signed distance field for a sphere
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistSphere(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double dist = SdfLength(pos) - that->params[capySdfParam_radius];

  // Return the distance
  return dist;
}

// Create a CapySdf for a sphere of given dimensions
// Inputs:
//   radius: radius of the sphere
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateSphere(double const radius) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_sphere;
  that.params[capySdfParam_radius] = radius;
  that.distCanonical = SdfDistSphere;
  return that;
}

// Allocate memory for a new CapySdf for a sphere of given dimensions and
// create it
// Inputs:
//   radius: radius of the sphere
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocSphere(double const radius) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateSphere(radius);
  return that;
}

// Get the value at a given position in a signed distance field for a cone
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistCone(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double q[2] = {sqrt(pos[0] * pos[0] + pos[2] * pos[2]), pos[1]};
  double k1[2] = {
    that->params[capySdfParam_topRadius],
    that->params[capySdfParam_height]
  };
  double k2[2] = {
    that->params[capySdfParam_topRadius] -
    that->params[capySdfParam_baseRadius],
    2.0 * that->params[capySdfParam_height]
  };
  double ca[2] = {
    q[0] - SdfMin(
      q[0],
      (
        q[1] < 0.0 ?
        that->params[capySdfParam_baseRadius] :
        that->params[capySdfParam_topRadius]
      )),
    fabs(q[1]) - that->params[capySdfParam_height]
  };
  double cb[2] = {0};
  loop(i, 2) {
    cb[i] = q[i] - k1[i];
    double k1q[2] = {k1[0] - q[0], k1[1] - q[1]};
    double c =
      (k1q[0] * k2[0] + k1q[1] * k2[1]) / (k2[0] * k2[0] + k2[1] * k2[1]);
    if(c < 0.0) c = 0.0; else if(c > 1.0) c = 1.0;
    cb[i] += k2[i] * c;
  }
  double s = ((cb[0] < 0.0 && ca[1] < 0.0) ? -1.0 : 1.0);
  double dist =
    s *
    sqrt(SdfMin(ca[0] * ca[0] + ca[1] * ca[1], cb[0] * cb[0] + cb[1] * cb[1]));

  // Return the distance
  return dist;
}

// Create a CapySdf for a cone (center at the origin, rotation
// axis along the y axis, top toward +y) of given dimensions and create it
// Inputs:
//   radius: radius of the sphere
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateCone(
  double const baseRadius,
  double const topRadius,
  double const height) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_cone;
  that.params[capySdfParam_baseRadius] = baseRadius;
  that.params[capySdfParam_topRadius] = topRadius;
  that.params[capySdfParam_height] = height;
  that.distCanonical = SdfDistCone;
  return that;
}


// Allocate memory for a new CapySdf for a cone (center at the origin, rotation
// axis along the y axis, top toward +y) of given dimensions and create it
// Inputs:
//   baseRadius: radius of the cone at
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocCone(
  double const baseRadius,
  double const topRadius,
  double const height) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateCone(baseRadius, topRadius, height);
  return that;
}


// Get the value at a given position in a signed distance field for a ellipsoid
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistEllipsoid(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double q[3] = {0.0};
  loop(i, 3) q[i] = pos[i] / that->params[capySdfParam_radiusX + i];
  double l = SdfLength(q);
  double r[3] = {0.0};
  loop(i, 3) r[i] = pos[i] - q[i] / l * that->params[capySdfParam_radiusX + i];
  double dist = (l > 1.0 ? SdfLength(r) : -SdfLength(r));

  // Return the distance
  return dist;
}

// Create a CapySdf for an ellipsoid of given dimensions (aligned with
// x, y,z axis)
// Inputs:
//   radiusX: radius of the ellipsoid along the x axis
//   radiusY: radius of the ellipsoid along the y axis
//   radiusZ: radius of the ellipsoid along the z axis
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateEllipsoid(
  double const radiusX,
  double const radiusY,
  double const radiusZ) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_ellipsoid;
  that.params[capySdfParam_radiusX] = radiusX;
  that.params[capySdfParam_radiusY] = radiusY;
  that.params[capySdfParam_radiusZ] = radiusZ;
  that.distCanonical = SdfDistEllipsoid;
  return that;
}

// Allocate memory for a new CapySdf for an ellipsoid of given dimensions
// (aligned with x, y,z axis) and create it
// Inputs:
//   radiusX: radius of the ellipsoid along the x axis
//   radiusY: radius of the ellipsoid along the y axis
//   radiusZ: radius of the ellipsoid along the z axis
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocEllipsoid(
  double const radiusX,
  double const radiusY,
  double const radiusZ) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateEllipsoid(radiusX, radiusY, radiusZ);
  return that;
}

// Get the value at a given position in a signed distance field for a torus
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistTorus(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double q[3];
  q[0] =
    sqrt(pos[0] * pos[0] + pos[2] * pos[2]) -
    that->params[capySdfParam_majorRadius];
  q[1] = pos[1];
  q[2] = 0.0;
  double dist = SdfLength(q) - that->params[capySdfParam_minorRadius];

  // Return the distance
  return dist;
}

// Create a CapySdf for a torus of given dimensions (in xz plane)
// Inputs:
//   majorRadius: major radius of the torus
//   minorRadius: minor radius of the torus
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateTorus(
  double const majorRadius,
  double const minorRadius) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_torus;
  that.params[capySdfParam_majorRadius] = majorRadius;
  that.params[capySdfParam_minorRadius] = minorRadius;
  that.distCanonical = SdfDistTorus;
  return that;
}

// Allocate memory for a new CapySdf for a torus of given dimensions (in xz
// plane) and create it
// Inputs:
//   majorRadius: major radius of the torus
//   minorRadius: minor radius of the torus
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocTorus(
  double const majorRadius,
  double const minorRadius) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateTorus(majorRadius, minorRadius);
  return that;
}

// Get the value at a given position in a signed distance field for a capsule
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistCapsule(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double q[3];
  q[0] = pos[0];
  if(pos[1] < 0) {
    q[1] = SdfMin(0, pos[1] + that->params[capySdfParam_halfLength]);
  } else {
    q[1] = SdfMax(0, pos[1] - that->params[capySdfParam_halfLength]);
  }
  q[2] = pos[2];
  double dist = SdfLength(q) - that->params[capySdfParam_radius];

  // Return the distance
  return dist;
}

// Create a CapySdf for a capsule (aligned with y axis) of given dimensions
// Inputs:
//   radius: radius of the capsule
//   length: length of the capsule
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateCapsule(
  double const radius,
  double const length) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_capsule;
  that.params[capySdfParam_radius] = radius;
  that.params[capySdfParam_halfLength] = length * 0.5;
  that.distCanonical = SdfDistCapsule;
  return that;
}

// Allocate memory for a new CapySdf for a capsule (aligned with y axis) of
// given dimensions and create it
// Inputs:
//   radius: radius of the capsule
//   length: length of the capsule
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocCapsule(
  double const radius,
  double const length) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateCapsule(radius, length);
  return that;
}

// Get the value at a given position in a signed distance field for a merge
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistMerge(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double dist = $(that->comps[0], dist)(pos);
  double d = $(that->comps[1], dist)(pos);
  double s = 0.0;
  if(that->smoothness > 1e-9) {
    double h =
      SdfMax(that->smoothness - fabs(dist - d), 0.0) / that->smoothness;
    s = h * h * h * 0.166666666 * that->smoothness;
  }
  dist = SdfMin(dist, d) - s;

  // Return the distance
  return dist;
}

// Get the value at a given position in a signed distance field for a difference
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistDifference(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double dist = $(that->comps[0], dist)(pos);
  double d = $(that->comps[1], dist)(pos);
  double s = 0.0;
  if(that->smoothness > 1e-9) {
    double h =
      SdfMax(that->smoothness - fabs(-(dist + d)), 0.0) / that->smoothness;
    s = h * h * h * 0.166666666 * that->smoothness;
  }
  dist = SdfMax(dist, -d) + s;

  // Return the distance
  return dist;
}

// Get the value at a given position in a signed distance field for an
// intersection
// Inputs:
//   that: the SDF shape
//   pos: the untransformed position
// Output:
//   Return the distance in the field (distance from the position to the
//   nearest point at the surface of the shape, positive if the position is
//   outside the shape, negative if it's inside)
static double SdfDistIntersection(
  CapySdf* const that,
    double const pos[3]) {

  // Get the distance in the signed distance field
  double dist = $(that->comps[0], dist)(pos);
  double d = $(that->comps[1], dist)(pos);
  double s = 0.0;
  if(that->smoothness > 1e-9) {
    double h =
      SdfMax(that->smoothness - fabs(dist - d), 0.0) / that->smoothness;
    s = h * h * h * 0.166666666 * that->smoothness;
  }
  dist = SdfMax(dist, d) + s;

  // Return the distance
  return dist;
}

// Create a CapySdf for a merge of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateMerge(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_merge;
  that.comps[0] = shapeA;
  that.comps[1] = shapeB;
  that.distCanonical = SdfDistMerge;
  return that;
}

// Allocate memory for a new CapySdf for a merge of two shapes and create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocMerge(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateMerge(shapeA, shapeB);
  loop(i, 2) that->comps[i]->parent = that;
  return that;
}

// Create a CapySdf for an union of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateUnion(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_union;
  that.comps[0] = shapeA;
  that.comps[1] = shapeB;
  that.distCanonical = SdfDistMerge;
  return that;
}

// Allocate memory for a new CapySdf for an union of two shapes and create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocUnion(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateUnion(shapeA, shapeB);
  loop(i, 2) that->comps[i]->parent = that;
  return that;
}

// Create a CapySdf for an intersection of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateIntersection(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_intersection;
  that.comps[0] = shapeA;
  that.comps[1] = shapeB;
  that.distCanonical = SdfDistIntersection;
  return that;
}


// Allocate memory for a new CapySdf for an intersection of two shapes and
// create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocIntersection(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateIntersection(shapeA, shapeB);
  loop(i, 2) that->comps[i]->parent = that;
  return that;
}

// Create a CapySdf for a difference of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateDifference(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_difference;
  that.comps[0] = shapeA;
  that.comps[1] = shapeB;
  that.distCanonical = SdfDistDifference;
  return that;
}

// Allocate memory for a new CapySdf for a difference of two shapes and
// create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocDifference(
  CapySdf* const shapeA,
  CapySdf* const shapeB) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateDifference(shapeA, shapeB);
  loop(i, 2) that->comps[i]->parent = that;
  return that;
}

// Create a CapySdf for a user defined distance field
// Inputs:
//   sdf: the user defined signed distance field (canonical shape untransformed,
//        taking a reference to the shape and a position in local coordinate
//        system)
// Output:
//   Return a CapySdf.
CapySdf CapySdfCreateUserDefined(sdfFun const sdf) {
  CapySdf that = CapySdfCreate();
  that.type = capySdf_user;
  that.distCanonical = sdf;
  return that;
}

// Allocate memory for a new CapySdf for a user defined distance field and
// create it
// Inputs:
//   sdf: the user defined signed distance field (canonical shape untransformed,
//        taking a reference to the shape and a position in local coordinate
//        system)
// Output:
//   Return a CapySdf.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocUserDefined(sdfFun const sdf) {
  CapySdf* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySdfCreateUserDefined(sdf);
  return that;
}

