// ---------------------------- SdfCsg.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_SDFCSG_H
#define CAPY_SDFCSG_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Sdf class.

// Getter and setter for the epsilon value
double CapySdfGetEpsilon(void);
void CapySdfSetEpsilon(double epsilon);

// Getter and setter for the maximum number of step (to avoid infinite loop)
// during ray marching.
size_t CapySdfGetRayMarchNbMaxStep(void);
void CapySdfSetRayMarchNbMaxStep(size_t const nb);

// Types of transformation applied to a CapySdfShape or CapySdfCsg
typedef enum CapySdfTransformType {
  capySdfTransformType_none,
  capySdfTransformType_translate,
  capySdfTransformType_scale,
  capySdfTransformType_rotate,
  capySdfTransformType_extrusion,
  capySdfTransformType_symmetry,
  capySdfTransformType_revolution,
  capySdfTransformType_twist,
} CapySdfTransformType;

// Transformation structure for CapySdfShape or CapySdfCsg
typedef struct CapySdfTransform {

  // Type of the transformation
  CapySdfTransformType type;
  CapyPad(CapySdfTransformType, type);

  // Transformation's values. For translate, the 3 first values are used.
  // For scale the first value is used. For rotation, the 4 values are
  // the quaternion values (x,y,z and theta). For extrusion, the first value
  // is used (distance from origin along z axis)
  double vals[4];
} CapySdfTransform;

// Maximum number of transformation per SdfShape
#define CAPY_MAX_NB_TRANSFORM_SDF 10

// Type of shape
typedef enum CapySdfType {
  capySdf_box, capySdf_sphere, capySdf_torus, capySdf_capsule,
  capySdf_ellipsoid, capySdf_cone, capySdf_plane,
  capySdf_merge, capySdf_union, capySdf_intersection, capySdf_difference,
  capySdf_user,
} CapySdfType;

// Indices of shape parameters
typedef enum CapySdfParam {
  capySdfParam_radius = 0, capySdfParam_halfLength,
  capySdfParam_majorRadius = 0, capySdfParam_minorRadius,
  capySdfParam_radiusX = 0, capySdfParam_radiusY, capySdfParam_radiusZ,
  capySdfParam_halfWidth = 0, capySdfParam_halfHeight, capySdfParam_halfDepth,
  capySdfParam_baseRadius = 0, capySdfParam_topRadius, capySdfParam_height,
  capySdfParam_normX = 0, capySdfParam_normY, capySdfParam_normZ,
  capySdfParam_dist,
  capySdfParam_nbParam = 4
} CapySdfParam;

// Predeclaration of CapySdf
typedef struct CapySdf CapySdf;

// Structure to memorise the result of ray marching
typedef struct CapyRayMarchingRes {

  // Flag to memorise if the ray intersects the shape
  bool flagIntersect;
  CapyPad(bool, flagIntersect);

  // Flag to memorise if the ray marching reached the maximum number of steps
  bool hasGaveUp;
  CapyPad(bool, hasGaveUp);

  // Reference to the shape of the nearest intersection
  CapySdf* shape;

  // Distance from the ray origin to the intersection
  double dist;

  // Shortest observed signed distance between the ray and a shape during the
  // steps of ray marching
  double distMin;

  // Coordinates of the intersection in world coordinate system
  double pos[3];

  // Coordinates of the intersection in the shape local coordinate system
  double posLocal[3];

  // Normal at the intersection in world coordinate system
  double normal[3];
} CapyRayMarchingRes;

// Sdf class
struct CapySdf {

  // Type of shape
  CapySdfType type;
  CapyPad(CapySdfType, type);

  // Transformations applied to the shpae
  CapySdfTransform transforms[CAPY_MAX_NB_TRANSFORM_SDF];

  // Thickness of the shape
  double thickness;

  // Roundness of the shape
  double roundness;

  // Parameters of the shape
  double params[capySdfParam_nbParam];

  // User data
  void* data;

  // Components for a CSG shape
  CapySdf* comps[2];

  // PArent for a CSG shape
  CapySdf* parent;

  // Smoothness for CSG
  double smoothness;

  // Destructor
  void (*destruct)(void);

  // Distance from a given position to the nearest point on the shape surface
  // accounting for the eventual transformations applied to the shape.
  // Input:
  //   pos: the given position
  // Output:
  //   Return the distance, less or equal to the exact distance.
  //   A positive distance indicates a posiiton outside the shape. A negative
  //   distance indicates a position inside the shape.
  double (*dist)(double const pos[3]);

  // Distance from a given position to the nearest point on the shape surface
  // not accounting for the eventual transformations applied to the shape.
  // Input:
  //   that: the SDF shape
  //   pos: the given position
  // Output:
  //   Return the distance. Exact if the position is outside the shape, less
  //   or equal to the exact distance if the position is inside the shape.
  //   A positive distance indicates a posiiton outside the shape. A negative
  //   distance indicates a position inside the shape.
  double (*distCanonical)(
    CapySdf* const that,
      double const pos[3]);

  // Apply a translation to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  //   u: the translation
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*translate)(
    size_t const iTrans,
    double const u[3]);

  // Apply a rotation to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  //   u: the rotation axis
  //   theta: the rotation angle in radians
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*rotate)(
    size_t const iTrans,
    double const u[3],
    double const theta);

  // Apply a scaling to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  //   s: the scaling factor
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*scale)(
    size_t const iTrans,
    double const s);

  // Apply an extrusion (along the z axis, relative to the origin) to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  //   l: the extrusion length
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*extrude)(
    size_t const iTrans,
    double const l);

  // Apply a symmetry (duplicate the x>=0 half-space relative to the yz plane)
  // to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*symmetry)(size_t const iTrans);

  // Apply a revolution (360 degrees rotation around y of the profile in the
  // xy x>=0 half-plane) to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*revolution)(size_t const iTrans);

  // Apply a twist (around y) to the shape
  // Inputs:
  //   iTrans: the index of the transformation
  //   freq: the frequency of the twist (one rotation per 'freq' unit along y,
  //         twist clockwise with +y)
  // Output:
  //   The transformation is set and will be applied when using the 'dist'
  //   method.
  void (*twist)(
    size_t const iTrans,
    double const freq);

  // Setter and getter for the user data
  void* (*getData)(void);
  void (*setData)(void* const data);

  // Normal at a given position (as the gradient in the field at that position)
  // Input:
  //   pos: the given position
  //   normal: the result normal
  // Output:
  //   'normal' is updated, normalised, oriented away from the shape.
  void (*normal)(
    double const pos[3],
          double normal[3]);

  // Setter and getter for the shape parameters
  void (*setParam)(
    CapySdfParam const iParam,
          double const val);
  double (*getParam)(CapySdfParam const iParam);

  // Search the intersection between a ray and the shape
  // Inputs:
  //   from: origin of the ray
  //   dir: direction of the ray, normalised
  //   res: result of the search
  // Output:
  //   If there is an intersection nearer than the current one in 'res', 'res'
  //   is updated, else it is left unchanged.
  void (*rayMarching)(
                 double const from[3],
                 double const dir[3],
    CapyRayMarchingRes* const res);
};

// Create a CapySdf (by defaut a unit box)
// Output:
//   Return a CapySdf
CapySdf CapySdfCreate(void);

// Allocate memory for a new CapySdf and create it (by defaut a unit box)
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAlloc(void);

// Free the memory used by a CapySdf* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySdf to free
void CapySdfFree(CapySdf** const that);

// Create a CapySdf for a box of given dimensions
// Inputs:
//   width: width of the box (x axis)
//   height: height of the box (y axis)
//   depth: depth of the box (z axis)
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateBox(
  double const width,
  double const height,
  double const depth);

// Allocate memory for a new CapySdf for a box of given dimensions and create it
// Inputs:
//   width: width of the box (x axis)
//   height: height of the box (y axis)
//   depth: depth of the box (z axis)
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocBox(
  double const width,
  double const height,
  double const depth);

// Create a CapySdf for a plane
// Inputs:
//   norm: normal of the plane (toward the 'out' side of the plane)
//   dist: distance from the origin in the direction of the normal
// Output:
//   Return a CapySdf
CapySdf CapySdfCreatePlane(
  double const norm[3],
  double const dist);

// Allocate memory for a new CapySdf for a plane and create it
// Inputs:
//   norm: normal of the plane (toward the 'out' side of the plane)
//   dist: distance from the origin in the direction of the normal
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocPlane(
  double const norm[3],
  double const dist);

// Create a CapySdf for a sphere of given dimensions
// Inputs:
//   radius: radius of the sphere
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateSphere(double const radius);

// Allocate memory for a new CapySdf for a sphere of given dimensions and
// create it
// Inputs:
//   radius: radius of the sphere
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocSphere(double const radius);

// Create a CapySdf for a cone (center at the origin, rotation
// axis along the y axis, top toward +y) of given dimensions and create it
// Inputs:
//   radius: radius of the sphere
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateCone(
  double const baseRadius,
  double const topRadius,
  double const height);

// Allocate memory for a new CapySdf for a cone (center at the origin, rotation
// axis along the y axis, top toward +y) of given dimensions and create it
// Inputs:
//   baseRadius: radius of the cone at
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocCone(
  double const baseRadius,
  double const topRadius,
  double const height);

// Create a CapySdf for an ellipsoid of given dimensions (aligned with
// x, y,z axis)
// Inputs:
//   radiusX: radius of the ellipsoid along the x axis
//   radiusY: radius of the ellipsoid along the y axis
//   radiusZ: radius of the ellipsoid along the z axis
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateEllipsoid(
  double const radiusX,
  double const radiusY,
  double const radiusZ);

// Allocate memory for a new CapySdf for an ellipsoid of given dimensions
// (aligned with x, y,z axis) and create it
// Inputs:
//   radiusX: radius of the ellipsoid along the x axis
//   radiusY: radius of the ellipsoid along the y axis
//   radiusZ: radius of the ellipsoid along the z axis
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocEllipsoid(
  double const radiusX,
  double const radiusY,
  double const radiusZ);

// Create a CapySdf for a torus of given dimensions (in xz plane)
// Inputs:
//   majorRadius: major radius of the torus
//   minorRadius: minor radius of the torus
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateTorus(
  double const majorRadius,
  double const minorRadius);

// Create a CapySdf for a capsule (aligned with y axis) of given dimensions
// Inputs:
//   radius: radius of the capsule
//   length: length of the capsule
// Output:
//   Return a CapySdf
CapySdf CapySdfCreateCapsule(
  double const radius,
  double const length);

// Allocate memory for a new CapySdf for a capsule (aligned with y axis) of
// given dimensions and create it
// Inputs:
//   radius: radius of the capsule
//   length: length of the capsule
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocCapsule(
  double const radius,
  double const length);

// Allocate memory for a new CapySdf for a torus of given dimensions (in xz
// plane) and create it
// Inputs:
//   majorRadius: major radius of the torus
//   minorRadius: minor radius of the torus
// Output:
//   Return a CapySdf
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocTorus(
  double const majorRadius,
  double const minorRadius);

// Create a CapySdf for a merge of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateMerge(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Allocate memory for a new CapySdf for a merge of two shapes and create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocMerge(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Create a CapySdf for an union of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateUnion(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Allocate memory for a new CapySdf for an union of two shapes and create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocUnion(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Create a CapySdf for an intersection of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateIntersection(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Allocate memory for a new CapySdf for an intersection of two shapes and
// create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocIntersection(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Create a CapySdf for a difference of two shapes
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
CapySdf CapySdfCreateDifference(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Allocate memory for a new CapySdf for a difference of two shapes and
// create it
// Inputs:
//   shapeA: the first shape
//   shapeB: the second shape
// Output:
//   Return a CapySdf. Component shapes can't be shared with between several
//   CSG, they must be dynamically allocated, and they are free-d by their
//   parent.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocDifference(
  CapySdf* const shapeA,
  CapySdf* const shapeB);

// Signed distance field function type declaration
typedef double (*sdfFun)(
  CapySdf* const that,
  double const pos[3]);

// Create a CapySdf for a user defined distance field
// Inputs:
//   sdf: the user defined signed distance field (canonical shape untransformed,
//        taking a reference to the shape and a position in local coordinate
//        system)
// Output:
//   Return a CapySdf.
CapySdf CapySdfCreateUserDefined(sdfFun const sdf);

// Allocate memory for a new CapySdf for a user defined distance field and
// create it
// Inputs:
//   sdf: the user defined signed distance field (canonical shape untransformed,
//        taking a reference to the shape and a position in local coordinate
//        system)
// Output:
//   Return a CapySdf.
// Exception:
//   May raise CapyExc_MallocFailed.
CapySdf* CapySdfAllocUserDefined(sdfFun const sdf);
#endif
