// ------------------------------ sort.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "sort.h"

// Swap two elements
#define swap(a, i, j, s)                         \
  char tmp[s];                                   \
  memcpy(tmp, (char*)a + i * s, s);              \
  memcpy((char*)a + i * s, (char*)a + j * s, s); \
  memcpy((char*)a + j * s, tmp, s)

// Partitioning function for quick sort
static size_t QuickSortPartition(
            void* const data,
           size_t const sizeElem,
           size_t const low,
           size_t const high,
  CapyComparator* const cmp) {
  size_t pivot = low + (high - low) / 2;
  char pivotElem[sizeElem];
  memcpy(pivotElem, (char*)data + pivot * sizeElem, sizeElem);
  size_t i = low - 1;
  size_t j = high + 1;
  while(1) {
    do { ++i; }
    while($(cmp, eval)((char*)data + i * sizeElem, pivotElem) < 0);
    do { --j; }
    while($(cmp, eval)((char*)data + j * sizeElem, pivotElem) > 0);
    if(i >= j) return j;
    swap(data, i, j, sizeElem);
  }
}

// Quick sort, recursion
// Input:
//       data: the array of data to sort
//   sizeElem: the size in byte of one element in the array
//        low: lower index
//       high: higher index
//        cmp: the comparator used to sort the array
static void CapyQuickSortRec(
            void* const data,
           size_t const sizeElem,
           size_t const low,
           size_t const high,
  CapyComparator* const cmp) {
  if(low < high) {
    size_t pivot = QuickSortPartition(data, sizeElem, low, high, cmp);
    CapyQuickSortRec(data, sizeElem, low, pivot, cmp);
    CapyQuickSortRec(data, sizeElem, pivot + 1, high, cmp);
  }
}

// Quick sort
// Input:
//       data: the array of data to sort
//   sizeElem: the size in byte of one element in the array
//     nbElem: the number of elements in the array
//        cmp: the comparator used to sort the array
void CapyQuickSort(
            void* const data,
           size_t const sizeElem,
           size_t const nbElem,
  CapyComparator* const cmp) {

  // Start the recursive sort if there is more than 1 element
  if(nbElem > 1) CapyQuickSortRec(data, sizeElem, 0, nbElem - 1, cmp);
}
