// ------------------------------- springmass.h ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_SPRINGMASS_H
#define CAPY_SPRINGMASS_H
#include "externalHeaders.h"
#include "cext.h"
#include "graph.h"
#include "rungekutta.h"

// Description:
// SpringMass class.

// CapySpringMassNode structure
typedef struct CapySpringMassNode {

  // Mass id
  size_t id;

  // Mass of the node
  double mass;

  // Node position
  CapyVec pos;

  // Node velocity
  CapyVec speed;

  // Node acceleration
  CapyVec acc;

  // Fixed mass (true: position stays constant)
  bool flagFixed;
  CapyPad(bool, flagFixed);

  // User data
  void* data;
} CapySpringMassNode;

// CapySpringMassLink structure
typedef struct CapySpringMassLink {

  // Spring coefficients ([0]: in compression regime, [1]: in extension regime)
  double coeffs[2];

  // Length at rest
  double restLength;

  // Links to the masses
  CapySpringMassNode* masses[2];

  // User data
  void* data;
} CapySpringMassLink;

// Derivative for the spring mass system
typedef struct CapySpringMassDeriv {

  // Inherit CapyMathFun
  struct CapyMathFunDef;

  // Acceleration force
  double* accs;

  // Destructor
  void (*destructCapyMathFun)(void);
} CapySpringMassDeriv;

// SpringMass object
typedef struct CapySpringMass {

  // Space dimension
  size_t dim;

  // Dampening coefficient (default: 1.0, in [0.0, 1.0], the lower the
  // stronger the dampening, 1.0 is no dampening)
  double dampening;

  // Graph of the spring-mass sytem
  CapyGraph* graph;

  // Derivative function
  CapySpringMassDeriv* deriv;

  // Runge-Kutta instance
  CapyRungeKutta* rk;

  // Nb max step when searching for stability (default: 1000)
  size_t nbMaxStep;

  // Epsilon value used to check stability (default: 1e-6)
  double epsilon;

  // Destructor
  void (*destruct)(void);

  // Add one mass
  // Input:
  //   id: id of the mass
  // Output:
  //   If no mass with the given id already exists a new one is added. The
  //   mass id is the id of the node in the graph. Return the mass. If created
  //   the new mass is initialised with a mass of 1.0 and all other params
  //   to 0.0.
  CapySpringMassNode* (*addMass)(size_t const id);

  // Get a mass given its id
  // Input:
  //   id: the id
  // Output:
  //   Return the mass, or NULL if there is no mass with the requested id
  CapySpringMassNode* (*getMass)(size_t const id);

  // Add one spring
  // Input:
  //   idA: id of the first mass
  //   idB: id of the second mass
  // Output:
  //   Add a new spring, initialised with spring coefficients of 1.0 and
  //   rest length of 1.0. Return the new spring. If there is no mass with the
  //   given id they are automatically created.
  CapySpringMassLink* (*addSpring)(
    size_t const idA,
    size_t const idB);

  // Get a spring given its masses id
  // Input:
  //   idA: the first mass id
  //   idB: the second mass id
  // Output:
  //   Return the spring, or NULL if there is no spring with the requested ids
  CapySpringMassLink* (*getSpring)(
    size_t const idA,
    size_t const idB);

  // Step the spring mass system
  // Input:
  //   deltaT: size of the step
  // Output:
  //   The mass properties are updated.
  void (*step)(double const deltaT);

  // Step the spring mass system until it stabilizes
  // Input:
  //   deltaT: size of the step
  // Output:
  //   The mass properties are updated.
  void (*stepToStableState)(double const deltaT);

  // Get the stress of the system
  // Output:
  //   Return the total of difference between rest length and actual length
  //   of all links
  double (*getStress)(void);

  // Get the stress of a link
  // Input:
  //   idA: the first mass id
  //   idB: the second mass id
  // Output:
  //   Return the difference between rest length and actual length
  double (*getSpringStress)(
    size_t const idA,
    size_t const idB);

  // Get the current length of a link
  // Input:
  //   idA: the first mass id
  //   idB: the second mass id
  // Output:
  //   Return the current length of the link
  double (*getSpringLength)(
    size_t const idA,
    size_t const idB);
} CapySpringMass;

// Create a CapySpringMass
// Input:
//   dim: space dimension
// Output:
//   Return a CapySpringMass
CapySpringMass CapySpringMassCreate(size_t const dim);

// Allocate memory for a new CapySpringMass and create it
// Input:
//   dim: space dimension
// Output:
//   Return a CapySpringMass
// Exception:
//   May raise CapyExc_MallocFailed.
CapySpringMass* CapySpringMassAlloc(size_t const dim);

// Free the memory used by a CapySpringMass* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySpringMass to free
void CapySpringMassFree(CapySpringMass** const that);
#endif
