// ------------------------------ strDecorator.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "strDecorator.h"

// Escape sequence for each option
// cf https://en.wikipedia.org/wiki/ANSI_escape_code
static char const* escapeSeqStart[capyStrDecoratorNbOption] = {
  "\033[1m",
  "\033[4m",
  "\033[5m",
  "\033[38;2;",
  "\033[48;2;",
};

static char const* escapeSeqEnd[capyStrDecoratorNbOption] = {
  "\033[0m",
  "\033[0m",
  "\033[0m",
  "\033[39m",
  "\033[49m",
};

// Activate an option
// Input:
//   option: the option to activate
static void Activate(CapyStrDecoratorOption option) {
  methodOf(CapyStrDecorator);
  that->activeOptions[option] = true;
}

// Deactivate an option
// Input:
//   option: the option to deactivate
static void Deactivate(CapyStrDecoratorOption option) {
  methodOf(CapyStrDecorator);
  that->activeOptions[option] = false;
}

// Get the status of an option
// Input:
//   option: the option to get the status of
// Ouput:
//   Returns true if the option is active, else false
static bool IsActive(CapyStrDecoratorOption option) {
  methodOf(CapyStrDecorator);
  return that->activeOptions[option];
}

// Print a decorated string on a stream
// Input:
//   stream: the stream on which to print
//      fmt: the format string to decorate and print
//      ...: the arguments of the format string
// Output:
//   Return the number of characters printed (excluding the terminating null
//   byte). If an output error is encountered, return a negative value.
static int Fprintf(
        FILE* const stream,
  char const* const fmt,
                    ...) {
  methodOf(CapyStrDecorator);
#if BUILD_MODE == 0
  assert(stream != NULL);
  assert(fmt != NULL);
#endif

  // Variable to memorise the returned value
  int ret = 0;

  // Pointer to the escape sequence used in the code commonalisation loop
  char const** escapeSeq = escapeSeqStart;

  // Loop to commonalise code
  loop(i, 2) {

    // If we are on the second pass
    if(i == 1) {

      // Print the string in argument (after the start sequences and before
      // the end sequences)
      va_list args;
      va_start(args, fmt);
      *capyRetInt = vfprintf(stream, fmt, args);
      va_end(args);
      if(*capyRetInt < 0) return *capyRetInt;
      else ret += *capyRetInt;
    }

    // Loop on the active options
    loop(iOpt, capyStrDecoratorNbOption) {
      if(that->activeOptions[iOpt]) {

        // Print the start/end sequence for the option
        *capyRetInt = fprintf(stream, "%s", escapeSeq[iOpt]);
        if(*capyRetInt < 0) return *capyRetInt;
        else ret += *capyRetInt;

        // If it's the foreground color or background color, and we are on
        // the first pass, add the color
        if(i == 0 && iOpt == capyStrDecoratorFgColor) {
          *capyRetInt = fprintf(
            stream, "%d;%d;%dm",
            (uint8_t)(255.0 * that->fgColor.RGB[0]),
            (uint8_t)(255.0 * that->fgColor.RGB[1]),
            (uint8_t)(255.0 * that->fgColor.RGB[2]));
          if(*capyRetInt < 0) return *capyRetInt;
          else ret += *capyRetInt;
        } else if(i == 0 && iOpt == capyStrDecoratorBgColor) {
          *capyRetInt = fprintf(
            stream, "%d;%d;%dm",
            (uint8_t)(255.0 * that->bgColor.RGB[0]),
            (uint8_t)(255.0 * that->bgColor.RGB[1]),
            (uint8_t)(255.0 * that->bgColor.RGB[2]));
          if(*capyRetInt < 0) return *capyRetInt;
          else ret += *capyRetInt;
        }
      }
    }

    // Switch the pointer to the escape sequence at the end of the first pass
    escapeSeq = escapeSeqEnd;
  }

  // Return the result value
  return ret;
}

// Set the foreground color used by the option capyStrDecoratorFgColor
// Input:
//   color: the RGB color of the foreground
static void SetFgColor(CapyColorData const* const color) {
  methodOf(CapyStrDecorator);
#if BUILD_MODE == 0
  assert(color != NULL);
#endif
  that->fgColor = *color;
}

// Set the background color used by the option capyStrDecoratorBgColor
// Input:
//   color: the RGB color of the foreground
static void SetBgColor(CapyColorData const* const color) {
  methodOf(CapyStrDecorator);
#if BUILD_MODE == 0
  assert(color != NULL);
#endif
  that->bgColor = *color;
}

// Free the memory used by a CapyStrDecorator
static void Destruct(void) {
  return;
}

// Create a CapyStrDecorator
// Output:
//   Return a CapyStrDecorator initialised with no active options
CapyStrDecorator CapyStrDecoratorCreate(void) {
  CapyStrDecorator that = {
    .fgColor = capyColorRGBAWhite,
    .bgColor = capyColorRGBABlack,
    .destruct = Destruct,
    .activate = Activate,
    .deactivate = Deactivate,
    .isActive = IsActive,
    .fprintf = Fprintf,
    .setFgColor = SetFgColor,
    .setBgColor = SetBgColor,
  };
  loop(iOpt, capyStrDecoratorNbOption) that.activeOptions[iOpt] = false;
  return that;
}

// Allocate memory for a new CapyStrDecorator and create it
// Output:
//   Return a CapyStrDecorator initialised with no active options
// Exception:
//   May raise CapyExc_MallocFailed.
CapyStrDecorator* CapyStrDecoratorAlloc(void) {
  CapyStrDecorator* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyStrDecoratorCreate();
  return that;
}

// Free the memory used by a CapyStrDecorator* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyStrDecorator to free
void CapyStrDecoratorFree(CapyStrDecorator** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Default decorators
CapyStrDecorator const capyDecoratorRed = {
  .fgColor = {.RGBA = {1.0, 0.0, 0.0, 1.0}},
  .bgColor = {.RGBA = {0.0, 0.0, 0.0, 1.0}},
  .activate = Activate,
  .deactivate = Deactivate,
  .isActive = IsActive,
  .fprintf = Fprintf,
  .setFgColor = SetFgColor,
  .setBgColor = SetBgColor,
  .activeOptions[capyStrDecoratorFgColor] = true,
};
