// ------------------------------ strDecorator.h ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_STRDECORATOR_H
#define CAPY_STRDECORATOR_H
#include "externalHeaders.h"
#include "color.h"

// Description:
// Class to embellish strings with ANSI escape code.

// Enumeration of available options to decorate a string
typedef enum CapyStrDecoratorOption {
  capyStrDecoratorBold,
  capyStrDecoratorUnderline,
  capyStrDecoratorSlowBlink,
  capyStrDecoratorFgColor,
  capyStrDecoratorBgColor,
  capyStrDecoratorNbOption
} CapyStrDecoratorOption;

// CapyStrDecorator object
typedef struct CapyStrDecorator {

  // Foreground and background color (default: white and black)
  CapyColorData fgColor;
  CapyColorData bgColor;

  // Flags to memorise the active options
  bool activeOptions[capyStrDecoratorNbOption];
  CapyPad(bool[capyStrDecoratorNbOption], 0);

  // Destructor
  void (*destruct)(void);

  // Activate an option
  // Input:
  //   option: the option to activate
  void (*activate)(CapyStrDecoratorOption option);

  // Deactivate an option
  // Input:
  //   option: the option to deactivate
  void (*deactivate)(CapyStrDecoratorOption option);

  // Get the status of an option
  // Input:
  //   option: the option to get the status of
  // Ouput:
  //   Returns true if the option is active, else false
  bool (*isActive)(CapyStrDecoratorOption option);

  // Print a decorated string on a stream
  // Input:
  //   stream: the stream on which to print
  //      fmt: the format string to decorate and print
  //      ...: the arguments of the format string
  // Output:
  //   Return the number of characters printed (excluding the terminating null
  //   byte). If an output error is encountered, return a negative value.
  int (*fprintf)(
          FILE* const stream,
    char const* const fmt,
                      ...);

  // Set the foreground color used by the option capyStrDecoratorFgColor
  // Input:
  //   color: the RGB color of the foreground
  void (*setFgColor)(CapyColorData const* const color);

  // Set the background color used by the option capyStrDecoratorBgColor
  // Input:
  //   color: the RGB color of the foreground
  void (*setBgColor)(CapyColorData const* const color);
} CapyStrDecorator;

// Create a CapyStrDecorator
// Output:
//   Return a CapyStrDecorator initialised with no active options
CapyStrDecorator CapyStrDecoratorCreate(void);

// Allocate memory for a new CapyStrDecorator and create it
// Output:
//   Return a CapyStrDecorator initialised with no active options
// Exception:
//   May raise CapyExc_MallocFailed.
CapyStrDecorator* CapyStrDecoratorAlloc(void);

// Free the memory used by a CapyStrDecorator* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyStrDecorator to free
void CapyStrDecoratorFree(CapyStrDecorator** const that);

// Default decorators
extern CapyStrDecorator const capyDecoratorRed;
#endif
