// ------------------------------ strDecorator.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "streamIo.h"

// Open the stream for a given pathname and given mode
// If the StreamIo was already opened it is closed before opening
// the new one.
// Input:
//   pathname: the pathname to the file of the stream
//       mode: the mode as in fopen()
// Exceptions:
//   May raise CapyExc_MallocFailed, CapyExc_StreamOpenError
static void Open(
  char const* const pathname,
  char const* const mode) {
  methodOf(CapyStreamIo);
#if BUILD_MODE == 0
  assert(pathname != NULL);
  assert(mode != NULL);
#endif

  // Ensure the stream is closed
  $(that, close)();

  // Copy the pathname
  safeSPrintf(&(that->pathname), "%s", pathname);

  // Open the stream
  that->stream = safeFOpen(pathname, mode);
}

// Close the stream
static void Close(void) {
  methodOf(CapyStreamIo);

  // If the stream is already opened, close it
  free(that->pathname);
  that->pathname = NULL;
  if(that->stream != NULL) fclose(that->stream);
  that->stream = NULL;
}

// Load the content of a text file line per line
// Output:
//   Return the content of the file as a newly allocated list of
//   array of char
// Exceptions:
//   May raise CapyExc_MallocFailed, CapyExc_StreamReadError
static CapyListArrChar* ReadLines(void) {
  methodOf(CapyStreamIo);
#if BUILD_MODE == 0
  assert(that->stream != NULL);
#endif

  // The result list of lines
  CapyListArrChar* lines;
  try {

    // Allocate memory for the result list of lines
    lines = CapyListArrCharAlloc();

    // Loop on the lines until the end of the file
    while(!feof(that->stream)) {

      // Declare a temporary buffer to read the line
      CapyArrChar line = CapyArrCharCreate(10);

      // Read the line character by character
      struct { size_t idx; } volatiles = { .idx = 0 };
      do {

        // If the read failed the value at this index will be undefined,
        // set it to keep things clean
        line.data[volatiles.idx] = '\0';

        // Temporary silence the CapyRaise function to avoid printing the
        // reading failure at the end of file
        FILE* backupStream = CapyGetRaiseStream();
        CapySetRaiseStream(NULL);

        // Try to read the next character
        try {
          safeFRead(that->stream, 1, line.data + volatiles.idx);
        } catch(CapyExc_StreamReadError) {

          // If the read failed while we haven't reached the end of the file,
          // this is really an error, else it's fine
          if(feof(that->stream)) CapyCancelExc();
        } endCatch;

        // Reset the stream
        CapySetRaiseStream(backupStream);

        // Forward the eventual exception
        CapyForwardExc();

        // Increment the index to the next character
        ++(volatiles.idx);

        // If the buffer is full, resize it with the double of its size
        if(volatiles.idx == line.size) $(&line, resize)(2 * line.size);
      } while(
        line.data[volatiles.idx - 1] != '\n' &&
        !feof(that->stream));

      // If the last read character was a line return, replace it with the
      // null character
      if(line.data[volatiles.idx - 1] == '\n') {
        line.data[volatiles.idx - 1] = '\0';
      }

      // Else, the end of file occured inside a line, add a null character
      else line.data[volatiles.idx] = '\0';

      // If we are not at the end of the file or the line is not empty,
      // resize the line to its exact size and add it to the list of lines
      if(!feof(that->stream) || volatiles.idx > 1) {
        $(&line, resize)(volatiles.idx + 1);
        $(lines, add)(line);
      } else $(&line, destruct)();
    }
  } catchDefault {

    // Free memory
    while($(lines, getSize)() != 0) {
      CapyArrChar line = $(lines, pop)();
      $(&line, destruct)();
    }
    CapyListArrCharFree(&lines);
  } endCatch;

  // Forward the eventual exception
  CapyForwardExc();

  // Return the lines
  return lines;
}

// Save a list of lines to a text file
// Input:
//   lines: a list of array of char, the lines to save
// Exceptions:
//   May raise CapyExc_StreamWriteError
static void WriteLines(CapyListArrChar const* const lines) {
  methodOf(CapyStreamIo);
#if BUILD_MODE == 0
  assert(lines != NULL);
  assert(that->stream != NULL);
#endif

  // Loop on the lines and write them
  CapyListArrCharIterator iter = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)lines,
    capyListIteratorType_forward);
  forEach(line, iter) {
    safeFPrintf(
      that->stream,
      "%s\n",
      line.data);
  }
}

// Load the raw content of a file
// Output:
//   Return the content of the file as an array of char
// Exceptions:
//   May raise CapyExc_MallocFailed, CapyExc_StreamReadError
static CapyArrChar* Read(void) {
  methodOf(CapyStreamIo);
#if BUILD_MODE == 0
  assert(that->stream != NULL);
#endif

  // The result data
  CapyArrChar* data = NULL;
  try {

    // Allocate memory for the result
    data = CapyArrCharAlloc(10);

    // Read the data byte per byte
    struct { size_t idx; } volatiles = { .idx = 0 };

    // Loop on the lines until the end of the file
    while(!feof(that->stream)) {

      // Try to read the next character
      try {
        safeFRead(that->stream, 1, data->data + volatiles.idx);
      } catch(CapyExc_StreamReadError) {

        // If the read failed while we haven't reached the end of the file,
        // this is really an error, else it's fine
        if(feof(that->stream)) CapyCancelExc();

        // If the read failed the value at this index will be undefined,
        // set it to keep things clean
        data->data[volatiles.idx] = 0;
      } endCatch;

      // Forward the exception
      CapyForwardExc();

      // Increment the index to the next character if we are not at the end
      if(!feof(that->stream)) ++(volatiles.idx);

      // If the buffer is full, resize it with the double of its size
      if(volatiles.idx == data->size) $(data, resize)(2 * data->size);
    }

    // Resize the data to its exact size
    $(data, resize)(volatiles.idx);
  } catchDefault {
    CapyArrCharFree(&data);
  } endCatch;

  // Forward the eventual exception
  CapyForwardExc();

  // Return the data
  return data;
}

// Save data to a file
// Input:
//   data: an array of char, the data to save
// Exceptions:
//   May raise CapyExc_StreamWriteError
static void Write(CapyArrChar const* const data) {
  methodOf(CapyStreamIo);
#if BUILD_MODE == 0
  assert(data != NULL);
  assert(that->stream != NULL);
#endif

  // Write the data
  safeFWrite(that->stream, data->size, data->data);
}

// Check if a stream is opened
// Output:
//   Return true if the stream is opened, else false
static bool IsOpened(void) {
  methodOf(CapyStreamIo);
  return (that->stream != NULL);
}

// Read bytes from a file
// Input:
//   data: pointer to where the data are written
//   size: the size in bytes of the read data
// Exceptions:
//   May raise CapyExc_StreamReadError
static void ReadBytes(
   void* const data,
  size_t const size) {
  methodOf(CapyStreamIo);
  safeFRead(that->stream, size, data);
}

// Write bytes to a file
// Input:
//   data: pointer to the data
//   size: the size in bytes of the data
// Exceptions:
//   May raise CapyExc_StreamWriteError
static void WriteBytes(
  void const* const data,
       size_t const size) {
  methodOf(CapyStreamIo);
  safeFWrite(that->stream, size, data);
}

// Get the size of the file associated to the stream.
// Output:
//   Return the size in bytes of the file, or 0 if the stream is not opened
//   or any error occured.
static size_t GetSize(void) {
  methodOf(CapyStreamIo);
  if(that->pathname == NULL) return 0;
  FileStat fileStatus;
  int ret = stat(that->pathname, &fileStatus);
  if(ret < 0) return 0;
  return (size_t)(fileStatus.st_size);
}

// Free the memory used by a CapyStreamIo
static void Destruct(void) {
  methodOf(CapyStreamIo);

  // Ensure the stream is closed
  $(that, close)();

  // Free the memory
  free(that->pathname);
  that->pathname = NULL;
  if(that->stream != NULL) fclose(that->stream);
  that->stream = NULL;
}

// Create a CapyStreamIo
// Output:
//   Return a CapyStreamIo
CapyStreamIo CapyStreamIoCreate(void) {

  // Create the StreamIo
  CapyStreamIo that = {
    .pathname = NULL,
    .stream = NULL,
    .destruct = Destruct,
    .open = Open,
    .close = Close,
    .writeLines = WriteLines,
    .readLines = ReadLines,
    .write = Write,
    .read = Read,
    .isOpened = IsOpened,
    .writeBytes = WriteBytes,
    .readBytes = ReadBytes,
    .getSize = GetSize,
  };

  // Return the StreamIo
  return that;
}

// Allocate memory for a new CapyStreamIo and create it
// Output:
//   Return a CapyStreamIo
// Exception:
//   May raise CapyExc_MallocFailed.
CapyStreamIo* CapyStreamIoAlloc(void) {

  // Allocate memory for the StreamIo
  CapyStreamIo* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the StreamIo
  *that = CapyStreamIoCreate();

  // Return the StreamIo
  return that;
}

// Free the memory used by a CapyStreamIo* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyStreamIo to free
void CapyStreamIoFree(CapyStreamIo** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
