// ---------------------------------- color.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_STREAM_IO_H
#define CAPY_STREAM_IO_H
#include "externalHeaders.h"
#include "list.h"

// Description:
// I/O stream class.

// StreamIo object
typedef struct CapyStreamIo {

  // The pathname to the stream (may be null)
  char* pathname;

  // The stream
  FILE* stream;

  // Destructor
  void (*destruct)(void);

  // Open the stream for a given pathname and given mode
  // If the StreamIo was already opened it is closed before opening
  // the new one.
  // Input:
  //   pathname: the pathname to the file of the stream
  //       mode: the mode as in fopen()
  // Exceptions:
  //   May raise CapyExc_MallocFailed, CapyExc_StreamOpenError
  void (*open)(
    char const* const pathname,
    char const* const mode);

  // Close the stream
  void (*close)(void);

  // Load the content of a text file line per line
  // Output:
  //   Return the content of the file as a newly allocated list of
  //   array of char
  // Exceptions:
  //   May raise CapyExc_MallocFailed, CapyExc_StreamReadError
  CapyListArrChar* (*readLines)(void);

  // Save a list of lines to a text file
  // Input:
  //   lines: a list of array of char, the lines to save
  // Exceptions:
  //   May raise CapyExc_StreamWriteError
  void (*writeLines)(CapyListArrChar const* const lines);

  // Load the raw content of a file
  // Output:
  //   Return the content of the file as an array of char
  // Exceptions:
  //   May raise CapyExc_MallocFailed, CapyExc_StreamReadError
  CapyArrChar* (*read)(void);

  // Save data to a file
  // Input:
  //   data: an array of char, the data to save
  // Exceptions:
  //   May raise CapyExc_StreamWriteError
  void (*write)(CapyArrChar const* const data);

  // Check if a stream is opened
  // Output:
  //   Return true if the stream is opened, else false
  bool (*isOpened)(void);

  // Read bytes from a file
  // Input:
  //   data: pointer to where the data are written
  //   size: the size in bytes of the read data
  // Exceptions:
  //   May raise CapyExc_StreamReadError
  void (*readBytes)(
     void* const data,
    size_t const size);

  // Write bytes to a file
  // Input:
  //   data: pointer to the data
  //   size: the size in bytes of the data
  // Exceptions:
  //   May raise CapyExc_StreamWriteError
  void (*writeBytes)(
    void const* const data,
         size_t const size);

  // Get the size of the file associated to the stream.
  // Output:
  //   Return the size in bytes of the file, or 0 if the stream is not opened
  //   or any error occured.
  size_t (*getSize)(void);
} CapyStreamIo;

// Create a CapyStreamIo
// Output:
//   Return a CapyStreamIo
CapyStreamIo CapyStreamIoCreate(void);

// Allocate memory for a new CapyStreamIo and create it
// Output:
//   Return a CapyStreamIo
// Exception:
//   May raise CapyExc_MallocFailed.
CapyStreamIo* CapyStreamIoAlloc(void);

// Free the memory used by a CapyStreamIo* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyStreamIo to free
void CapyStreamIoFree(CapyStreamIo** const that);
#endif
