// -------------------------- supportVectorMachine.c -------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "supportVectorMachine.h"
#include "random.h"

// Free the memory used by a CapySVMKernel
static void SVMKernelDestruct(void) {
  return;
}

// Create a CapySVMKernel
// Output:
//   Return a CapySVMKernel
CapySVMKernel CapySVMKernelCreate(void) {
  CapySVMKernel that;
  that.destruct = SVMKernelDestruct;
  that.eval = NULL;
  that.exportToCFun = NULL;
  that.exportToJavascript = NULL;
  return that;
}

// Evaluation function for a linear kernel
// Input:
//   u: first input vector
//   v: second input vector
// Output:
//   Return the similarity or distance between the two input vectors
static double KernelLinearEval(
  CapyVec const* const u,
  CapyVec const* const v) {
  double dot;
  CapyVecDot(u, v, &dot);
  return dot;
}

// Export the kernel as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the kernel is written on the
//   stream.
static void ExportKernelLinearToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Linear kernel\n");
  fprintf(
    stream,
    "double %s("
    "double const* const u, double const* const v, size_t const n) {\n",
    name);
  fprintf(
    stream,
    "  double dot = 0.0;\n");
  fprintf(
    stream,
    "  for(size_t i = 0; i < n; ++i) {\n");
  fprintf(
    stream,
    "    dot += u[i] * v[i];\n");
  fprintf(
    stream,
    "  }\n");
  fprintf(
    stream,
    "  return dot;\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the kernel as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the kernel is written on
//   the stream.
static void ExportKernelLinearToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(stream, "// Linear kernel\n");
  fprintf(stream, "function %s(u, v, n) {\n", name);
  fprintf(stream, "  let dot = 0.0;\n");
  fprintf(stream, "  for(let i = 0; i < n; i = i + 1) {\n");
  fprintf(stream, "    dot += u[i] * v[i];\n");
  fprintf(stream, "  }\n");
  fprintf(stream, "  return dot;\n");
  fprintf(stream, "}\n");
}

// Free the memory used by a CapySVMKernelLinear
static void KernelLinearDestruct(void) {
  methodOf(CapySVMKernelLinear);
  $(that, destructCapySVMKernel)();
}

// Create a CapySVMKernelLinear
// Output:
//   Return a CapySVMKernelLinear
CapySVMKernelLinear CapySVMKernelLinearCreate(void) {
  CapySVMKernelLinear that;
  CapyInherits(that, CapySVMKernel, ());
  that.eval = KernelLinearEval;
  that.destruct = KernelLinearDestruct;
  that.exportToCFun = ExportKernelLinearToCFun;
  that.exportToJavascript = ExportKernelLinearToJavascript;
  return that;
}

// Allocate memory for a new CapySVMKernelLinear and create it
// Output:
//   Return a CapySVMKernelLinear
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMKernelLinear* CapySVMKernelLinearAlloc(void) {
  CapySVMKernelLinear* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySVMKernelLinearCreate();
  return that;
}

// Free the memory used by a CapySVMKernelLinear* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMKernelLinear to free
void CapySVMKernelLinearFree(CapySVMKernelLinear** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Evaluation function for a polynomial kernel
// Input:
//   u: first input vector
//   v: second input vector
// Output:
//   Return the similarity or distance between the two input vectors
static double KernelPolynomialEval(
  CapyVec const* const u,
  CapyVec const* const v) {
  methodOf(CapySVMKernelPolynomial);
  double dot;
  CapyVecDot(u, v, &dot);
  return pow(dot + that->theta, that->power);
}

// Export the kernel as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the kernel is written on the
//   stream.
static void ExportKernelPolynomialToCFun(
        FILE* const stream,
  char const* const name) {
  methodOf(CapySVMKernelPolynomial);
  fprintf(
    stream,
    "// Polynomial kernel\n");
  fprintf(
    stream,
    "double %s("
    "double const* const u, double const* const v, size_t const n) {\n",
    name);
  fprintf(
    stream,
    "  double dot = 0.0;\n");
  fprintf(
    stream,
    "  for(size_t i = 0; i < n; ++i) {\n");
  fprintf(
    stream,
    "    dot += u[i] * v[i];\n");
  fprintf(
    stream,
    "  }\n");
  fprintf(
    stream,
    "  dot = pow(dot + %lf, %lf);\n", that->theta, that->power);
  fprintf(
    stream,
    "  return dot;\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the kernel as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the kernel is written on
//   the stream.
static void ExportKernelPolynomialToJavascript(
        FILE* const stream,
  char const* const name) {
  methodOf(CapySVMKernelPolynomial);
  fprintf(stream, "// Linear kernel\n");
  fprintf(stream, "function %s(u, v, n) {\n", name);
  fprintf(stream, "  let dot = 0.0;\n");
  fprintf(stream, "  for(let i = 0; i < n; i = i + 1) {\n");
  fprintf(stream, "    dot += u[i] * v[i];\n");
  fprintf(stream, "  }\n");
  fprintf(
    stream,
    "  dot = Math.pow(dot + %lf, %lf);\n",
    that->theta, that->power);
  fprintf(stream, "  return dot;\n");
  fprintf(stream, "}\n");
}

// Free the memory used by a CapySVMKernelPolynomial
// Input:
//   that: the CapySVMKernelPolynomial to free
static void KernelPolynomialDestruct(void) {
  methodOf(CapySVMKernelPolynomial);
  $(that, destructCapySVMKernel)();
}

// Create a CapySVMKernelPolynomial
// Output:
//   Return a CapySVMKernelPolynomial
CapySVMKernelPolynomial CapySVMKernelPolynomialCreate(void) {
  CapySVMKernelPolynomial that;
  CapyInherits(that, CapySVMKernel, ());
  that.destruct = KernelPolynomialDestruct;
  that.eval = KernelPolynomialEval;
  that.exportToCFun = ExportKernelPolynomialToCFun;
  that.exportToJavascript = ExportKernelPolynomialToJavascript;
  that.power = 1.0;
  that.theta = 0.0;
  return that;
}

// Allocate memory for a new CapySVMKernelPolynomial and create it
// Output:
//   Return a CapySVMKernelPolynomial
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMKernelPolynomial* CapySVMKernelPolynomialAlloc(void) {
  CapySVMKernelPolynomial* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySVMKernelPolynomialCreate();
  return that;
}

// Free the memory used by a CapySVMKernelPolynomial* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMKernelPolynomial to free
void CapySVMKernelPolynomialFree(CapySVMKernelPolynomial** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Evaluation function for a Gaussian kernel
// Input:
//   u: first input vector
//   v: second input vector
// Output:
//   Return the similarity or distance between the two input vectors
static double KernelGaussianEval(
  CapyVec const* const u,
  CapyVec const* const v) {
  methodOf(CapySVMKernelGaussian);
  CapyVec w = CapyVecCreate(u->dim);
  CapyVecSub(u, v, &w);
  double n = CapyVecGetSquaredNorm(&w);
  CapyVecDestruct(&w);
  return exp(-1.0 * that->gamma * n);
}

// Export the kernel as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the kernel is written on the
//   stream.
static void ExportKernelGaussianToCFun(
        FILE* const stream,
  char const* const name) {
  methodOf(CapySVMKernelGaussian);
  fprintf(
    stream,
    "// Gaussian (radial basis) kernel\n");
  fprintf(
    stream,
    "double %s("
    "double const* const u, double const* const v, size_t const n) {\n",
    name);
  fprintf(
    stream,
    "  double x = 0.0;\n");
  fprintf(
    stream,
    "  for(size_t i = 0; i < n; ++i) {\n");
  fprintf(
    stream,
    "    double y = u[i] - v[i];\n");
  fprintf(
    stream,
    "    x += y * y;\n");
  fprintf(
    stream,
    "  }\n");
  fprintf(
    stream,
    "  double z = exp(-1.0 * %lf * x);\n", that->gamma);
  fprintf(
    stream,
    "  return z;\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the kernel as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the kernel is written on
//   the stream.
static void ExportKernelGaussianToJavascript(
        FILE* const stream,
  char const* const name) {
  methodOf(CapySVMKernelGaussian);
  fprintf(stream, "// Gaussian (radial basis) kernel\n");
  fprintf(stream, "function %s(u, v, n) {\n", name);
  fprintf(stream, "  let x = 0.0;\n");
  fprintf(stream, "  for(let i = 0; i < n; i = i + 1) {\n");
  fprintf(stream, "    let y = u[i] - v[i];\n");
  fprintf(stream, "    x += y * y;\n");
  fprintf(stream, "  }\n");
  fprintf(stream, "  let z = Math.exp(-1.0 * %lf * x);\n", that->gamma);
  fprintf(stream, "  return z;\n");
  fprintf(stream, "}\n");
}

// Free the memory used by a CapySVMKernelGaussian
// Input:
//   that: the CapySVMKernelGaussian to free
static void KernelGaussianDestruct(void) {
  methodOf(CapySVMKernelGaussian);
  $(that, destructCapySVMKernel)();
}

// Create a CapySVMKernelGaussian
// Output:
//   Return a CapySVMKernelGaussian
CapySVMKernelGaussian CapySVMKernelGaussianCreate(void) {
  CapySVMKernelGaussian that;
  CapyInherits(that, CapySVMKernel, ());
  that.destruct = KernelGaussianDestruct;
  that.eval = KernelGaussianEval;
  that.exportToCFun = ExportKernelGaussianToCFun;
  that.exportToJavascript = ExportKernelGaussianToJavascript;
  that.gamma = 1.0;
  return that;
}

// Allocate memory for a new CapySVMKernelGaussian and create it
// Output:
//   Return a CapySVMKernelGaussian
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMKernelGaussian* CapySVMKernelGaussianAlloc(void) {
  CapySVMKernelGaussian* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySVMKernelGaussianCreate();
  return that;
}

// Free the memory used by a CapySVMKernelGaussian* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMKernelGaussian to free
void CapySVMKernelGaussianFree(CapySVMKernelGaussian** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Free the memory used by a CapySVMEvaluation
// Input:
//   that: the CapySVMEvaluation to free
static void EvaluationDestruct(void) {
  methodOf(CapySVMEvaluation);
  $(that, destructCapyPredictorEvaluation)();
}

// Create a CapySVMEvaluation
// Output:
//   Return a CapySVMEvaluation
CapySVMEvaluation CapySVMEvaluationCreate(void) {
  CapySVMEvaluation that;
  CapyInherits(that, CapyPredictorEvaluation, ());
  that.destruct = EvaluationDestruct;
  return that;
}

// Allocate memory for a new CapySVMEvaluation and create it
// Output:
//   Return a CapySVMEvaluation
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMEvaluation* CapySVMEvaluationAlloc(void) {
  CapySVMEvaluation* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySVMEvaluationCreate();
  return that;
}

// Free the memory used by a CapySVMEvaluation* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySVMEvaluation to free
void CapySVMEvaluationFree(CapySVMEvaluation** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Functions implementing the Sequential Minimalist Optimisation algorithm
// cf: www.microsoft.com/en-us/research/wp-content/uploads/2016/02/tr-98-14.pdf
typedef struct SMOEvalArg {
  CapyMat const* mat;
  CapyRangeSize range;
  double const* alpha;
  CapySVMKernel* kernel;
  CapyVec const* x;
  double u;
} SMOEvalArg;

static void* SMOEvalThread(void* arg) {

  // Cast the argument
  CapyMat const* mat = ((SMOEvalArg*)arg)->mat;
  CapyRangeSize range = ((SMOEvalArg*)arg)->range;
  double const* alpha = ((SMOEvalArg*)arg)->alpha;
  CapySVMKernel* kernel = ((SMOEvalArg*)arg)->kernel;
  CapyVec const* x = ((SMOEvalArg*)arg)->x;

  // Evaluate the SVM on the thread's assigned rows
  loopRange(j, range) {
    CapyVec xj = {.dim = mat->nbCol - 1, .vals = mat->vals + j * mat->nbCol};
    double alphaj = alpha[j];
    double yj = mat->vals[(j + 1) * mat->nbCol - 1];
    ((SMOEvalArg*)arg)->u += yj * alphaj * $(kernel, eval)(&xj, x);
  }
  return NULL;
}

static double SMOEval(
  CapySupportVectorMachine* const that,
             CapyMat const* const mat,
             CapyVec const* const x,
              double const* const alpha) {

  // Split the evaluation into several threads
  size_t nbThread = 10;
  if(nbThread > mat->nbRow) nbThread = 1;
  pthread_t thread[nbThread];
  SMOEvalArg threadArgs[nbThread];
  loop(i, nbThread) threadArgs[i] = (SMOEvalArg){0};
  size_t nbRowPerThread = mat->nbRow / nbThread;
  loop(i, nbThread) {
    threadArgs[i] = (SMOEvalArg){
      .mat = mat,
      .range = {.min = i * nbRowPerThread, .max = (i + 1) * nbRowPerThread - 1},
      .alpha = alpha,
      .kernel = that->kernel,
      .x = x,
      .u = 0.0,
    };
    if(i == nbThread - 1) threadArgs[i].range.max = mat->nbRow - 1;
    int ret = pthread_create(thread + i, NULL, SMOEvalThread, threadArgs + i);
    if(ret != 0) raiseExc(CapyExc_ForkFailed);
  }

  // Wait for the thread to terminate and update the result
  double u = 0.0;
  loop(i, nbThread) {
    int ret = pthread_join(thread[i], NULL);
    if(ret != 0) raiseExc(CapyExc_ForkFailed);
    u += threadArgs[i].u;
  }
  u -= that->bias;

  // Return the result
  return u;
}

static double SMOError(
  CapySupportVectorMachine* const that,
             CapyMat const* const mat,
                     size_t const iRow,
                    double* const alpha) {
  CapyVec xi = {.dim = mat->nbCol - 1, .vals = mat->vals + iRow * mat->nbCol};
  double yi = mat->vals[(iRow + 1) * mat->nbCol - 1];
  double ui = SMOEval(that, mat, &xi, alpha);
  double error = ui - yi;
  return error;
}

// Structure for the arguments of SMOErrorUpdateThread
typedef struct SMOErrorUpdateArg {
  CapyMat const* mat;
  CapyRangeSize range;
  CapySVMKernel* kernel;
  double y1;
  double y2;
  CapyVec p1;
  CapyVec p2;
  double deltaAlpha1;
  double deltaAlpha2;
  double deltaB;
  double* error;
} SMOErrorUpdateArg;

static void* SMOErrorUpdateThread(void* arg) {
  CapyMat const* mat = ((SMOErrorUpdateArg*)arg)->mat;
  CapyRangeSize range = ((SMOErrorUpdateArg*)arg)->range;
  CapySVMKernel* kernel = ((SMOErrorUpdateArg*)arg)->kernel;
  double y1 = ((SMOErrorUpdateArg*)arg)->y1;
  double y2 = ((SMOErrorUpdateArg*)arg)->y2;
  CapyVec p1 = ((SMOErrorUpdateArg*)arg)->p1;
  CapyVec p2 = ((SMOErrorUpdateArg*)arg)->p2;
  double deltaAlpha1 = ((SMOErrorUpdateArg*)arg)->deltaAlpha1;
  double deltaAlpha2 = ((SMOErrorUpdateArg*)arg)->deltaAlpha2;
  double deltaB = ((SMOErrorUpdateArg*)arg)->deltaB;
  double* error = ((SMOErrorUpdateArg*)arg)->error;
  loopRange(i, range) {
    CapyVec pi =
      {.dim = mat->nbCol - 1, .vals = mat->vals + i * mat->nbCol};
    double k1 = $(kernel, eval)(&p1, &pi);
    double k2 = $(kernel, eval)(&p2, &pi);
    error[i] += y1 * deltaAlpha1 * k1 + y2 * deltaAlpha2 * k2 - deltaB;
  }
  return NULL;
}

static bool SMOStep(
  CapySupportVectorMachine* const that,
             CapyMat const* const mat,
                     size_t const iRow,
                     size_t const jRow,
                    double* const alpha,
                    double* const error) {
  double alph1 = alpha[iRow];
  double y1 = mat->vals[(iRow + 1) * mat->nbCol - 1];
  double alph2 = alpha[jRow];
  double y2 = mat->vals[(jRow + 1) * mat->nbCol - 1];
  double E1 = error[iRow];
  double E2 = error[jRow];
  double s = y1 * y2;

  // Compute L and H using eq 13 and 14
  double L = 0.0;
  double H = 0.0;
  if(s < 0.0) {
    L = fmax(0.0, alph2 - alph1);
    H = fmin(that->coeffRelax, that->coeffRelax + alph2 - alph1);
  } else {
    L = fmax(0.0, alph2 + alph1 - that->coeffRelax);
    H = fmin(that->coeffRelax, alph2 + alph1);
  }
  if(fabs(L - H) < DBL_EPSILON) return false;

  // Compute eta using eq 15
  CapyVec point1 =
    {.dim = mat->nbCol - 1, .vals = mat->vals + iRow * mat->nbCol};
  CapyVec point2 =
    {.dim = mat->nbCol - 1, .vals = mat->vals + jRow * mat->nbCol};
  double k11 = $(that->kernel, eval)(&point1, &point1);
  double k12 = $(that->kernel, eval)(&point1, &point2);
  double k22 = $(that->kernel, eval)(&point2, &point2);
  double eta = k11 + k22 - 2.0 * k12;

  // Compute the new second Lagrangian multiplier
  double a2 = 0.0;
  if(eta > DBL_EPSILON) {

    // The Kernel obeys Mercer's conditions, no problem
    a2 = alph2 + y2 * (E1 - E2) / eta;
    if(a2 < L) a2 = L;
    else if(a2 > H) a2 = H;
  } else {

    // The Kernel doesn't obey Mercer's conditions, calculate the objective
    // function at L and H using eq 19
    double f1 = y1 * (E1 + that->bias) - alph1 * k11 - s * alph2 * k12;
    double f2 = y2 * (E2 + that->bias) - s * alph1 * k12 - alph2 * k22;
    double L1 = alph1 + s * (alph2 - L);
    double H1 = alph1 + s * (alph2 - H);
    double Lobj =
      L1 * f1 + L * f2 + 0.5 * L1 * L1 * k11 + 0.5 * L * L * k22 +
      s * L * L1 * k12;
    double Hobj =
      H1 * f1 + H * f2 + 0.5 * H1 * H1 * k11 + 0.5 * H * H * k22 +
      s * H * H1 * k12;

    // Move the second Lagrangian toward the smallest value
    if(Lobj < Hobj - DBL_EPSILON) a2 = L;
    else if(Hobj < Lobj - DBL_EPSILON) a2 = H;
    else a2 = alph2;
  }

  // If the second Lagrangian hasn't changed, no need to go further
  if(fabs(a2 - alph2) < DBL_EPSILON) return false;

  // Compute the first second Lagrangian multiplier, eq 18
  // Here a1 is guaranteed to stay in [0,C] due to the way we calculate it and
  // a2
  double a1 = alph1 + s * (alph2 - a2);

  // Compute the new bias
  double deltaAlpha1 = a1 - alph1;
  double deltaAlpha2 = a2 - alph2;
  double b1 = E1 + y1 * deltaAlpha1 * k11 + y2 * deltaAlpha2 * k12 + that->bias;
  double b2 = E2 + y1 * deltaAlpha1 * k12 + y2 * deltaAlpha2 * k22 + that->bias;
  double b;
  if(that->tolerance < a1 && a1 < that->coeffRelax - that->tolerance) {
    b = b1;
  } else if(that->tolerance < a2 && a2 < that->coeffRelax - that->tolerance) {
    b = b2;
  } else b = 0.5 * (b1 + b2);

  // Update the Lagrangian
  alpha[iRow] = a1;
  alpha[jRow] = a2;

  // Update the error cache, we can avoid the costly SMOEval by calculating
  // directly the delta error based on the modification of the two Lagrangian
  CapyVec p1 =
    {.dim = mat->nbCol - 1, .vals = mat->vals + iRow * mat->nbCol};
  CapyVec p2 =
    {.dim = mat->nbCol - 1, .vals = mat->vals + jRow * mat->nbCol};
  double deltaB = b - that->bias;
  size_t nbThread = 10;
  if(nbThread > mat->nbRow) nbThread = 1;
  pthread_t thread[nbThread];
  SMOErrorUpdateArg threadArgs[nbThread];
  size_t nbRowPerThread = mat->nbRow / nbThread;
  loop(i, nbThread) {
    threadArgs[i] = (SMOErrorUpdateArg){
      .mat = mat,
      .range = {.min = i * nbRowPerThread, .max = (i + 1) * nbRowPerThread - 1},
      .kernel = that->kernel,
      .y1 = y1,
      .y2 = y2,
      .p1 = p1,
      .p2 = p2,
      .deltaAlpha1 = deltaAlpha1,
      .deltaAlpha2 = deltaAlpha2,
      .deltaB = deltaB,
      .error = error,
    };
    if(i == nbThread - 1) threadArgs[i].range.max = mat->nbRow - 1;
    int ret =
      pthread_create(thread + i, NULL, SMOErrorUpdateThread, threadArgs + i);
    if(ret != 0) raiseExc(CapyExc_ForkFailed);
  }

  // Wait for the thread to terminate and update the result
  loop(i, nbThread) {
    int ret = pthread_join(thread[i], NULL);
    if(ret != 0) raiseExc(CapyExc_ForkFailed);
  }

  // Upate the bias
  that->bias = b;

  // If we reach here, the step was successful
  return true;
}

static bool SMOViolatesKKT(
  CapySupportVectorMachine* const that,
             CapyMat const* const mat,
                     size_t const iRow,
              double const* const alpha) {
  double yi = mat->vals[(iRow + 1) * mat->nbCol - 1];
  CapyVec xi = {.dim = mat->nbCol - 1, .vals = mat->vals + iRow * mat->nbCol};
  double ui = SMOEval(that, mat, &xi, alpha);
  if(alpha[iRow] < that->tolerance) {
    if(yi * ui < 1.0 - that->tolerance) return true;
  } else if(alpha[iRow] > that->coeffRelax - that->tolerance) {
    if(yi * ui > 1.0 + that->tolerance) return true;
  } else {
    if(fabs(yi * ui - 1.0) > that->tolerance) return true;
  }
  return false;
}

static size_t SMOCountKKTViolation(
  CapySupportVectorMachine* const that,
             CapyMat const* const mat,
              double const* const alpha) {
  size_t nb = 0;
  loop(i, mat->nbRow) nb += SMOViolatesKKT(that, mat, i, alpha);
  return nb;
}

static size_t SMOShiftRow(
  size_t const jRow,
  size_t const shiftRow,
  size_t const nbRow) {
  size_t shiftJRow = 0;
  if(jRow >= nbRow - shiftRow) shiftJRow = jRow - (nbRow - shiftRow);
  else shiftJRow = jRow + shiftRow;
  return shiftJRow;
}

static size_t SMOExamine(
  CapySupportVectorMachine* const that,
             CapyMat const* const mat,
                     size_t const iRow,
                    double* const alpha,
                    double* const error,
                CapyRandom* const rng) {

  // Prepare a shifting value to loop over the rows without biasing toward
  // the first ones
  CapyRangeSize range = {.min = 0, .max = mat->nbRow - 1};
  size_t shiftRow = $(rng, getSizeRange)(&range);

  // If the Lagrangian multiplier of the iRow-th row violates the KKT
  // condition, it is eligible for optimisation
  if(SMOViolatesKKT(that, mat, iRow, alpha)) {

    // Select the second Lagrangian multiplier such as to maximise |E1 - E2|
    double max = 0.0;
    size_t jRow = 0;
    loop(j, mat->nbRow) {
      double e = fabs(error[iRow] - error[j]);
      if(max < e) {
        max = e;
        jRow = j;
      }
    }

    // Step the two selected Lagrangians
    bool ret = SMOStep(that, mat, iRow, jRow, alpha, error);
    if(ret) return 1;

    // If we reach here, it means we coudn't update the chosen pair of
    // Lagrangians. Try the next heuristic in the hierarchy.
    // Loop over all rows with non bounded Lagrangians
    size_t jMaxRow = jRow;
    loop(j, mat->nbRow) {
      jRow = SMOShiftRow(j, shiftRow, mat->nbRow);
      if(
        jRow != iRow &&
        jRow != jMaxRow &&
        that->tolerance < alpha[jRow] &&
        alpha[jRow] < that->coeffRelax - that->tolerance
      ) {

        // Step the two selected Lagrangians
        ret = SMOStep(that, mat, iRow, jRow, alpha, error);
        if(ret) return 1;
      }
    }

    // If we reach here, it means we coudn't update the chosen pair of
    // Lagrangians. Try the next heuristic in the hierarchy.
    // Loop over all possible second Lagrangian (here only bounded one remain)
    loop(j, mat->nbRow) {
      jRow = SMOShiftRow(j, shiftRow, mat->nbRow);
      if(jRow != iRow && jRow != jMaxRow) {
        if (
          alpha[jRow] <= that->tolerance ||
          that->coeffRelax - that->tolerance <= alpha[jRow]) {

          // Step the two selected Lagrangians
          ret = SMOStep(that, mat, iRow, jRow, alpha, error);
          if(ret) return 1;
        }
      }
    }
  }

  // If we reach here, it was completely impossible to update a pair of
  // Lagrangian using the iRow-th row for the first Lagrangian. Simply ignore
  // it and the outer loop will continue with the next row for the first
  // Lagrangian.
  return 0;
}

// Train the predictor on a dataset
// Input:
//   dataset: the dataset
// Output:
//   The predictor is trained.
// Exception:
//   May raise CapyExc_UnsupportedFormat
static void Train(CapyDataset const* const dataset) {
  methodOf(CapySupportVectorMachine);

  // Guard against unsolvable coefficient of relaxation
  if(that->coeffRelax <= that->tolerance) {
    raiseExc(CapyExc_InvalidParameters);
    return;
  }

  // Ensure the predicted field is categorical
  size_t idxOutput = $(dataset, getIdxOutputField)(that->iOutput);
  if(dataset->fields[idxOutput].type != capyDatasetFieldType_cat) {
    raiseExc(CapyExc_UnsupportedFormat);
    return;
  }

  // Allocate memory for the scaling ranges
  loop(i, that->nbInput) $(that->scalingFrom + i, destruct)();
  free(that->scalingFrom);
  that->nbInput = $(dataset, getNbInput)();
  safeMalloc(that->scalingFrom, that->nbInput);
  loop(i, that->nbInput) that->scalingFrom[i] = CapyRangeDoubleCreate(0.0, 1.0);

  // Convert the dataset to a matrix
  CapyMat mat = $(that, cvtDatasetToMat)(dataset);

  // Preprocess the input features in the training data
  $(that, scaleTrainingInputFeatures)(&mat, dataset);

  // Preprocess the output target in the training data
  loop(i, mat.nbRow) {
    if(fabs(mat.vals[(i + 1) * mat.nbCol - 1] - (double)(that->iCat)) < 1e-3) {
      mat.vals[(i + 1) * mat.nbCol - 1] = 1.0;
    } else {
      mat.vals[(i + 1) * mat.nbCol - 1] = -1.0;
    }
  }

  // Create the random number generator
  CapyRandom rng = CapyRandomCreate(that->seed);

  // Make sure to discard previous training
  CapyVecDestruct(&(that->lambda));
  CapyMatDestruct(&(that->support));

  // Allocate memory for the Lagrangian multipliers for each row
  double* alpha = NULL;
  safeMalloc(alpha, mat.nbRow);
  if(!alpha) return;

  // The Lagrangian multipliers must be in [0, C], and sum_i(y_i*alpha_i)=0
  // then initialise them to 0.0
  loop(i, mat.nbRow) alpha[i] = 0.0;

  // Reset the bias
  that->bias = 0.0;

  // Allocate memory for the error cache for each row
  double* error = NULL;
  safeMalloc(error, mat.nbRow);
  if(!error) return;

  // Calculate the initial error for each row with the initial value of the
  // Lagrangian multipliers
  loop(i, mat.nbRow) error[i] = SMOError(that, &mat, i, alpha);

  // Variables for the outer loop of the sequential minimal optimization
  // algorithm
  size_t nbChanged = 0;
  bool examineAll = true;

  // Counter to avoid infinite loop in case something went wrong,
  // give up if the counter becomes bigger than 100 times the number of rows
  // in the dataset
  size_t count = 0;
  size_t countThreshold = mat.nbRow;
  if(that->nbMaxIterTraining > 0) countThreshold = that->nbMaxIterTraining;

  // Outer loop
  while((nbChanged > 0 || examineAll) && count < countThreshold) {
    nbChanged = 0;

    // Iteration on the whole dataset
    if(examineAll) {
      if(that->verbose) {
        size_t nbViolation = SMOCountKKTViolation(that, &mat, alpha);
        println(
          "SVM training, #%lu/%lu, Nb violation %lu/%lu",
          count, countThreshold, nbViolation, mat.nbRow);
      }
      loop(iRow, mat.nbRow) {
        nbChanged += SMOExamine(that, &mat, iRow, alpha, error, &rng);
      }

    // Iteration on rows with non-bound Lagrangian multiplier
    } else {
      loop(iRow, mat.nbRow) {
        if(
          that->tolerance < alpha[iRow] &&
          alpha[iRow] < that->coeffRelax - that->tolerance
        ) {
          nbChanged += SMOExamine(that, &mat, iRow, alpha, error, &rng);
        }
      }
    }

    // Update conditions for outer loop
    if(examineAll) examineAll = false;
    else if(nbChanged == 0) examineAll = true;

    // Update the counter
    ++count;
  }

  // Count the number of support vector (rows with a strictly positive
  // Lagrangian multiplier)
  size_t nbSupport = 0;
  loop(i, mat.nbRow) if(alpha[i] > 1.0e-9) ++nbSupport;

  // Allocate memory for the support vectors and copy of Lagrangian multiplier
  that->support = CapyMatCreate(mat.nbCol, nbSupport);
  that->lambda = CapyVecCreate(nbSupport);

  // Copy the support vectors and their respective Lagrangian multiplier
  size_t iSupport = 0;
  loop(i, mat.nbRow) if(alpha[i] > 1.0e-9) {
    loop(j, mat.nbCol) {
      that->support.vals[iSupport * mat.nbCol + j] =
        mat.vals[i * mat.nbCol + j];
    }
    that->lambda.vals[iSupport] = alpha[i];
    ++iSupport;
  }

  // Memorise the reduction in number of support vector
  that->reducNbSupportVector =
    1.0 - ((double)nbSupport / (double)(mat.nbRow));

  // Free memory
  CapyMatDestruct(&mat);
  free(alpha);
  free(error);
}

// Predict the category of an input
// Input:
//   inp: the input (of dimension equal to the support vectors dimension)
// Output:
//   Return the result of prediction
static CapyPredictorPrediction Predict(CapyVec const* const inp) {
  methodOf(CapySupportVectorMachine);
  CapyPredictorPrediction pred;

  // Apply input features scaling
  CapyVec scaledInp = CapyVecCreate(inp->dim);
  loop(i, inp->dim) scaledInp.vals[i] = inp->vals[i];
  $(that, scaleInputFeatures)(&scaledInp);
  pred.confidence = 0.0;
  loop(i, that->support.nbRow) {
    CapyVec u = {
      .dim = that->support.nbCol - 1,
      .vals = that->support.vals + i * that->support.nbCol
    };
    pred.confidence +=
      that->lambda.vals[i] * $(that->kernel, eval)(&u, &scaledInp) *
      u.vals[that->support.nbCol - 1];
  }
  pred.confidence -= that->bias;
  pred.category = (pred.confidence >= 0.0);
  pred.confidence = fabs(pred.confidence);

  // Free memory
  CapyVecDestruct(&scaledInp);

  // Return the predicition
  return pred;
}

// Convert a CapyDataset into a CapyMat usable by the predictor
// Input:
//   dataset: the dataset to be converted
// Output:
//   Return a matrix formatted as necessary
static CapyMat CvtDatasetToMat(CapyDataset const* const dataset) {
  methodOf(CapySupportVectorMachine);
  CapyMat mat = $(dataset, cvtToMatForSingleCatPredictor)(that->iOutput);
  return mat;
}

// Set the kernel
// Input:
//   kernel: the kernel
// Output:
//   The reference to the kernel updated.
static void SetKernel(CapySVMKernel* const kernel) {
  methodOf(CapySupportVectorMachine);
  that->kernel = kernel;
}

// Evaluate the predictor on a dataset
// Input:
//   dataset: the dataset
// Output:
//   Return the evaluation of the predictor.
static CapyPredictorEvaluation* Evaluate(CapyDataset const* const dataset) {
  methodOf(CapySupportVectorMachine);

  // Variable to memorise the results
  CapySVMEvaluation* eval = CapySVMEvaluationAlloc();

  // Convert the dataset into a matrix
  CapyMat mat = $(that, cvtDatasetToMat)(dataset);

  // Allocate memory for the confusion matrix
  size_t nbCategory =
    $(dataset, getNbValOutputField)(that->iOutput);
  safeMalloc(eval->confusionMatrix, nbCategory * nbCategory);
  if(!(eval->confusionMatrix)) return NULL;
  loop(i, nbCategory * nbCategory) eval->confusionMatrix[i] = 0;

  // Get the number of input fields
  size_t nbInput = $(dataset, getNbInput)();

  // Loop on the rows
  loop(iRow, dataset->nbRow) {

    // Cast the row into an input vector
    CapyVec inpPredict = {
      .dim = nbInput,
      .vals = mat.vals + iRow * mat.nbCol
    };

    // Predict the category for that row
    CapyPredictorPrediction resPredict = $(that, predict)(&inpPredict);

    // Update the confusion matrix
    size_t idx =
      (size_t)round(mat.vals[(iRow + 1) * mat.nbCol - 1]) * nbCategory +
      resPredict.category;
    eval->confusionMatrix[idx] += 1;
  }

  // Calculate the overall accuracy from the confusion matrix
  size_t sum = 0;
  loop(i, nbCategory) {
    sum += eval->confusionMatrix[i * nbCategory + i];
  }
  eval->accuracies[capyPredictorAccuracyMeasure_accuracy] =
    (double)sum / (double)(dataset->nbRow);

  // Memorise the reduction in number of support vector
  eval->reducNbSupportVector = that->reducNbSupportVector;

  // Free memory
  CapyMatDestruct(&mat);

  // Return the results
  return (CapyPredictorEvaluation*)eval;
}

// Clone a CapySupportVectorMachine
// Output:
//   Return a clone of the predictor.
static void* Clone(void) {
  methodOf(CapySupportVectorMachine);
  CapySupportVectorMachine* clone = NULL;
  safeMalloc(clone, 1);
  if(!clone) return NULL;
  *clone = *that;
  clone->lambda = CapyVecCreate(that->lambda.dim);
  loop(i, that->lambda.dim) clone->lambda.vals[i] = that->lambda.vals[i];
  clone->support = CapyMatCreate(that->support.nbCol, that->support.nbRow);
  loop(i, that->support.nbCol * that->support.nbRow) {
    clone->support.vals[i] = that->support.vals[i];
  }
  return clone;
}

// Export the predictor as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
//   dataset: the training dataset
// Output:
//   A ready to use C function implementing the predictor is written on the
//   stream. See the comment exported with the function to know how to use
//   the exported function.
static void ExportToCFun(
               FILE* const stream,
         char const* const name,
  CapyDataset const* const dataset) {
  methodOf(CapySupportVectorMachine);

  // Write the include-s
  fprintf(stream, "#include <stdlib.h>\n");
  fprintf(stream, "#include <stdio.h>\n");
  fprintf(stream, "#include <string.h>\n");
  fprintf(stream, "#include <math.h>\n");
  fprintf(stream, "\n");

  // Export the kernel as a C function
  size_t kernelFunNameLength = strlen(name) + 7;
  char kernelFunName[kernelFunNameLength];
  snprintf(kernelFunName, kernelFunNameLength, "%sKernel", name);
  $(that->kernel, exportToCFun)(stream, kernelFunName);

  // Write the prediction function's comment block
  fprintf(stream, "// Support vector machine prediction\n");
  fprintf(stream, "// Input:\n");
  size_t nbInput = that->support.nbCol - 1;
  fprintf(
    stream,
    "// u: the predicted input, array of %lu double values as follow:\n",
    nbInput);
  loop(iInput, nbInput) {
    size_t iFieldInput = $(dataset, getIdxInputField)(iInput);
    fprintf(
      stream,
      "// u[%lu]: [%s], ",
      iInput,
      dataset->fields[iFieldInput].label);
    if(dataset->fields[iFieldInput].type == capyDatasetFieldType_num) {
      fprintf(
        stream,
        "trained on [%.9lf, %.9lf]\n",
        dataset->fields[iFieldInput].range.min,
        dataset->fields[iFieldInput].range.max);
    } else {
      fprintf(stream, "encoded as ");
      unsigned long n = dataset->fields[iFieldInput].nbCategoryVal - 1;
      if(n < 1) n = 1;
      loop(iVal, dataset->fields[iFieldInput].nbCategoryVal) {
        fprintf(
          stream,
          "[%s]=%lu%s",
          dataset->fields[iFieldInput].categoryVals[iVal], iVal,
          iVal < (dataset->fields[iFieldInput].nbCategoryVal - 1) ? ", " : "");
      }
      fprintf(stream, "\n");
    }
  }
  fprintf(stream, "// Output:\n");
  size_t iFieldOutput =
    $(dataset, getIdxOutputField)(that->iOutput);
  char const* lblPredCat = dataset->fields[iFieldOutput].label;
  char const* lblPredVal =
    dataset->fields[iFieldOutput].categoryVals[that->iCat];
  char const* lblPredNotVal =
    dataset->fields[iFieldOutput].categoryVals[1 - that->iCat];
  fprintf(
    stream,
    "// Return a positive value if the predicted category [%s] for the given\n"
    "// input is [%s], else return a negative value. The larger the absolute\n"
    "// value of the returned value, the more confident the prediction is.\n",
    lblPredCat, lblPredVal);

  // Write the prediction function's header
  fprintf(
    stream,
    "int %s(double const* const u) {\n",
    name);

  // Write the support vector values
  size_t nbSupport = that->support.nbRow;
  fprintf(
    stream,
    "  double const support[%lu][%lu] = {\n",
    nbSupport, nbInput + 1);
  loop(iSupport, nbSupport) {
    fprintf(stream, "    {");
    loop(iInput, nbInput + 1) {
      fprintf(
        stream,
        "%a%s",
        that->support.vals[iSupport * (nbInput + 1) + iInput],
        iInput < nbInput ? ", " : "");
    }
    fprintf(stream, "},\n");
  }
  fprintf(stream, "  };\n");

  // Write the lambda values
  fprintf(
    stream,
    "  double const lambda[%lu] = {",
    nbSupport);
  loop(iSupport, nbSupport) {
    fprintf(
      stream,
      "%a%s",
      that->lambda.vals[iSupport], iSupport < nbSupport - 1 ? ", " : "");
  }
  fprintf(stream, "};\n");

  // Scale the input features
  $(that, exportScaleInputToCFun)(stream);

  // Write the prediction algorithm
  fprintf(stream, "  double x = 0.0;\n");
  fprintf(stream, "  for(int i = 0; i < %lu; ++i) {\n", nbSupport);
  fprintf(
    stream,
    "    x += lambda[i] * support[i][%lu] * %sKernel(support[i], w, %lu);\n",
    nbInput, name, nbInput);
  fprintf(stream, "  }\n");
  fprintf(stream, "  x -= (double)%a;\n", that->bias);
  fprintf(stream, "  return x;\n");
  fprintf(stream, "}\n");
  fprintf(stream, "\n");

  // Write the driver function
  fprintf(stream, "// Driver function\n");
  fprintf(stream, "int main(int argc, char** argv) {\n");
  fprintf(stream, "  if(argc != %lu) {\n", nbInput + 1);
  fprintf(stream, "    printf(\"Expect %lu arguments\\n\");\n", nbInput);
  fprintf(stream, "    return 1;\n");
  fprintf(stream, "  }\n");
  fprintf(stream, "  double u[%lu] = {};\n", nbInput);
  loop(iInput, nbInput) {
    size_t iFieldInput = $(dataset, getIdxInputField)(iInput);
    if(dataset->fields[iFieldInput].type == capyDatasetFieldType_num) {
      fprintf(stream, "  u[%lu] = atof(argv[%lu]);\n", iInput, iInput + 1);
    } else {
      loop(iVal, dataset->fields[iFieldInput].nbCategoryVal) {
        unsigned long n = dataset->fields[iFieldInput].nbCategoryVal - 1;
        if(n < 1) n = 1;
        fprintf(
          stream,
          "  if(strcmp(argv[%lu], \"%s\") == 0) u[%lu] = %lu.0;\n",
          iInput + 1,
          dataset->fields[iFieldInput].categoryVals[iVal],
          iInput, iVal);
      }
    }
  }
  fprintf(stream, "\n");
  fprintf(stream, "  double pred = %s(u);\n", name);
  fprintf(
    stream,
    "  printf(\"%%s\\n\", pred > 0.0 ? \"%s\" : \"%s\");\n",
    lblPredVal, lblPredNotVal);
  fprintf(stream, "  return 0;\n");
  fprintf(stream, "}\n");
  fprintf(stream, "\n");
}

// Export the predictor as a HTML web app
// Input:
//   stream: the stream where to export
//   title: the title of the web app
//   dataset: the training dataset
//   expectedAccuracy: the expected accuracy of the predictor (in [0,1])
// Output:
//   A ready to use web app implementing the predictor is written on the
//   stream.
static void ExportToHtml(
               FILE* const stream,
         char const* const title,
  CapyDataset const* const dataset,
              double const expectedAccuracy) {
  methodOf(CapySupportVectorMachine);

  // Write the head of the page (up to <body>)
  fprintf(stream, "<!DOCTYPE html>\n");
  fprintf(stream, "<html>\n");

  // Write the body
  $(that, exportBodyToHtml)(stream, title, dataset, expectedAccuracy);

  // Write the javascript
  fprintf(stream, "<script>\n");
  fprintf(
    stream,
    "function elem(id) { return document.getElementById(id); }\n");
  $(that->kernel, exportToJavascript)(stream, "kernel");
  fprintf(stream, "function predict() {\n");
  fprintf(stream, "  let u = getInput();\n");
  size_t nbInput = $(dataset, getNbInput)();
  size_t nbSupport = that->support.nbRow;
  fprintf(stream, "  let support = [];\n");
  loop(iSupport, nbSupport) {
    fprintf(stream, "  support[%lu] = [", iSupport);
    loop(iInput, nbInput + 1) {
      fprintf(
        stream,
        "%s%.9lf",
        (iInput == 0 ? "" : ", "),
        that->support.vals[iSupport * (nbInput + 1) + iInput]);
    }
    fprintf(stream, "];\n");
  }
  fprintf(stream, "  let lambda = [");
  loop(iSupport, nbSupport) {
    fprintf(
      stream,
      "%s%.9lf",
      (iSupport == 0 ? "" : ", "),
      that->lambda.vals[iSupport]);
  }
  fprintf(stream, "];\n");
  fprintf(stream, "  let x = 0.0;\n");
  fprintf(stream, "  for(let i = 0; i < %ld; i = i + 1) {\n", nbSupport);
  fprintf(
    stream,
    "    x += lambda[i] * support[i][%lu] * kernel(support[i], u, %lu);\n",
    nbInput, nbInput);
  fprintf(stream, "  }\n");
  fprintf(stream, "  x -= %lf;\n", that->bias);
  fprintf(
    stream,
    "  elem(\"spanConfidence\").innerHTML = Math.abs(x).toFixed(2);\n");
  fprintf(stream, "  elem(\"pred0\").checked = false;\n");
  fprintf(stream, "  elem(\"pred1\").checked = false;\n");
  if(that->iCat == 0) {
    fprintf(stream, "  if(x > 0.0) elem(\"pred0\").checked = true;\n");
    fprintf(stream, "  else elem(\"pred1\").checked = true;\n");
  } else {
    fprintf(stream, "  if(x > 0.0) elem(\"pred1\").checked = true;\n");
    fprintf(stream, "  else elem(\"pred0\").checked = true;\n");
  }
  fprintf(stream, "}\n");
  fprintf(stream, "</script>\n");

  // Write the end of the page
  fprintf(stream, "</html>\n");
}

// Free the memory used by a CapySupportVectorMachine
static void Destruct(void) {
  methodOf(CapySupportVectorMachine);
  $(that, destructCapyPredictor)();
  CapyVecDestruct(&(that->lambda));
  CapyMatDestruct(&(that->support));
}

// Create a CapySupportVectorMachine
// Output:
//   Return a CapySupportVectorMachine
CapySupportVectorMachine CapySupportVectorMachineCreate(void) {
  CapySupportVectorMachine that;
  CapyInherits(that, CapyPredictor, (capyPredictorType_categorical));
  that.kernel = NULL;
  that.coeffRelax = 1.0e-2;
  that.tolerance = 1.0e-3;
  that.lambda = (CapyVec){0};
  that.support = (CapyMat){0};
  that.verbose = false;
  that.nbMaxIterTraining = 0;
  that.bias = 0.0;
  that.destruct = Destruct;
  that.train = Train;
  that.predict = Predict;
  that.cvtDatasetToMat = CvtDatasetToMat;
  that.setKernel = SetKernel;
  that.evaluate = Evaluate;
  that.seed = 0;
  that.iCat = 1;
  that.clone = Clone;
  that.exportToCFun = ExportToCFun;
  that.exportBodyToHtml = that.exportToHtml;
  that.exportToHtml = ExportToHtml;
  return that;
}

// Allocate memory for a new CapySupportVectorMachine and create it
// Output:
//   Return a CapySupportVectorMachine
// Exception:
//   May raise CapyExc_MallocFailed.
CapySupportVectorMachine* CapySupportVectorMachineAlloc(void) {
  CapySupportVectorMachine* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySupportVectorMachineCreate();
  return that;
}

// Free the memory used by a CapySupportVectorMachine* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySupportVectorMachine to free
void CapySupportVectorMachineFree(CapySupportVectorMachine** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
