// -------------------------- supportVectorMachine.h -------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_SUPPORT_VECTOR_MACHINE_H
#define CAPY_SUPPORT_VECTOR_MACHINE_H
#include "externalHeaders.h"
#include "cext.h"
#include "predictor.h"
#include "random.h"

// Description:
// Support vector machine class.

// SVMKernel parent class
#define CapySVMKernelDef        \
struct {                        \
  void (*destruct)(void);       \
  double (*eval)(               \
    CapyVec const* const u,     \
    CapyVec const* const v);    \
  void (*exportToCFun)(         \
    FILE* const stream,         \
    char const* const name);    \
  void (*exportToJavascript)(   \
    FILE* const stream,         \
    char const* const name);    \
}

// SVMKernel structure
typedef CapySVMKernelDef CapySVMKernel;

// Create a CapySVMKernel
// Output:
//   Return a CapySVMKernel
CapySVMKernel CapySVMKernelCreate(void);

// SVMKernelLinear class
typedef struct CapySVMKernelLinear {

  // Inherits CapySVMKernel
  CapySVMKernelDef;

  // Destructor
  void (*destructCapySVMKernel)(void);
} CapySVMKernelLinear;

// Create a CapySVMKernelLinear
// Output:
//   Return a CapySVMKernelLinear
CapySVMKernelLinear CapySVMKernelLinearCreate(void);

// Allocate memory for a new CapySVMKernelLinear and create it
// Output:
//   Return a CapySVMKernelLinear
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMKernelLinear* CapySVMKernelLinearAlloc(void);

// Free the memory used by a CapySVMKernelLinear
// Input:
//   that: the CapySVMKernelLinear to free
void CapySVMKernelLinearDestruct(CapySVMKernelLinear* const that);

// Free the memory used by a CapySVMKernelLinear* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMKernelLinear to free
void CapySVMKernelLinearFree(CapySVMKernelLinear** const that);

// SVMKernelPolynomial class
typedef struct CapySVMKernelPolynomial {

  // Inherits CapySVMKernel
  CapySVMKernelDef;

  // Power of the polynomial (default: 1.0)
  double power;

  // Threshold (default: 0.0)
  double theta;

  // Destructor
  void (*destructCapySVMKernel)(void);
} CapySVMKernelPolynomial;

// Create a CapySVMKernelPolynomial
// Output:
//   Return a CapySVMKernelPolynomial
CapySVMKernelPolynomial CapySVMKernelPolynomialCreate(void);

// Allocate memory for a new CapySVMKernelPolynomial and create it
// Output:
//   Return a CapySVMKernelPolynomial
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMKernelPolynomial* CapySVMKernelPolynomialAlloc(void);

// Free the memory used by a CapySVMKernelPolynomial* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMKernelPolynomial to free
void CapySVMKernelPolynomialFree(CapySVMKernelPolynomial** const that);

// SVMKernelGaussian class (Gaussian Radial Basis kernel, aka Gaussian kernel
// or Radial Basis kernel)
typedef struct CapySVMKernelGaussian {

  // Inherits CapySVMKernel
  CapySVMKernelDef;

  // Gamma (default: 1.0)
  double gamma;

  // Destructor
  void (*destructCapySVMKernel)(void);
} CapySVMKernelGaussian;

// Create a CapySVMKernelGaussian
// Output:
//   Return a CapySVMKernelGaussian
CapySVMKernelGaussian CapySVMKernelGaussianCreate(void);

// Allocate memory for a new CapySVMKernelGaussian and create it
// Output:
//   Return a CapySVMKernelGaussian
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMKernelGaussian* CapySVMKernelGaussianAlloc(void);

// Free the memory used by a CapySVMKernelGaussian* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMKernelGaussian to free
void CapySVMKernelGaussianFree(CapySVMKernelGaussian** const that);

// PredictorEvaluation, class to memorise the result of evaluation of a
// predictor
typedef struct CapySVMEvaluation {

  // Parent class
  CapyPredictorEvaluationDef;

  // Reduction coefficient of the support vector (=1-nbSupport/nbExample)
  double reducNbSupportVector;

  // Destructor
  void (*destructCapyPredictorEvaluation)(void);
} CapySVMEvaluation;

// Create a CapySVMEvaluation
// Output:
//   Return a CapySVMEvaluation
CapySVMEvaluation CapySVMEvaluationCreate(void);

// Allocate memory for a new CapySVMEvaluation and create it
// Output:
//   Return a CapySVMEvaluation
// Exception:
//   May raise CapyExc_MallocFailed.
CapySVMEvaluation* CapySVMEvaluationAlloc(void);

// Free the memory used by a CapySVMEvaluation* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySVMEvaluation to free
void CapySVMEvaluationFree(CapySVMEvaluation** const that);

// SupportVectorMachine object
typedef struct CapySupportVectorMachine {

  // Inherit CapyPredictor
  CapyPredictorDef;

  // Reference to the used kernel
  CapySVMKernel* kernel;

  // Lagrangian multipliers
  CapyVec lambda;

  // Bias
  double bias;

  // Support vectors (one vector per row, input values (normalised in [0,1])
  // followed by category in {-1,1}
  CapyMat support;

  // Tolerance (default: 1.0e-3)
  double tolerance;

  // Margin relaxation coefficient (default: 1e-2, the higher the more
  // sensitive to outliers, must be strictly greater than that.tolerance)
  double coeffRelax;

  // Seed for the pseudo random generator used to shuffle the rows during
  // training
  CapyRandomSeed_t seed;

  // Index of the predicted category (default: 1) 
  size_t iCat;

  // Reduction coefficient of the support vector (=1-nbSupport/nbExample),
  // used during evaluation of the SVM
  double reducNbSupportVector;

  // Maximum number of iteration during training (default: 0, if equal to 0
  // the number rows in the dataset is used instead)
  size_t nbMaxIterTraining;

  // Flag for verbose mode (default: false)
  bool verbose;
  CapyPad(bool, 1);

  // Destructor
  void (*destructCapyPredictor)(void);

  // Function herited from the parent to export the body to HTML. 
  // Input:
  //   stream: the stream where to export
  //   title: the title of the web app
  //   dataset: the training dataset
  //   expectedAccuracy: the expected accuracy of the predictor (in [0,1])
  // Output:
  //   The <head> and <body> part of a ready to use web app implementing the
  //   predictor is written on the stream. The web app can be completed by
  //   calling exportToHtml on the predictor to write the <script> part.
  void (*exportBodyToHtml)(
                 FILE* const stream,
           char const* const title,
    CapyDataset const* const dataset,
                double const expectedAccuracy);

  // Set the kernel
  // Input:
  //   kernel: the kernel
  // Output:
  //   The reference to the kernel updated.
  void (*setKernel)(CapySVMKernel* const kernel);
} CapySupportVectorMachine;

// Create a CapySupportVectorMachine
// Output:
//   Return a CapySupportVectorMachine
CapySupportVectorMachine CapySupportVectorMachineCreate(void);

// Allocate memory for a new CapySupportVectorMachine and create it
// Output:
//   Return a CapySupportVectorMachine
// Exception:
//   May raise CapyExc_MallocFailed.
CapySupportVectorMachine* CapySupportVectorMachineAlloc(void);

// Free the memory used by a CapySupportVectorMachine* and reset '*that' to
// NULL
// Input:
//   that: a pointer to the CapySupportVectorMachine to free
void CapySupportVectorMachineFree(CapySupportVectorMachine** const that);
#endif
