// -------------------------------- trycatch.c --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "trycatch.h"

// Stack of jmp_buf to memorise the TryCatch blocks
// To avoid exposing this variable to the user, implement any code using
// it as functions here instead of in the #define-s of trycatch.h
static _Thread_local jmp_buf tryCatchExcJmp[CAPY_MAX_TRYCATCHLVL];

// Index of the next TryCatch block in the stack of jmp_buf
// To avoid exposing this variable to the user, implement any code using
// it as functions here instead of in the #define-s of trycatch.h
static _Thread_local int16_t tryCatchExcLvl = 0;

// ID of the last raised exception
// To avoid exposing this variable to the user, implement any code using
// it as functions here instead of in the #define-s of trycatch.h
// Do not use the type enum CapyException to allow the user to extend
// the list of exceptions with user-defined exceptions outside of enum
// CapyException.
static _Thread_local CapyException_t tryCatchExc = 0;

// Flag to memorise if we are inside a catch block at a given exception level
static _Thread_local bool flagInCatchBlock[CAPY_MAX_TRYCATCHLVL] = {false};

// Label for the TryCatchExceptions
static char const* exceptionStr[CapyExc_LastID] = {
  "",
  "Too many conversion functions for CapyException, "
  "see trycatch.c:nbMaxUserDefinedExcToStr",
  "Dynamic memory (re-)allocation failed",
  "Opening the stream failed",
  "Reading from the stream failed",
  "Writing to the stream failed",
  "Not receiving the expected data from a stream "
  "(invalid file, corrupted transmission, ...)",
  "Failed to create a child process",
  "The definition of the arguments of the CLI, or the given arguments, "
  "are invalid",
  "The execution reached an unexpected situation and couldn't proceed",
  "A value or the result of a calculation overflowed the valid range",
  "No color chart could be located in the image",
  "Matrix inversion failed",
  "Trying to access or use a node with invalid index in a graph",
  "Trying to access or use a list element with invalid index in a graph",
  "Couldn't retrieve the current time",
  "Couldn't achieve QR decomposition",
  "Unsupported format",
  "An algorithm couldn't execute given the input parameters it received",
  "Floating point value imprecision affects computation",
  "The RSA modulus is smaller than the ciphered/deciphered message",
  "Trying to access a state with invalid index in a MarkovDecisionProcess",
  "Trying to access an action with invalid index in a MarkovDecisionProcess",
  "Potential infinite loop detected and aborted"
};

// Buffer to build default label for user defined exceptions
// Size of the buffer is calculated as length of "User-defined exception
//  ()" plus enough space to hold the representation of an int
static char userDefinedExceptionDefaultLabel[50];

// Max number of user-defined functions used to convert user-defined
// exception ID to strings
#define nbMaxUserDefinedExcToStr 256

// Current number of user-defined functions used to convert user-defined
// exception to strings
static int nbUserDefinedExcToStr = 0;

// Pointers to user-defined functions used to convert user-defined
// exception ID to strings
static char const* (*userDefinedExcToStr[nbMaxUserDefinedExcToStr])(
  CapyException_t);

// Stream to print out a message each time Raise is called
static FILE* streamRaise = NULL;

// Function called at the beginning of a TryCatch block to guard against
// overflow of the stack of jump_buf. If an overflow does occur,
// immediately exit with code EXIT_FAILURE.
void CapyTryCatchGuardOverflow(void) {

  // If the max level of incursion is reached
  if(tryCatchExcLvl == CAPY_MAX_TRYCATCHLVL) {

    // Print a message on the standard error output and exit
    fprintf(
      stderr,
      "TryCatch blocks recursive incursion overflow, exiting. "
      "(You can try to raise the value of CAPY_MAX_TRYCATCHLVL in "
      "trycatch.h, it was: %d)\n",
      CAPY_MAX_TRYCATCHLVL);
    exit(EXIT_FAILURE);
  }
}

// Function called to get the jmp_buf on the top of the stack when
// starting a new TryCatch block
// Output:
//   Remove the jmp_buf on the top of the stack and return it
jmp_buf* CapyTryCatchGetJmpBufOnStackTop(void) {

  // Reset the last raised exception
  tryCatchExc = 0;

  // Memorise the current jmp_buf at the top of the stack
  jmp_buf* topStack = tryCatchExcJmp + tryCatchExcLvl;

  // Move the index of the top of the stack of jmp_buf to the upper level
  tryCatchExcLvl++;

  // Return the jmp_buf previously at the top of the stack
  return topStack;
}

// Function called to raise the CapyException 'exc'. An uncaught
// exception has no effect and simply fall through.
// Input:
//        exc: The exception to raise. Do not use the type enum
//             CapyException to allow the user to extend the list of
//             exceptions with user-defined exception outside of enum
//             CapyException.
//   filename: File where the exception has been raised
//       line: Line where the exception has been raised
void CapyRaise(
  CapyException_t const exc,
      char const* const filename,
          int16_t const line) {
#if BUILD_MODE == 0
  assert(filename != NULL);
#endif

  // If the stream to record exception raising is set and the raised
  // exception do not come from trycatch.c (to avoid unnecessary
  // repeatition in the trace), print the exception
  // on the stream
  bool retStrCmp = strcmp(filename, __FILE__);
  if(streamRaise != NULL && retStrCmp != 0) {
    fprintln(
      streamRaise,
      "Exception (%s) raised in %s, line %d.",
      CapyExcToStr(exc),
      filename,
      line);
  }
  if(tryCatchExcLvl > 0) {

    // Memorise the last raised exception to be able to handle it if
    // it reaches the default case in the swith statement of the TryCatch
    // block
    tryCatchExc = exc;

    // Get the level in the stack where to jump back
    int jumpTo = (tryCatchExcLvl > 0 ? tryCatchExcLvl - 1 : 0);

    // Call longjmp with the appropriate jmp_buf in the stack and the
    // raised TryCatchException.
    longjmp(tryCatchExcJmp[jumpTo], exc);
  }
}

// Function called when entering a catch block
void CapyTryCatchEnterCatchBlock(void) {

  // Update the flag
  flagInCatchBlock[tryCatchExcLvl - 1] = true;
}

// Function called when exiting a catch block
void CapyTryCatchExitCatchBlock(void) {

  // Update the flag
  flagInCatchBlock[tryCatchExcLvl] = false;
}

// Function called at the end of a TryCatch block
void CapyTryCatchEnd(void) {

  // The execution has reached the end of the current TryCatch block,
  // move back to the lower level in the stack of jmp_buf
  if(tryCatchExcLvl > 0) tryCatchExcLvl--;
}

// Function to get the ID of the last raised exception
// Output:
//   Return the id of the last raised exception
CapyException_t CapyGetLastExcId(void) {

  // Return the ID
  return tryCatchExc;
}

// Function to convert an exception ID to char*
// Input:
//   exc: The exception ID
// Output:
//   Return the stringified exception
char const* CapyExcToStr(CapyException_t exc) {

  // Declare the pointer to the result string
  char const* excStr = NULL;

  // If the exception ID is one of TryCatchException
  if(exc < CapyExc_LastID) excStr = exceptionStr[exc];

  // Loop on user-defined conversion functions
  loop(iFun, nbUserDefinedExcToStr) {

    // Get the conversion using this function
    char const* str = (*userDefinedExcToStr[iFun])(exc);

    // If the exception ID could be converted
    if(str != NULL) {

      // If there was already another result of conversion,
      // it means there is an ID conflict
      if(excStr != NULL) {
        fprintln(
          stderr,
          "!!! TryCatch: Exception ID conflict, "
          "between (%s) and (%s) !!!",
          str,
          excStr);
      }

      // Update the pointer the result
      excStr = str;
    }
  }

  // If we haven't find a conversion yet
  if(excStr == NULL) {

    // Create a default string
    sprintf(
      userDefinedExceptionDefaultLabel,
      "User-defined exception (%d)",
      exc);
    excStr = userDefinedExceptionDefaultLabel;
  }

  // Return the converted exception to string
  return excStr;
}

// Function to add a function used by TryCatch to convert user-defined
// exception to a string. The function in argument must return NULL if
// its argument is not an exception ID it is handling, else a pointer to
// a string.
// It is highly recommended to provide conversion functions to cover
// all the user defined exceptions as it also allows TryCatch to detect
// conflict between exception IDs.
// Input:
//   fun: The conversion function to add
void CapyAddExcToStrFun(char const* (*fun)(CapyException_t)) {
#if BUILD_MODE == 0
  assert(fun != NULL);
#endif

  // If the buffer of pointer to conversion function is full, raise
  // the exception TooManyExcToStrFun
  if(nbUserDefinedExcToStr >= nbMaxUserDefinedExcToStr) {
    raiseExc(CapyExc_TooManyExcToStrFun);
    return;
  }

  // Loop on the pointer to conversion functions,
  // if this is the function in argument, avoid adding it again
  loop(iFun, nbUserDefinedExcToStr) {
    if(userDefinedExcToStr[iFun] == fun) return;
  }

  // If we reach here this conversion function hasn't been added yet,
  // add it
  userDefinedExcToStr[nbUserDefinedExcToStr] = fun;

  // Increment the number of conversion functions
  ++nbUserDefinedExcToStr;
}

// Set the stream on which to print messages when exception are raised, set
// it to NULL to turn off messages. (By default it's NULL)
// Input:
//   stream: The stream to used
void CapySetRaiseStream(FILE* const stream) {

  // Set the stream
  streamRaise = stream;
}

// Get the stream on which the messages are print when exception are raised.
// Output:
//   Return the used stream
FILE* CapyGetRaiseStream(void) {
  return streamRaise;
}

// Function to forward the current exception, if any, to the eventual
// TryCatch block including the CapyForwardExc call.
// Note that even if an exception is forwarded from inside a parallel
// section a TryCatch block outside the parallel section may not see it.
void CapyForwardExc(void) {

  // If there is a currently raised exception, reraise it
  if(tryCatchExc != 0) raiseExc(tryCatchExc);
}

// Function to cancel the current exception if any
void CapyCancelExc(void) {

  // Cancel the exception
  tryCatchExc = 0;
}
