// -------------------------------- trycatch.h --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_TRYCATCH_H
#define CAPY_TRYCATCH_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Exception management.

// Size of the stack of TryCatch blocks, define how many recursive incursion
// of TryCatch blocks can be done, overflow is checked at the beginning of
// each TryCatch blocks with CapyTryCatchGuardOverflow()
#define CAPY_MAX_TRYCATCHLVL 256

// Type of an exception ID. This must be compatible with an enumeration type.
typedef int CapyException_t;

// List of exceptions ID, must start at 1 (0 is reserved for the setjmp at
// the beginning of the TryCatch blocks). If the user wants to define and use
// its own exceptions, their ID must not be in [0, CapyExc_LastID - 1]
typedef enum CapyException {
  CapyExc_TooManyExcToStrFun = 1,
  CapyExc_MallocFailed,
  CapyExc_StreamOpenError,
  CapyExc_StreamReadError,
  CapyExc_StreamWriteError,
  CapyExc_InvalidStream,
  CapyExc_ForkFailed,
  CapyExc_InvalidCLIArg,
  CapyExc_UndefinedExecution,
  CapyExc_NumericalOverflow,
  CapyExc_NoColorChartInImage,
  CapyExc_MatrixInversionFailed,
  CapyExc_InvalidNodeIdx,
  CapyExc_InvalidElemIdx,
  CapyExc_NoTimeAvailable,
  CapyExc_QRDecompositionFailed,
  CapyExc_UnsupportedFormat,
  CapyExc_InvalidParameters,
  CapyExc_FloatingPointImprecision,
  CapyExc_RSAModulusTooSmall,
  CapyExc_InvalidStateIdx,
  CapyExc_InvalidActionIdx,
  CapyExc_InfiniteLoop,
  CapyExc_LastID
} CapyException;

// Function called at the beginning of a TryCatch block to guard against
// overflow of the stack of jump_buf
void CapyTryCatchGuardOverflow(void);

// Function called to get the jmp_buf on the top of the stack when
// starting a new TryCatch block
// Output:
//   Remove the jmp_buf on the top of the stack and return it
jmp_buf* CapyTryCatchGetJmpBufOnStackTop(void);

// Function called when entering a catch block
void CapyTryCatchEnterCatchBlock(void);

// Function called when exiting a catch block
void CapyTryCatchExitCatchBlock(void);

// Function called at the end of a TryCatch block
void CapyTryCatchEnd(void);

// Head of the TryCatch block, to be used as
//
// try {
//   /*... code of the TryCatch block here ...*/
//
// Comments on the macro:
//   // Guard against recursive incursion overflow
//   CapyTryCatchGuardOverflow();
//   // Memorise the jmp_buf on the top of the stack, setjmp returns 0
//   switch(setjmp(*CapyTryCatchGetJmpBufOnStackTop())) {
//     // Entry point for the code of the TryCatch block
//     case 0:
#define try                                             \
  CapyTryCatchGuardOverflow();                          \
  switch(setjmp(*CapyTryCatchGetJmpBufOnStackTop())) {  \
    case 0:

// Catch segment in the TryCatch block, to be used as
//
// try {
//   /*... code of the TryCatch block here ...*/
// } catch (/*... one of CapyException or user-defined exception ...*/) {
//   /*... code executed if the exception has been raised in the
//     TryCatch block ...*/
//
// Comments on the macro:
//      // Exit the previous Catch block
//      CapyTryCatchExitCatchBlock();
//      // End of the previous case
//      break;
//    // case of the raised exception
//    case e:
//      // Flag the entrance into the Catch block
//      CapyTryCatchEnterCatchBlock();
#define catch(e)                    \
      CapyTryCatchExitCatchBlock(); \
      break;                        \
    case e:                         \
      CapyTryCatchEnterCatchBlock();

// Macro to assign several exceptions to one Catch segment in the TryCatch
// block, to be used as
//
// try {
//   /*... code of the TryCatch block here ...*/
// } catch (/*... one of CapyException or user-defined exception ...*/)
// catchAlso (/*... another one ...*/) {
// /*... as many catchAlso statement as you need ...*/
//   /*... code executed if one of the exception has been raised in the
//     TryCatch block ...
//     (Use CapyGetLastExcId() if you need to know which exception as
//     been raised) */
//
// Comments on the macro:
//      // Avoid the fall through warning due to the
//      // CapyTryCatchEnterCatchBlock() at the entrance of the catch case
//      /* fall through */
//    // case of the raised exception
//    case e:
//      // Flag the entrance into the Catch block
//      CapyTryCatchEnterCatchBlock();
#define catchAlso(e) /* fall through */ \
    case e:                             \
      CapyTryCatchEnterCatchBlock();

// Macro to declare the default Catch segment in the TryCatch
// block, must be the last Catch segment in the TryCatch block,
// to be used as
//
// try {
//   /*... code of the TryCatch block here ...*/
// } catchDefault {
//   /*... code executed if an exception has been raised in the
//     TryCatch block and hasn't been catched by a previous Catch segment...
//     (Use CapyGetLastExcId() if you need to know which exception as
//     been raised) */
//
// Comments on the macro:
//      // Exit the previous Catch block
//      CapyTryCatchExitCatchBlock();
//      // End of the previous case
//      break;
//    // default case
//    default:
//      // Flag the entrance into the Catch block
//      CapyTryCatchEnterCatchBlock();
#define catchDefault                \
      CapyTryCatchExitCatchBlock(); \
      break;                        \
    default:                        \
        CapyTryCatchEnterCatchBlock();

// Tail of the TryCatch block, to be used as
//
// try {
//   /*... code of the TryCatch block here ...*/
// } endCatch;
//
// Comments on the macro:
//      // Exit the previous Catch block
//      CapyTryCatchExitCatchBlock();
//      // End of the previous case
//      break;
//    // default case, i.e. any raised exception which hasn't been catched
//    // by a previous Catch is catched here
//    default:
//      // Processing of uncatched exception
//      CapyTryCatchDefault();
//  // End of the switch statement at the head of the TryCatch block
//  }
//  // Post processing of the TryCatchBlock
//  CapyTryCatchEnd()
#define endCatch                    \
      CapyTryCatchExitCatchBlock(); \
      break;                        \
  }                                 \
  CapyTryCatchEnd()

// Function called to raise the CapyException 'exc'. An uncaught
// exception has no effect and simply fall through.
// Input:
//        exc: The CapyException to raise. Do not use the type enum
//             CapyException to allow the user to extend the list of
//             exceptions with user-defined exception outside of enum
//             CapyException.
//   filename: File where the exception has been raised
//       line: Line where the exception has been raised
void CapyRaise(
  CapyException_t const exc,
      char const* const filename,
          int16_t const line);

// Macro to raise the CapyException exc, to be used as
//
// raiseExc( /* ... one of CapyException or user-defined exception ... */);
#define raiseExc(exc) CapyRaise(exc, __FILE__, __LINE__)

// Macro to recatch and forward an exception. This is usefull when an exception
// may be raised by a handler, in which case the trace loose track of where
// the exception has occured. By recatching the block of code susceptible
// of triggering the handler, one can ensure the trace will properly indicates
// this block of code as the source of the exception. To be used as
//
// recatch(/* ... block of code eventually raising an exception ... */);
#define recatch(block)                                   \
  do {                                                   \
    Try { block; }                                       \
    catchDefault { raiseExc(CapyTryCatchGetLastExc()); } \
    endCatch;                                            \
  } while(false)

// Function to get the ID of the last raised exception
// Output:
//   Return the id of the last raised exception
CapyException_t CapyGetLastExcId(void);

// Function to convert an exception ID to char*. It uses the conversion
// functions provided via CapyAddExcToStrFun() to convert user defined
// exception. CapyException exceptions are converted automatically.
// Input:
//   exc: The exception ID
// Output:
//   Return the stringified exception
char const* CapyExcToStr(CapyException_t const exc);

// Function to add a function used by TryCatch to convert user-defined
// exception to a string. The function in argument must return NULL if
// its argument is not an exception ID it is handling, else a pointer to
// a string.
// It is highly recommended to provide conversion functions to cover
// all the user defined exceptions as it also allows TryCatch to detect
// conflict between exception IDs.
// Input:
//   fun: The conversion function to add
void CapyAddExcToStrFun(char const* (*fun)(CapyException_t));

// Set the stream on which to print messages when exception are raised, set
// it to NULL to turn off messages. (By default it's NULL)
// Input:
//   stream: The stream to used
void CapySetRaiseStream(FILE* const stream);

// Get the stream on which the messages are print when exception are raised.
// Output:
//   Return the used stream
FILE* CapyGetRaiseStream(void);

// Function to forward the current exception, if any, to the eventual
// TryCatch block including the CapyForwardExc call.
// Note that even if an exception is forwarded from inside a parallel
// section a TryCatch block outside the parallel section may not see it.
void CapyForwardExc(void);

// Function to cancel the current exception if any
void CapyCancelExc(void);

// End of the guard against multiple inclusion
#endif
