// ---------------------------- turtlegraphic.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "turtlegraphic.h"
#include "pen.h"

// List of states
CapyDefList(CapyTurtleGraphicStack, CapyTurtleGraphicState)

// Destructor of a CapyTurtleGraphicState
static void StateDestruct(void) {
  methodOf(CapyTurtleGraphicState);
  CapyVecDestruct(&(that->pos));
  CapyVecDestruct(&(that->step));
}

// Forward command
// Input:
//   step: the turtle moves forward 'step' steps while drawing a line
static void Forward(uint64_t const step) {
  methodOf(CapyTurtleGraphic);
  double from[2];
  loop(i, 2) from[i] = that->state.pos.vals[i];
  $(that, move)(step);
  if(that->img) {
    CapyPen pen = CapyPenCreate();
    pen.color = that->color;
    $(&pen, drawLine)(from, that->state.pos.vals, that->img);
    $(&pen, destruct)();
  }
}

// Move command
// Input:
//   step: the turtle moves forward 'step' steps without drawing a line
static void Move(uint64_t const step) {
  methodOf(CapyTurtleGraphic);
  loop(i, step) {
    CapyVecAdd(&(that->state.pos), &(that->state.step), &(that->state.pos));
  }
}

// Turn command
// Input:
//   theta: 'theta' is added to the turtle direction
static void Turn(double const theta) {
  methodOf(CapyTurtleGraphic);
  CapyVec2DSetAngle(
    &(that->state.step),
    CapyVec2DGetAngle(&(that->state.step), &capyXAxis2D) + theta,
    &(that->state.step));
}

// Resize command
// Input:
//   scale: the turle step distance is multiplied by 'scale'
static void Resize(double const scale) {
  methodOf(CapyTurtleGraphic);
  double len = CapyVecGetNorm(&(that->state.step)) * scale;
  CapyVecNormalise(&(that->state.step));
  CapyVecMul(&(that->state.step), len, &(that->state.step));
}

// Push command
// Output:
//   The current state is added on the top of the stack
static void Push(void) {
  methodOf(CapyTurtleGraphic);
  $(that->stack, push)(that->state);
  CapyVec pos = CapyVecCreate(2);
  loop(i, 2) pos.vals[i] = that->state.pos.vals[i];
  that->state.pos = pos;
  CapyVec step = CapyVecCreate(2);
  loop(i, 2) step.vals[i] = that->state.step.vals[i];
  that->state.step = step;
}

// Pop command
// Output:
//   The state on the top of the stack is removed and becomes the current
//   state.
static void Pop(void) {
  methodOf(CapyTurtleGraphic);
  if(that->stack->head) {
    $(&(that->state), destruct)();
    that->state = $(that->stack, pop)();
  }
}

// Free the memory used by a CapyTurtleGraphic
static void Destruct(void) {
  methodOf(CapyTurtleGraphic);
  CapyVecDestruct(&(that->state.pos));
  CapyVecDestruct(&(that->state.step));
  CapyTurtleGraphicStackFree(&(that->stack));
}

// Create a CapyTurtleGraphic
// Output:
//   Return a CapyTurtleGraphic
CapyTurtleGraphic CapyTurtleGraphicCreate(void) {
  CapyTurtleGraphic that = {
    .state = {
      .pos = CapyVecCreate(2),
      .step = CapyVecCreate(2),
      .destruct = StateDestruct
    },
    .stack = CapyTurtleGraphicStackAlloc(),
    .img = NULL,
    .color = capyColorRGBAWhite,
    .destruct = Destruct,
    .forward = Forward,
    .move = Move,
    .turn = Turn,
    .resize = Resize,
    .push = Push,
    .pop = Pop,
  };
  that.state.step.vals[0] = 1.0;
  return that;
}

// Allocate memory for a new CapyTurtleGraphic and create it
// Output:
//   Return a CapyTurtleGraphic
// Exception:
//   May raise CapyExc_MallocFailed.
CapyTurtleGraphic* CapyTurtleGraphicAlloc(void) {
  CapyTurtleGraphic* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyTurtleGraphicCreate();
  return that;
}

// Free the memory used by a CapyTurtleGraphic* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyTurtleGraphic to free
void CapyTurtleGraphicFree(CapyTurtleGraphic** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
