// ---------------------------------- turtlegraphic.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_TURTLEGRAPHIC_H
#define CAPY_TURTLEGRAPHIC_H
#include "externalHeaders.h"
#include "cext.h"
#include "capymath.h"
#include "list.h"
#include "image.h"

// Description:
// TurtleGraphic class.

// Structure to memorise the state of the turtle
typedef struct CapyTurtleGraphicState {

  // Step vector
  CapyVec step;

  // Position
  CapyVec pos;

  // Destructor
  void (*destruct)(void);
} CapyTurtleGraphicState;

// List of states
CapyDecList(CapyTurtleGraphicStack, CapyTurtleGraphicState)

// TurtleGraphic object
typedef struct CapyTurtleGraphic {

  // Current state (initially, stepLength=1, theta=0);
  CapyTurtleGraphicState state;

  // List of stacked states
  CapyTurtleGraphicStack* stack;

  // Image on which the turle draws (default: null)
  CapyImg* img;

  // Color of the line (default: white)
  CapyColorData color;

  // Destructor
  void (*destruct)(void);

  // Forward command
  // Input:
  //   step: the turtle moves forward 'step' steps while drawing a line
  void (*forward)(uint64_t const step);

  // Move command
  // Input:
  //   step: the turtle moves forward 'step' steps without drawing a line
  void (*move)(uint64_t const step);

  // Turn command
  // Input:
  //   theta: 'theta' is added to the turtle direction
  void (*turn)(double const theta);

  // Resize command
  // Input:
  //   scale: the turle step distance is multiplied by 'scale'
  void (*resize)(double const scale);

  // Push command
  // Output:
  //   The current state is added on the top of the stack
  void (*push)(void);

  // Pop command
  // Output:
  //   The state on the top of the stack is removed and becomes the current
  //   state.
  void (*pop)(void);
} CapyTurtleGraphic;

// Create a CapyTurtleGraphic
// Output:
//   Return a CapyTurtleGraphic
CapyTurtleGraphic CapyTurtleGraphicCreate(void);

// Allocate memory for a new CapyTurtleGraphic and create it
// Output:
//   Return a CapyTurtleGraphic
// Exception:
//   May raise CapyExc_MallocFailed.
CapyTurtleGraphic* CapyTurtleGraphicAlloc(void);

// Free the memory used by a CapyTurtleGraphic* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyTurtleGraphic to free
void CapyTurtleGraphicFree(CapyTurtleGraphic** const that);
#endif
