// ---------------------- verletintegration.c ---------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "verletintegration.h"

// Approximate X(t) using Verlet integration
// Input:
//   t: value for which we want to approximate X(t)
//   vals: values [t0, X(t0), X'(t0)]
// Output:
//   Update 'vals' with the approximated values at 't'
static void Eval(
   double const t,
  double* const vals) {
  methodOf(CapyVerletIntegration);
  while(vals[0] < t) {
    $(that, step)(vals);
  }
}

// Run one step
// Input:
//   vals: current values [t0, X(t0), X'(t0)]
// Output:
//   'vals' is updated with the result of one step of Verlet integration
static void Step(double* const vals) {
  methodOf(CapyVerletIntegration);

  // Calculate the derivative at current time step, the derivative should not
  // rely on X' (only on t and X)
  double* const d0 = that->vecs[0].vals;
  $(that->derivative, eval)(vals, d0);

  // Update t and X
  loop(i, that->derivative->dimIn) {
    if(i == 0) {
      vals[i] += that->deltaT;
    } else if(i <= that->dimVar) {
      vals[i] += that->deltaT * (
        vals[that->dimVar + i] + 0.5 * d0[i - 1] * that->deltaT);
    }
  }

  // Calculate the derivative at next time step, the derivative should not
  // rely on X' (only on t and X)
  double* const d1 = that->vecs[1].vals;
  $(that->derivative, eval)(vals, d1);

  // Update X'
  loop(i, that->derivative->dimIn) {
    if(i > that->dimVar) {
      vals[i] += that->deltaT * 0.5 * (
        d0[i - that->dimVar - 1] + d1[i - that->dimVar - 1]);
    }
  }
}

// Free the memory used by a CapyVerletIntegration
static void Destruct(void) {
  methodOf(CapyVerletIntegration);
  CapyVecDestruct(&(that->initVal));
  loop(i, 2) CapyVecDestruct(that->vecs + i);
  *that = (CapyVerletIntegration){0};
}

// Create a CapyVerletIntegration
// Input:
//   derivative: the derivative function, should not rely on X' (only on t and
//               X)
// Output:
//   Return a CapyVerletIntegration
CapyVerletIntegration CapyVerletIntegrationCreate(
  CapyMathFun* const derivative) {
  if(derivative->dimIn < 2) {
    raiseExc(CapyExc_InvalidParameters);
  }
  CapyVerletIntegration that = {
    .derivative = derivative,
    .dimVar = (derivative->dimIn - 1) / 2,
    .initVal = CapyVecCreate(derivative->dimIn),
    .deltaT = 1e-3,
    .eval = Eval,
    .step = Step,
    .destruct = Destruct,
  };
  if(
    derivative->dimOut != that.dimVar ||
    that.dimVar * 2 + 1 != derivative->dimIn
  ) {
    raiseExc(CapyExc_InvalidParameters);
  }
  loop(i, 2) that.vecs[i] = CapyVecCreate(that.dimVar);
  return that;
}

// Allocate memory for a new CapyVerletIntegration and create it
// Input:
//   derivative: the derivative function, should not rely on X' (only on t and
//               X)
// Output:
//   Return a CapyVerletIntegration
// Exception:
//   May raise CapyExc_MallocFailed.
CapyVerletIntegration* CapyVerletIntegrationAlloc(
  CapyMathFun* const derivative) {
  CapyVerletIntegration* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyVerletIntegrationCreate(derivative);
  return that;
}

// Free the memory used by a CapyVerletIntegration* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyVerletIntegration to free
void CapyVerletIntegrationFree(CapyVerletIntegration** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
