// --------------------------- verletintegration.h --------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_VERLETINTEGRATION_H
#define CAPY_VERLETINTEGRATION_H
#include "externalHeaders.h"
#include "cext.h"
#include "mathfun.h"

// Description:
// VerletIntegration implementation.

// VerletIntegration object
typedef struct CapyVerletIntegration {

  // Derivative function F().
  // F(t, X, X') = X''
  // The derivative should not rely on X' (only on t and X)
  CapyMathFun* derivative;

  // Number of variables (automatically set based on derivation.nbInput and
  // order)
  size_t dimVar;

  // Initial values [t0, X0, X'0] (default: all 0)
  CapyVec initVal;

  // Temporary vectors for calculation
  CapyVec vecs[2];

  // Discretise step delta t (default: 1e-3)
  double deltaT;

  // Approximate X(t) using Verlet integration
  // Input:
  //   t: value for which we want to approximate X(t)
  //   vals: values [t0, X(t0), X'(t0)]
  // Output:
  //   Update 'vals' with the approximated values at 't'
  void (*eval)(
     double const t,
    double* const vals);

  // Run one step
  // Input:
  //   vals: values [t0, X(t0), X'(t0)]
  // Output:
  //   'vals' is updated with the result of one step of Verlet integration
  void (*step)(double* const vals);

  // Destructor
  void (*destruct)(void);
} CapyVerletIntegration;

// Create a CapyVerletIntegration
// Input:
//   derivative: the derivative function, should not rely on X' (only on t and
//               X)
// Output:
//   Return a CapyVerletIntegration
CapyVerletIntegration CapyVerletIntegrationCreate(
  CapyMathFun* const derivative);

// Allocate memory for a new CapyVerletIntegration and create it
// Input:
//   derivative: the derivative function, should not rely on X' (only on t and
//               X)
// Output:
//   Return a CapyVerletIntegration
// Exception:
//   May raise CapyExc_MallocFailed.
CapyVerletIntegration* CapyVerletIntegrationAlloc(
  CapyMathFun* const derivative);

// Free the memory used by a CapyVerletIntegration* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyVerletIntegration to free
void CapyVerletIntegrationFree(CapyVerletIntegration** const that);
#endif
