// ------------------------------ voting.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "voting.h"

// Set a voter's ranking for a given choice
// Input:
//   iVoter: id of the voter
//   iRanking: updated ranking
//   iChoice: id of the choice for the given voter and ranking
// Output:
//   The choice for he given rank and voter is updated
static void RankedChoiceVotingSetRanking(
   size_t const iVoter,
  uint8_t const iRanking,
  uint8_t const iChoice) {
  methodOf(CapyRankedChoiceVoting);
  that->votes[iVoter].rankings[iRanking] = iChoice;
}

// Get the result of election
// Output:
//   Check the integrity of the votes and if they are correct return the
//   result, else raise CapyExc_InvalidParameters. The voters' choice is
//   updated.
static CapyRankedChoiceVotingResult RankedChoiceVotingGetResult(void) {
  methodOf(CapyRankedChoiceVoting);

  // Variable to memorise the result
  CapyRankedChoiceVotingResult result = {0};

  // Loop on the voter
  loop(iVoter, that->nbVoter) {

    // Loop on the rankings
    loop(iChoice, that->nbChoice) {

      // If the id of the choice is invalid, raise an exception
      if(that->votes[iVoter].rankings[iChoice] >= that->nbChoice) {
        raise(CapyExc_InvalidParameters);
      }

      // If the id of that ranking is set at another ranking, raise an exception
      for(int jChoice = iChoice + 1; jChoice < that->nbChoice; ++jChoice) {
        uint8_t const* rankings = that->votes[iVoter].rankings;
        if(rankings[iChoice] == rankings[jChoice]) {
          raise(CapyExc_InvalidParameters);
        }
      }
    }
  }

  // Variable to memorise the active choice
  bool activeChoices[256] = {0};
  loop(iChoice, that->nbChoice) activeChoices[iChoice] = true;

  // Variable to memorise the number of active choices
  uint8_t nbActiveChoice = that->nbChoice;

  // Loop until there are more than one choice
  while(nbActiveChoice > 1) {

    // Dispatch the votes
    loop(i, 256) if(activeChoices[i]) result.nbVotes[i] = 0;
    loop(iVoter, that->nbVoter) {
      CapyRankedChoiceVote const* const voter = that->votes + iVoter;
      result.nbVotes[voter->rankings[voter->choice]] += 1;
    }

    // Get the number of vote of the active choice with minimal number of vote
    size_t nbMinVote = that->nbVoter;
    loop(iChoice, that->nbChoice) if(activeChoices[iChoice]) {
      if(nbMinVote > result.nbVotes[iChoice]) {
        nbMinVote = result.nbVotes[iChoice];
      }
    }

    // Eliminate active choices with minimal number of vote
    nbActiveChoice = 0;
    loop(iChoice, that->nbChoice) if(activeChoices[iChoice]) {
      if(nbMinVote == result.nbVotes[iChoice]) {
        activeChoices[iChoice] = false;
      } else {
        nbActiveChoice += 1;
      }
    }

    // What happens if nbActiveChoice==0 ? It means there is a tie between
    // remaining choices. Should we continue by stepping all voters ? I can't
    // find descriptions of this case. In the present implementation it
    // simply stops.
    //
    // If there are still more than one choice
    if(nbActiveChoice > 1) {

      // Loop on the voters
      loop(iVoter, that->nbVoter) {
        CapyRankedChoiceVote* const voter = that->votes + iVoter;

        // If the voter's current choice is eliminated
        if(activeChoices[voter->rankings[voter->choice]] == false) {

          // Set the voter's choice to its next choice among active choices
          do {
            voter->choice += 1;
          } while(activeChoices[voter->rankings[voter->choice]] == false);
        }
      }
    }
  }
  return result;
}

// Free the memory used by a CapyRankedChoiceVoting
static void Destruct(void) {
  methodOf(CapyRankedChoiceVoting);
  loop(iVoter, that->nbVoter) {
    free(that->votes[iVoter].rankings);
  }
  free(that->votes);
}

// Create a CapyRankedChoiceVoting
// Input:
//   nbChoice: number of choice
//   nbVoter: number of voter
// Output:
//   Return a CapyRankedChoiceVoting
CapyRankedChoiceVoting CapyRankedChoiceVotingCreate(
  uint8_t const nbChoice,
   size_t const nbVoter) {
  CapyRankedChoiceVoting that = {
    .nbChoice = nbChoice,
    .nbVoter = nbVoter,
    .destruct = Destruct,
    .setRanking = RankedChoiceVotingSetRanking,
    .getResult = RankedChoiceVotingGetResult,
  };
  safeMalloc(that.votes, nbVoter);
  loop(iVoter, nbVoter) {
    safeMalloc(that.votes[iVoter].rankings, nbChoice);
    loop(iChoice, nbChoice) {
      that.votes[iVoter].rankings[iChoice] = (uint8_t)iChoice;
    }
    that.votes[iVoter].choice = 0;
  }
  return that;
}

// Allocate memory for a new CapyRankedChoiceVoting and create it
// Input:
//   nbChoice: number of choice
//   nbVoter: number of voter
// Output:
//   Return a CapyRankedChoiceVoting
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRankedChoiceVoting* CapyRankedChoiceVotingAlloc(
  uint8_t const nbChoice,
   size_t const nbVoter) {
  CapyRankedChoiceVoting* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyRankedChoiceVotingCreate(nbChoice, nbVoter);
  return that;
}

// Free the memory used by a CapyRankedChoiceVoting* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRankedChoiceVoting to free
void CapyRankedChoiceVotingFree(CapyRankedChoiceVoting** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
