// ---------------------------------- voting.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_VOTING_H
#define CAPY_VOTING_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Voting systems implementation.

// Choice for one voter in the RankedChoiceVoting
typedef struct CapyRankedChoiceVote {

  // Current choice (default: 0)
  uint8_t choice;
  CapyPad(uint8_t, choice);

  // Ranking of choices (initialised to 0, 1, 2, ...)
  uint8_t* rankings;
} CapyRankedChoiceVote;

// Result of the voting
typedef struct CapyRankedChoiceVotingResult {

  // Number of votes for each choice before being eliminated (so the winner(s)
  // can be determined by decreasing number of votes).
  size_t nbVotes[256];
} CapyRankedChoiceVotingResult;

// RankedChoiceVoting object
typedef struct CapyRankedChoiceVoting {

  // Number of choice
  uint8_t nbChoice;
  CapyPad(uint8_t, nbChoice);

  // Number of voter
  size_t nbVoter;

  // Voter's choices
  CapyRankedChoiceVote* votes;

  // Destructor
  void (*destruct)(void);

  // Set a voter's ranking for a given choice
  // Input:
  //   iVoter: id of the voter
  //   iRanking: updated ranking
  //   iChoice: id of the choice for the given voter and ranking
  // Output:
  //   The choice for he given rank and voter is updated
  void (*setRanking)(
     size_t const iVoter,
    uint8_t const iRanking,
    uint8_t const iChoice);

  // Get the result of election
  // Output:
  //   Check the integrity of the votes and if they are correct return the
  //   result, else raise CapyExc_InvalidParameters. The voters' choice is
  //   updated.
  CapyRankedChoiceVotingResult (*getResult)(void);
} CapyRankedChoiceVoting;

// Create a CapyRankedChoiceVoting
// Input:
//   nbChoice: number of choice
//   nbVoter: number of voter
// Output:
//   Return a CapyRankedChoiceVoting.
CapyRankedChoiceVoting CapyRankedChoiceVotingCreate(
  uint8_t const nbChoice,
   size_t const nbVoter);

// Allocate memory for a new CapyRankedChoiceVoting and create it
// Input:
//   nbChoice: number of choice
//   nbVoter: number of voter
// Output:
//   Return a CapyRankedChoiceVoting
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRankedChoiceVoting* CapyRankedChoiceVotingAlloc(
  uint8_t const nbChoice,
   size_t const nbVoter);

// Free the memory used by a CapyRankedChoiceVoting* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRankedChoiceVoting to free
void CapyRankedChoiceVotingFree(CapyRankedChoiceVoting** const that);
#endif
