// -------------------------------- x11display.c --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "x11display.h"

// Default display name
static char const* const defaultDisplayName = ":0.0";

// Size of the buffer event
#define EVT_BUFFER_SIZE 1024

// Indices for the frame and refresh rates property of CapyX11Display
typedef enum RateIdx {
  frame, refresh
} RateIdx;

// CapyX11Display structure
struct CapyX11Display {

  // X11 data
  Display* display;
  Window window;
  Window parentWindow;
  long evtMask;
  Atom wmDelete;
  XEvent event;
  char* displayName;
  GC gc;
  XImage image;

  // Dimensions of the display
  CapyX11Dims_t width;
  CapyX11Dims_t height;

  // Flag to memorise if the child thread is running
  bool isChildRunning;
  CapyPad(bool, isChildRunning);

  // Flag to memorise the request for the termination of the child
  bool flagTerminateThread;
  CapyPad(bool, flagTerminateThread);

  // Thread data
  pthread_t thread;
  pthread_attr_t threadAttr;

  // Counters and timers for the frame rate and refresh rate
  struct {
    size_t count;
    struct timespec start;
    struct timespec stop;
  } rates[2];

  // Timer for the whole window (measure time during display only)
  struct timespec timer;

  // Event buffer
  CapyX11DisplayEvt evts[EVT_BUFFER_SIZE];
  size_t evtHead;
  size_t evtTail;

  // Last known position of the pointer
  CapyX11DisplayPos pointerPos;

  // Buffers to manage displayed data
  CapyX11DisplayBuffer buffers[3];
  int inputBuffer;
  int outputBuffer;
};

// Set the title of the window
// Input:
//    that: pointer to the CapyX11Display instance
//   title: the window's new title
void CapyX11DisplaySetTitle(
  CapyX11Display const* const that,
            char const* const title) {
  XStoreName(that->display, that->window, title);
}

// Set a pixel color in the buffer.
// Input:
//   pos: the pixel coordinates (0,0 at top left corner, y downward)
//   rgb: the color to be set
// Output:
//   Set the pixel's rgb values. Do nothing if the coordinates are out of
//   the buffer.
static void SetPixel(
  CapyX11DisplayPos const* const pos,
         CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  if(
    pos->x >= 0 &&
    pos->x < that->width &&
    pos->y >= 0 &&
    pos->y < that->height
  ) {
    loop(i, 3) {
      that->rgb[(pos->y * that->width + pos->x) * 3 + i] = rgb->vals[i];
    }
  }
}

// Set an antialiased pixel color in the buffer.
// Input:
//   pos: the pixel coordinates (0,0 at top left corner, y downward)
//   rgb: the color to be set
// Output:
//   Set the pixel's rgb values. Do nothing if the coordinates are out of
//   the buffer. The antialiasing is done using the Xiaolin Wu's method
static void SetAntialiasedPixel(
  CapyPoint2D const* const pos,
   CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);

  // If the pixel is in the buffer or has a neighbour in the buffer
  if(
    pos->x >= -1.0 &&
    pos->x <= (double)(that->width) &&
    pos->y >= -1.0 &&
    pos->y < (double)(that->height)
  ) {

    // 3x3 matrix of pixels position around the pixel
    CapyX11Dims_t pixels[3][3][2] = {
      {
        {(CapyX11Dims_t)(pos->x) - 1, (CapyX11Dims_t)(pos->y) - 1},
        {(CapyX11Dims_t)(pos->x), (CapyX11Dims_t)(pos->y) - 1},
        {(CapyX11Dims_t)(pos->x) + 1, (CapyX11Dims_t)(pos->y) - 1}
      },
      {
        {(CapyX11Dims_t)(pos->x) - 1, (CapyX11Dims_t)(pos->y)},
        {(CapyX11Dims_t)(pos->x), (CapyX11Dims_t)(pos->y)},
        {(CapyX11Dims_t)(pos->x) + 1, (CapyX11Dims_t)(pos->y)}
      },
      {
        {(CapyX11Dims_t)(pos->x) - 1, (CapyX11Dims_t)(pos->y) + 1},
        {(CapyX11Dims_t)(pos->x), (CapyX11Dims_t)(pos->y) + 1},
        {(CapyX11Dims_t)(pos->x) + 1, (CapyX11Dims_t)(pos->y) + 1}
      },
    };

    // Calculate the weight of each pixel in the matrix
    double dummy;
    double const dx = modf(pos->x, &dummy) - 0.5;
    double const dy = modf(pos->y, &dummy) - 0.5;
    double weights[3][3] = {
      {
        (dx > 0.0 ? 0.0 : -dx * 2.0) * (dy > 0.0 ? 0.0 : -dy * 2.0),
        (1.0 - fabs(dx * 2.0)) * (dy > 0.0 ? 0.0 : -dy * 2.0),
        (dx < 0.0 ? 0.0 : dx * 2.0) * (dy > 0.0 ? 0.0 : -dy * 2.0),
      },
      {
        (dx > 0.0 ? 0.0 : -dx * 2.0) * (1.0 - fabs(dy * 2.0)),
        (1.0 - fabs(dx * 2.0)) * (1.0 - fabs(dy * 2.0)),
        (dx < 0.0 ? 0.0 : dx * 2.0) * (1.0 - fabs(dy * 2.0)),
      },
      {
        (dx > 0.0 ? 0.0 : -dx * 2.0) * (dy < 0.0 ? 0.0 : dy * 2.0),
        (1.0 - fabs(dx * 2.0)) * (dy < 0.0 ? 0.0 : dy * 2.0),
        (dx < 0.0 ? 0.0 : dx * 2.0) * (dy < 0.0 ? 0.0 : dy * 2.0),
      },
    };

    // Loop on each pixel
    loop(i, 3) loop(j, 3) {

      // If the pixel as a strictly positive weight and is in the buffer
      if(
        weights[i][j] > 0.0 &&
        pixels[i][j][0] >= 0 && pixels[i][j][0] < that->width &&
        pixels[i][j][1] >= 0 && pixels[i][j][1] < that->height
      ) {

        // Set the color pixel to a linear combination of the current color
        // and the requested color according to the weight
        CapyX11Dims_t iPixel =
          3 * (pixels[i][j][1] * that->width + pixels[i][j][0]);
        loop(k, 3) {
          CapyX11RGB_t v = (CapyX11RGB_t)(
            weights[i][j] * (double)(rgb->vals[k]) +
            (1.0 - weights[i][j]) * (double)(that->rgb[iPixel + k]));
          that->rgb[iPixel + k] = v;
        }
      }
    }
  }
}

// Fill a rectangle with a color.
// Input:
//   rect: the rectangle
//   rgb: the color to be set
// Output:
//   Set the pixel's rgb values inside the rectangle (clipped to the buffer
//   dimensions).
static void DrawFilledRectangle(
  CapyRectangle const* const rect,
     CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  CapyX11Dims_t rangeX[2];
  CapyX11Dims_t rangeY[2];
  rangeX[0] = (CapyX11Dims_t)(rect->corners[0].x);
  rangeX[1] = (CapyX11Dims_t)(rect->corners[1].x);
  if(rangeX[1] >= that->width) rangeX[1] -= 1;
  rangeY[0] = (CapyX11Dims_t)(rect->corners[0].y);
  rangeY[1] = (CapyX11Dims_t)(rect->corners[1].y);
  if(rangeY[1] >= that->height) rangeY[1] -= 1;
  for(CapyX11Dims_t x = rangeX[0]; x <= rangeX[1]; ++x) {
    for(CapyX11Dims_t y = rangeY[0]; y <= rangeY[1]; ++y) {
      loop(i, 3) that->rgb[(y * that->width + x) * 3 + i] = rgb->vals[i];
    }
  }
}

// Clear the buffer.
// Input:
//   rgb: the color to used to clear the buffer
// Output:
//   The whole buffer is set to the given color.
static void Clear(CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  for(CapyX11Dims_t x = 0; x < that->width; ++x) {
    for(CapyX11Dims_t y = 0; y < that->height; ++y) {
      loop(i, 3) that->rgb[(y * that->width + x) * 3 + i] = rgb->vals[i];
    }
  }
}

// Reset the buffer.
// Input:
//   flagColor: if true the buffer is reset to white, else it is reset to
//              black
// Output:
//   The whole buffer is reset.
static void Reset(bool const flagColor) {
  methodOf(CapyX11DisplayBuffer);
  memset(
    that->rgb, flagColor ? 255 : 0, (size_t)(that->width * that->height * 3));
}

// Draw a line (Bresenham algorithm).
// Input:
//   seg: the segment to draw
//   rgb: the color of the line
// Output:
//   Draw a one pixel line (clipped to the buffer dimensions).
static void DrawLine(
  CapySegment const* const seg,
   CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  CapyX11Dims_t x0 = (CapyX11Dims_t)(seg->points[0].x);
  CapyX11Dims_t y0 = (CapyX11Dims_t)(seg->points[0].y);
  CapyX11Dims_t x1 = (CapyX11Dims_t)(seg->points[1].x);
  CapyX11Dims_t y1 = (CapyX11Dims_t)(seg->points[1].y);
  CapyX11Dims_t dx = abs(x1 - x0);
  CapyX11Dims_t sx = x0 < x1 ? 1 : -1;
  CapyX11Dims_t dy = -abs(y1 - y0);
  CapyX11Dims_t sy = y0 < y1 ? 1 : -1;
  CapyX11Dims_t error = dx + dy;
  while(1) {
    if(
      x0 >= 0 && x0 < that->width &&
      y0 >= 0 && y0 < that->height
    ) {
      $(that, setPixel)(&(CapyX11DisplayPos){{x0, y0}}, rgb);
    }
    if(x0 == x1 && y0 == y1) break;
    CapyX11Dims_t e2 = 2 * error;
    if(e2 >= dy) {
      if (x0 == x1) break;
      error += dy;
      x0 += sx;
    }
    if(e2 <= dx) {
      if (y0 == y1) break;
      error += dx;
      y0 += sy;
    }
  }
}

// Draw a triangle.
// Input:
//   tri: the triangle to draw
//   rgb: the color of the line
// Output:
//   Draw a one pixel line triangle (clipped to the buffer dimensions).
static void DrawTriangle(
  CapyTriangle const* const tri,
    CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  CapySegment seg = CapySegmentCreate();
  seg.points[0] = tri->corners[0];
  seg.points[1] = tri->corners[1];
  $(that, drawLine)(&seg, rgb);
  seg.points[0] = tri->corners[1];
  seg.points[1] = tri->corners[2];
  $(that, drawLine)(&seg, rgb);
  seg.points[0] = tri->corners[2];
  seg.points[1] = tri->corners[0];
  $(that, drawLine)(&seg, rgb);
  $(&seg, destruct)();
}

// Draw a line with a gradient color.
// Input:
//   seg: the segment to draw
//   rgbFrom: color at teh start of the segment
//   rgbTo: color at the end of the sgment
// Output:
//   Draw a one pixel line (clipped to the buffer dimensions).
static void DrawLineGradientColor(
  CapySegment const* const seg,
   CapyX11RGB const* const rgbFrom,
   CapyX11RGB const* const rgbTo) {
  methodOf(CapyX11DisplayBuffer);
  CapyX11Dims_t x0 = (CapyX11Dims_t)(seg->points[0].x);
  CapyX11Dims_t y0 = (CapyX11Dims_t)(seg->points[0].y);
  CapyX11Dims_t x1 = (CapyX11Dims_t)(seg->points[1].x);
  CapyX11Dims_t y1 = (CapyX11Dims_t)(seg->points[1].y);
  CapyX11Dims_t dx = abs(x1 - x0);
  CapyX11Dims_t sx = x0 < x1 ? 1 : -1;
  CapyX11Dims_t dy = -abs(y1 - y0);
  CapyX11Dims_t sy = y0 < y1 ? 1 : -1;
  CapyX11Dims_t error = dx + dy;
  if(dx == 0 && dy == 0) return;
  double z0 = 0.0;
  double idz = 0.0;
  CapyX11Dims_t* z = NULL;
  if(dx > -dy) {
    z0 = x0;
    idz = 1.0 / (x1 - x0);
    z = &x0;
  } else {
    z0 = y0;
    idz = 1.0 / (y1 - y0);
    z = &y0;
  }
  while(1) {
    if(
      x0 >= 0 && x0 < that->width &&
      y0 >= 0 && y0 < that->height
    ) {
      double const t = ((double)(*z) - z0) * idz;
      CapyX11RGB rgb;
      loop(i, 3) {
        double v =
          (1.0 - t) * (double)(rgbFrom->vals[i]) +
          t * (double)(rgbTo->vals[i]);
        rgb.vals[i] = (CapyX11RGB_t)v;
      }
      $(that, setPixel)(&(CapyX11DisplayPos){{x0, y0}}, &rgb);
    }
    if(x0 == x1 && y0 == y1) break;
    CapyX11Dims_t e2 = 2 * error;
    if(e2 >= dy) {
      if (x0 == x1) break;
      error += dy;
      x0 += sx;
    }
    if(e2 <= dx) {
      if (y0 == y1) break;
      error += dx;
      y0 += sy;
    }
  }
}

// Draw a triangle filled with a gradient color.
// Input:
//   tri: the triangle to draw
//   rgbA: the color of the first corner
//   rgbB: the color of the second corner
//   rgbC: the color of the third corner
// Output:
//   Draw a filled triangle (clipped to the buffer dimensions).
static void DrawTriangleGradientColor(
  CapyTriangle const* const tri,
    CapyX11RGB const* const rgbA,
    CapyX11RGB const* const rgbB,
    CapyX11RGB const* const rgbC) {
  methodOf(CapyX11DisplayBuffer);
  double xmin = tri->corners[2].x;
  double xmax = xmin;
  double ymin = tri->corners[2].y;
  double ymax = ymin;
  loop(i, 2) {
    if(xmin > tri->corners[i].x) xmin = tri->corners[i].x;
    if(xmax < tri->corners[i].x) xmax = tri->corners[i].x;
    if(ymin > tri->corners[i].y) ymin = tri->corners[i].y;
    if(ymax < tri->corners[i].y) ymax = tri->corners[i].y;
  }
  CapyX11Dims_t xpmin = (CapyX11Dims_t)floor(xmin);
  CapyX11Dims_t xpmax = (CapyX11Dims_t)ceil(xmax);
  CapyX11Dims_t ypmin = (CapyX11Dims_t)floor(ymin);
  CapyX11Dims_t ypmax = (CapyX11Dims_t)ceil(ymax);
  if(xpmin < 0) xpmin = 0;
  if(xpmax >= that->width) xpmax = that->width - 1;
  if(ypmin < 0) ypmin = 0;
  if(ypmax >= that->height) ypmax = that->height - 1;
  for(CapyX11Dims_t x = xpmin; x <= xpmax; x += 1) {
    for(CapyX11Dims_t y = ypmin; y <= ypmax; y += 1) {
      CapyPoint2D pos = { .x = x, .y = y };
      double coords[3];
      $(tri, getBarycentricCoord)(&pos, coords);
      if(coords[0] >= 0.0 && coords[1] >= 0.0 && coords[2] >= 0.0) {
        CapyX11RGB rgb;
        loop(i, 3) {
          double v =
            coords[0] * (double)(rgbA->vals[i]) +
            coords[1] * (double)(rgbB->vals[i]) +
            coords[2] * (double)(rgbC->vals[i]);
          rgb.vals[i] = (CapyX11RGB_t)v;
        }
        $(that, setPixel)(&(CapyX11DisplayPos){{x, y}}, &rgb);
      }
    }
  }
}

// Draw a bezier.
// Input:
//   bezier: the bezier
//   rgb: the color of the line
// Output:
//   Draw a one pixel bezier line (clipped to the buffer dimensions).
static void DrawBezier(
  CapyBezier const* const bezier,
  CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  CapyBezierIterator iter = CapyBezierIteratorCreate(bezier);
  forEach(pos, iter) {
    if(
      pos.out[0] >= 0 && pos.out[0] < that->width &&
      pos.out[1] >= 0 && pos.out[1] < that->height
    ) {
      $(that, setAntialiasedPixel)(
        &(CapyPoint2D){{{pos.out[0], pos.out[1]}}}, rgb);
    }
  }
  $(&iter, destruct)();
}

// Draw a circle (Midpoint circle algorithm).
// Input:
//   circle: the circle
//   rgb: the color of the line
// Output:
//   Draw a one pixel circle line (clipped to the buffer dimensions).
static void DrawCircle(
  CapyCircle const* const circle,
  CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  CapyX11Dims_t f = (CapyX11Dims_t)round(1.0 - circle->radius);
  CapyX11Dims_t ddF_x = 0;
  CapyX11Dims_t ddF_y = (CapyX11Dims_t)round(-2.0 * circle->radius);
  CapyX11Dims_t x = 0;
  CapyX11Dims_t y = (CapyX11Dims_t)round(circle->radius);
  $(that, setPixel)(
    &(CapyX11DisplayPos){
      {
        (CapyX11Dims_t)(circle->center.x),
        (CapyX11Dims_t)(circle->center.y + circle->radius)
      }
    }, rgb);
  $(that, setPixel)(
    &(CapyX11DisplayPos){
      {
        (CapyX11Dims_t)(circle->center.x),
        (CapyX11Dims_t)(circle->center.y - circle->radius)
      }
    }, rgb);
  $(that, setPixel)(
    &(CapyX11DisplayPos){
      {
        (CapyX11Dims_t)(circle->center.x + circle->radius),
        (CapyX11Dims_t)(circle->center.y)
      }
    }, rgb);
  $(that, setPixel)(
    &(CapyX11DisplayPos){
      {
        (CapyX11Dims_t)(circle->center.x - circle->radius),
        (CapyX11Dims_t)(circle->center.y)
      }
    }, rgb);
  while(x < y) {
    if(f >= 0) {
      y--;
      ddF_y += 2;
      f += ddF_y;
    }
    x++;
    ddF_x += 2;
    f += ddF_x + 1;
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x + x),
          (CapyX11Dims_t)(circle->center.y + y)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x - x),
          (CapyX11Dims_t)(circle->center.y + y)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x + x),
          (CapyX11Dims_t)(circle->center.y - y)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x - x),
          (CapyX11Dims_t)(circle->center.y - y)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x + y),
          (CapyX11Dims_t)(circle->center.y + x)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x - y),
          (CapyX11Dims_t)(circle->center.y + x)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x + y),
          (CapyX11Dims_t)(circle->center.y - x)
        }
      },
      rgb);
    $(that, setPixel)(
      &(CapyX11DisplayPos){
        {
          (CapyX11Dims_t)(circle->center.x - y),
          (CapyX11Dims_t)(circle->center.y - x)
        }
      },
      rgb);
  }
}

// Draw a filled circle (Midpoint circle algorithm).
// Input:
//   circle: the circle
//   rgb: the color of the circle
// Output:
//   Draw a filled circle (clipped to the buffer dimensions).
static void DrawFilledCircle(
  CapyCircle const* const circle,
  CapyX11RGB const* const rgb) {
  methodOf(CapyX11DisplayBuffer);
  CapyX11Dims_t f = (CapyX11Dims_t)round(1.0 - circle->radius);
  CapyX11Dims_t ddF_x = 0;
  CapyX11Dims_t ddF_y = (CapyX11Dims_t)round(-2.0 * circle->radius);
  CapyX11Dims_t x = 0;
  CapyX11Dims_t y = (CapyX11Dims_t)round(circle->radius);
  CapySegment seg = {
    .points = {
      {.x = circle->center.x + circle->radius, .y = circle->center.y},
      {.x = circle->center.x - circle->radius, .y = circle->center.y}
    }
  };
  $(that, drawLine)(&seg, rgb);
  while(x < y) {
    if(f >= 0) {
      y--;
      ddF_y += 2;
      f += ddF_y;
    }
    x++;
    ddF_x += 2;
    f += ddF_x + 1;
    CapySegment seg1 = {
      .points = {
        {.x = circle->center.x + x, .y = circle->center.y + y},
        {.x = circle->center.x - x, .y = circle->center.y + y}
      }
    };
    $(that, drawLine)(&seg1, rgb);
    CapySegment seg2 = {
      .points = {
        {.x = circle->center.x + y, .y = circle->center.y + x},
        {.x = circle->center.x - y, .y = circle->center.y + x}
      }
    };
    $(that, drawLine)(&seg2, rgb);
    CapySegment seg3 = {
      .points = {
        {.x = circle->center.x + x, .y = circle->center.y - y},
        {.x = circle->center.x - x, .y = circle->center.y - y}
      }
    };
    $(that, drawLine)(&seg3, rgb);
    CapySegment seg4 = {
      .points = {
        {.x = circle->center.x + y, .y = circle->center.y - x},
        {.x = circle->center.x - y, .y = circle->center.y - x}
      }
    };
    $(that, drawLine)(&seg4, rgb);
  }
}

// Convert the buffer into an image
// Output:
//   Return a CapyImg.
static CapyImg* ToImg(void) {
  methodOf(CapyX11DisplayBuffer);

  // Create the image with the current output buffer dimensions
  CapyImgDims dims = {
    .width = (CapyImgDims_t)(that->width),
    .height = (CapyImgDims_t)(that->height)
  };
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, dims);

  // Copy the output buffer in the image
  loop(y, dims.height) loop(x, dims.width) {
    CapyImgPos pos = {.x = (CapyImgPos_t)x, .y = (CapyImgPos_t)y};
    CapyColorData color;
    loop(i, (CapyImgDims_t)3) {
      color.rgb[i] =
        (double)(that->rgb[(y * dims.width + x) * 3 + i]) / 255.0;
    }
    $(img, setColor)(&pos, &color);
  }

  // Return the image
  return img;
}

// Allocate memory and create a new CapyX11Display instance
// Input:
//         width: initial width of the display
//        height: initial height of the display
//   displayName: the display name as hostname:number.screen_number, if NULL
//                ":0.0" is used instead
//         title: the title of the window
// Output:
//   Return a new CapyX11Display instance, or NULL if it couldn't be created
CapyX11Display* CapyX11DisplayAlloc(
  CapyX11Dims_t const width,
  CapyX11Dims_t const height,
    char const* const displayName,
    char const* const title) {

  // Allocate memory for the instance
  CapyX11Display* that = NULL;
  safeMalloc(that, 1);
  if(that == NULL) return NULL;
  memset(that, 0, sizeof(CapyX11Display));

  // Initialise dimensions
  that->width = width;
  that->height = height;

  // Set the display name
  char const* name = defaultDisplayName;
  if(displayName != NULL) name = displayName;
  that->displayName = strCreate("%s", name);

  // Initialise the buffers
  size_t sizeBuffer = (size_t)(width * height * 3) * sizeof(CapyX11RGB_t);
  loop(i, 3) {
    *((CapyX11Dims_t*)&(that->buffers[i].width)) = width;
    *((CapyX11Dims_t*)&(that->buffers[i].height)) = height;
    *((CapyX11RGB_t**)&(that->buffers[i].rgb)) = malloc(sizeBuffer);
    if(that->buffers[i].rgb == NULL) raiseExc(CapyExc_MallocFailed);
    else memset(that->buffers[i].rgb, 0, sizeBuffer);
    that->buffers[i].setPixel = SetPixel;
    that->buffers[i].setAntialiasedPixel = SetAntialiasedPixel;
    that->buffers[i].drawFilledRectangle = DrawFilledRectangle;
    that->buffers[i].clear = Clear;
    that->buffers[i].reset = Reset;
    that->buffers[i].drawLine = DrawLine;
    that->buffers[i].drawTriangle = DrawTriangle;
    that->buffers[i].drawLineGradientColor = DrawLineGradientColor;
    that->buffers[i].drawTriangleGradientColor = DrawTriangleGradientColor;
    that->buffers[i].drawCircle = DrawCircle;
    that->buffers[i].drawFilledCircle = DrawFilledCircle;
    that->buffers[i].drawBezier = DrawBezier;
    that->buffers[i].toImg = ToImg;
  }
  that->inputBuffer = 2;
  that->outputBuffer = 0;

  // Open the connection to the server
  that->display = XOpenDisplay(that->displayName);
  if(that->display == NULL) {
    CapyX11DisplayFree(&that);
    return NULL;
  }

  // Get the parent window of the window we are going to create, we use the
  // the root of the display
  that->parentWindow = XDefaultRootWindow(that->display);

  // Set the graphical context to the default one
  that->gc = XDefaultGC(that->display, 0);

  // Prepare the XImage wrapper for the rgba array
  // ZPixmap: interlaced RGBA channels, XYPixmap: RGBA channels in
  // sequential order
  // bitmap_pad: equal to the number of bit per pixel given there is no
  // offset on scanlines (xoffset = 0)
  // depth: number of bit per pixel
  that->image = (XImage){
    .xoffset = 0,
    .format = ZPixmap,
    .byte_order = MSBFirst,
    .bitmap_pad = 32,
    .depth = 24,
    .bits_per_pixel = 24,
    .red_mask = 0xff0000,
    .green_mask = 0x00ff00,
    .blue_mask = 0x0000ff,
    .obdata = NULL,
    .f = {0},
  };
  XInitImage(&(that->image));

  // Create the window
  that->window = XCreateSimpleWindow(
    that->display, that->parentWindow, 0, 0,
    (unsigned int)(that->width), (unsigned int)(that->height),
    0, XBlackPixel(that->display, 0), XBlackPixel(that->display, 0));

  // Set the window's title
  CapyX11DisplaySetTitle(that, title);

  // Create the event mask for the event types we want the window to respond to
  that->evtMask =
    ExposureMask | StructureNotifyMask |
    KeyPressMask | KeyReleaseMask |
    ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
  XSelectInput(that->display, that->window, that->evtMask);

  // Avoid the window to be closed by the user pressing the X button
  // When this happens it is said that a ClientMessage event whould be sent
  // and could be processed as follow
  // switch(event.type) {
  //   case ClientMessage:
  //     if ((Atom)(event.xclient.data.l[0]) == wmDelete)
  // however I don't receive that event and can't understand why, so I give up
  that->wmDelete = XInternAtom(that->display, "WM_DELETE_WINDOW", false);
  XSetWMProtocols(that->display, that->window, &(that->wmDelete), 1);

  // Return the new instance
  return that;
}

// Free the resource used by a CapyX11Display instance
// Input:
//   that: pointer to the CapyX11Display instance
// Output:
//   *that is set to NULL, the display is closed if it has been shown, all
//   memory allocated by the instance is released
void CapyX11DisplayFree(CapyX11Display** const that) {
  if(that == NULL || *that == NULL) return;
  CapyX11DisplayHide(*that);
  if((*that)->display != NULL) XCloseDisplay((*that)->display);
  if((*that)->displayName != NULL) free((*that)->displayName);
  loop(i, 3) free((*that)->buffers[i].rgb);
  *that = NULL;
}

// Repaint the window as necessary
// Input:
//   that: the CapyX11Display instance
static void RepaintWindow(CapyX11Display* const that) {

  // If a new buffer has been published
  int idxBuffer = (that->outputBuffer + 1) % 3;
  if(idxBuffer != that->inputBuffer) {

    // Update the output buffer index
    that->outputBuffer = idxBuffer;

    // Send the new buffer content to the server
    // Here, the buffer dimensions may be different from the current window
    // dimensions. This is fine, XPutImage take care of that for us.
    CapyX11DisplayBuffer* buffer = that->buffers + that->outputBuffer;
    that->image.width = buffer->width;
    that->image.height = buffer->height;
    that->image.data = (char*)(buffer->rgb);
    that->image.bytes_per_line = 3 * buffer->width;
    XPutImage(
      that->display, that->window, that->gc, &(that->image),
      0, 0, 0, 0,
      (unsigned int)(buffer->width), (unsigned int)(buffer->height));

    // Update the refresh rate
    ++(that->rates[refresh].count);
  }
}

// Queue a new event if the queue is not full
// Input:
//   that: pointer to the CapyX11Display instance
//    evt: the event to queue
static void AddEventToQueue(
    CapyX11Display* const that,
  CapyX11DisplayEvt const evt) {

  // If the buffer is not full
  size_t nextIdx = (that->evtHead + 1) % EVT_BUFFER_SIZE;
  if(nextIdx != that->evtTail) {

    // Add the event in the buffer and update the head
    that->evts[that->evtHead] = evt;
    that->evtHead = nextIdx;
  }
}

// Process a key event
static void ProcessKeyEvent(
  CapyX11Display* const that,
       XKeyEvent* const keyEvent) {

  // If the key event is 'ctrl-c' (the key combination, not the signal)
  // hide the window, else add the event to the queue
  unsigned long keysym = XLookupKeysym(keyEvent, 0);
  if(keyEvent->state == ControlMask && keysym == XK_c) {
    that->flagTerminateThread = true;
  } else {
    CapyX11DisplayEvt evt = {
      .type = (
        that->event.type == KeyPress ? capyX11Evt_press : capyX11Evt_release),
      .src = capyX11Evt_key,
      .time = (CapyChronoTime_t)(keyEvent->time),
      .state = keyEvent->state,
      .val = keysym,
      .pos = (CapyX11DisplayPos){.x = keyEvent->x, .y = keyEvent->y},
    };
    AddEventToQueue(that, evt);
  }
}

// Process a mouse button event
static void ProcessButtonEvent(
      CapyX11Display* const that,
  XButtonEvent const* const btnEvent) {

  // Add the event to the queue
  CapyX11DisplayEvt evt = {
    .type = (
      btnEvent->type == ButtonPress ? capyX11Evt_press : capyX11Evt_release),
    .src = capyX11Evt_mouse,
    .time = (CapyChronoTime_t)(btnEvent->time),
    .state = btnEvent->state,
    .val = btnEvent->button,
    .pos = (CapyX11DisplayPos){.x = btnEvent->x, .y = btnEvent->y},
  };
  AddEventToQueue(that, evt);
}

// Process an event from the X11 server.
// Input:
//   that: pointer to the CapyX11Display instance
static void ProcessEvent(CapyX11Display* const that) {
  switch(that->event.type) {

    // Expose event
    case Expose:
      RepaintWindow(that);
      break;

    // Pointer motion event
    case MotionNotify:
      that->pointerPos.x = ((XMotionEvent*)&(that->event))->x;
      that->pointerPos.y = ((XMotionEvent*)&(that->event))->y;
      break;

    // Window structure change event
    case ConfigureNotify:
      that->width = ((XConfigureEvent*)&(that->event))->width;
      that->height = ((XConfigureEvent*)&(that->event))->height;
      break;

    // Key event
    case KeyPress:
      ProcessKeyEvent(that, (XKeyEvent*)&(that->event));
      break;
    case KeyRelease:
      ProcessKeyEvent(that, (XKeyEvent*)&(that->event));
      break;

    // Button event
    case ButtonPress:
      ProcessButtonEvent(that, (XButtonEvent*)&(that->event));
      break;
    case ButtonRelease:
      ProcessButtonEvent(that, (XButtonEvent*)&(that->event));
      break;

    // Other events are simply ignored
    default:
      break;
  }
}

// Main loop executed in its own thread. Update the content of the display,
// and process the events from the X11 server
// Input:
//   arg: to be cast to a CapyX11Display*
// Output:
//   Always return NULL.
static void* Main(void* arg) {

  // Cast the argument
  CapyX11Display* that = (CapyX11Display*)arg;

  // Reset the rates
  loop(i, 2) {
    that->rates[i].count = 0;
    clock_gettime(CLOCK_MONOTONIC, &(that->rates[i].start));
  }

  // Reset the flag to control the main loop
  that->flagTerminateThread = false;

  // Main loop
  while(that->flagTerminateThread == false) {

    // Ask the server if there is an event waiting in the queue, if so process
    // it
    bool waitingEvent =
      XCheckWindowEvent(
        that->display, that->window, that->evtMask, &(that->event));
    if(waitingEvent) ProcessEvent(that);

    // Repaint the window as necessary
    RepaintWindow(that);
  }

  // Unmap the display
  XUnmapWindow(that->display, that->window);

  // Flush the remaining events
  CapyX11DisplayFlushEvent(that);

  // Set the flag
  that->isChildRunning = false;

  // No return value at the end of the thread
  return NULL;
}

// Display the CapyX11Display instance and starts its main loop in a separate
// thread. The thread will end when the window has focus and the user presses
// ctrl-c (the key combination, not the signal), or CapyX11DisplayHide() is called,
// or CapyX11DisplayFree() is called.
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayShow(CapyX11Display* const that) {

  // If the display is not already displayed
  if(that->isChildRunning == false) {

    // Map the window to the display
    XMapWindow(that->display, that->window);

    // Start the main loop in its own thread
    that->thread = (pthread_t){0};
    pthread_create(&(that->thread), &(that->threadAttr), Main, that);
    that->isChildRunning = true;

    // Wait for the thread to start
    while(!CapyX11DisplayIsShown(that));

    // Start the timer
    clock_gettime(CLOCK_MONOTONIC, &(that->timer));
  }
}

// Hide the CapyX11Display instance. The display is unmapped and its thread is
// terminated but its resources are not freed and it is ready to be shown again
// anytime.
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayHide(CapyX11Display* const that) {

  // If the child thread is running
  if(that->isChildRunning) {

    // Set the flag to terminate the main loop
    that->flagTerminateThread = true;

    // Join the thread
    pthread_join(that->thread, NULL);
    that->thread = (pthread_t){0};
  }
}

// Check if a display is currently shown
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return true if the display is shown, else false
bool CapyX11DisplayIsShown(CapyX11Display const* const that) {
  return that->isChildRunning;
}

// Get the average rate (code commonalisation for CapyX11DisplayGetFrameRate and
// CapyX11DisplayGetRefreshRate)
// Input:
//   that: the CapyX11Display instance
//   type: type of rate measured (frame or refresh)
// Output:
//   Return the rate
static double GetRate(
  CapyX11Display* const that,
          RateIdx const type) {

  // Measure the rate
  clock_gettime(CLOCK_MONOTONIC, &(that->rates[type].stop));
  double count = (double)(that->rates[type].count);
  double timeMs =
    (double)(
      (
        that->rates[type].stop.tv_sec * 1000 +
        that->rates[type].stop.tv_nsec / 1000000
      ) - (
        that->rates[type].start.tv_sec * 1000 +
        that->rates[type].start.tv_nsec / 1000000
      )
    );
  double rate = count / timeMs * 1000.0;

  // Reset the timer and counter
  clock_gettime(CLOCK_MONOTONIC, &(that->rates[type].start));
  that->rates[type].count = 0;

  // Return the rate
  return rate;
}

// Get the average refresh rate (number of times the window content is updated
// per second) since the last call to CapyX11DisplayGetRefreshRate() or
// CapyX11DisplayShow()
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the refresh rate
double CapyX11DisplayGetRefreshRate(CapyX11Display* const that) {
  return GetRate(that, refresh);
}

// Get the average frame rate (number of times the image buffer is updated
// per second) since the last call to CapyX11DisplayGetFrameRate() or
// CapyX11DisplayShow()
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the frame rate
double CapyX11DisplayGetFrameRate(CapyX11Display* const that) {
  return GetRate(that, frame);
}

// Return the next event in the queue of events
// Input:
//   that: the CapyX11Display instance
// Output:
//   Pop the next event from the queue and return it. If there was no event in
//   the queue an event with type == 0 is returned instead.
CapyX11DisplayEvt CapyX11DisplayGetNextEvent(CapyX11Display* const that) {

  // If there are events waiting in the buffer
  if (that->evtHead != that->evtTail) {

    // Update the head and tail and return the event
    CapyX11DisplayEvt evt = that->evts[that->evtTail];
    that->evtTail = (that->evtTail + 1) % EVT_BUFFER_SIZE;
    return evt;

  // Else, there is no event waiting in the buffer
  } else {

    // Return a "no event"
    return (CapyX11DisplayEvt){0};
  }
}

// Flush the event queue
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayFlushEvent(CapyX11Display* const that) {
  that->evtHead = that->evtTail;
}

// Get the las known position of the pointer in the display
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the last known position.
CapyX11DisplayPos CapyX11DisplayGetPointerPos(
  CapyX11Display const* const that) {
  return that->pointerPos;
}

// Get the current rendering buffer.
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the buffer to write on. The content of the buffer is undefined.
//   The width and height of the buffer may be different from the previous
//   buffer. 
CapyX11DisplayBuffer CapyX11DisplayGetBuffer(CapyX11Display* const that) {

  // Ensure the buffer's dimensions match the current window dimensions
  CapyX11DisplayBuffer* buffer = that->buffers + that->inputBuffer;
  CapyX11Dims_t width = that->width;
  CapyX11Dims_t height = that->height;
  if(width != buffer->width || height != buffer->height) {
    unsigned char* rgb = *((unsigned char**)&(buffer->rgb));
    safeRealloc(rgb, (size_t)(width * height * 3));
    if(rgb != NULL) memset(rgb, 0, (size_t)(width * height * 3));
    *((unsigned char**)&(buffer->rgb)) = rgb;
    *((CapyX11Dims_t*)&(buffer->width)) = width;
    *((CapyX11Dims_t*)&(buffer->height)) = height;
  }

  // Return the buffer
  return *buffer;
}

// Publish the current editing buffer
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayPublish(CapyX11Display* const that) {

  // If the third buffer is available
  int idxBuffer = (that->inputBuffer + 1) % 3;
  if(idxBuffer != that->outputBuffer) {

    // Update the input buffer index
    that->inputBuffer = idxBuffer;
  }

  // Update the frame rate
  ++(that->rates[frame].count);
}

// Get the elapsed time in millisecond since the display is shown
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the elapsed time.
CapyChronoTime_t CapyX11DisplayGetTimeMs(CapyX11Display const* const that) {

  // Measure and return the elapsed time
  struct timespec now;
  clock_gettime(CLOCK_MONOTONIC, &now);
  CapyChronoTime_t timeMs =
    (now.tv_sec * 1000 + now.tv_nsec / 1000000) -
    (that->timer.tv_sec * 1000 + that->timer.tv_nsec / 1000000);
  return timeMs;
}

// Take a screen shot of the display
// Input:
//   that: pointer to the CapyX11Display instance
// Output:
//   Return a CapyImg.
CapyImg* CapyX11DisplayScreenshot(CapyX11Display const* const that) {
  CapyX11DisplayBuffer const* buffer = that->buffers + that->outputBuffer;
  return $(buffer, toImg)();
}
