// -------------------------------- x11display.c --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_X11_DISPLAY
#define CAPY_X11_DISPLAY
#include "externalHeaders.h"
#include "cext.h"
#include "geometricShape.h"
#include "chrono.h"
#include "image.h"

// Description:
// Class to display graphics on screen, using X11.

// X11Display opaque structure
typedef struct CapyX11Display CapyX11Display;

// Types of events
typedef enum CapyX11Evt {
  capyX11Evt_noEvent,
  capyX11Evt_press,
  capyX11Evt_release,
} CapyX11Evt;

// Types of source of events
typedef enum CapyX11Src {
  capyX11Evt_key,
  capyX11Evt_mouse,
} CapyX11Src;

// Type for the dimensions of a CapyX11Display
typedef CapyImgPos_t CapyX11Dims_t;

// CapyX11DisplayPos structure returned by CapyX11DisplayGetPointerPos()
typedef CapyImgPos CapyX11DisplayPos;

// X11DisplayEvt structure returned by CapyX11DisplayGetNextEvent()
typedef struct CapyX11DisplayEvt {

  // 0: no event, 1: press, 2: release
  CapyX11Evt type;

  // 0: key, 1: mouse
  CapyX11Src src;

  // Time of the event in millisecond (comes from XLib, not sure relative
  // to what)
  CapyChronoTime_t time;

  // key or button mask, 'or' combination of:
  // NoModMask, Button1Mask, Button2Mask, Button3Mask, Button4Mask,
  // Button5Mask, ShiftMask, LockMask, ControlMask, Mod1Mask, Mod2Mask,
  // Mod3Mask, Mod4Mask, and Mod5Mask
  size_t state;

  // key/button value, one of:
  // for key: NoSymbol or the XK_... macro (see complete list here:
  // /usr/include/X11/keysymdef.h) which map to ASCII (XK_a == 'a')
  // for button: Button1, Button2, Button3, Button4, Button5 
  size_t val;

  // Coordinates of the pointer at the time of event
  CapyX11DisplayPos pos;
} CapyX11DisplayEvt;

// Type of one channel in pixel's RGB data
typedef uint8_t CapyX11RGB_t;

// Type of pixel's RGB data for a CapyX11Display
typedef struct CapyX11RGB {

  // Position (0,0 is the top-left corner, x toward right, y toward bottom)
  union {
    CapyX11RGB_t vals[3];
    struct __attribute__((packed)) { CapyX11RGB_t r, g, b;};
  };
} CapyX11RGB;

// CapyX11DisplayBuffer structure returned by CapyX11DisplayBufferPublish()
typedef struct {

  // Dimensions of the buffer
  CapyX11Dims_t const width;
  CapyX11Dims_t const height;

  // RGB buffers, three channels, ordered by row of RGB pixel value in [0, 255]
  CapyX11RGB_t* const rgb;

  // Set a pixel color in the buffer.
  // Input:
  //   pos: the pixel coordinates (0,0 at top left corner, y downward)
  //   rgb: the color to be set
  // Output:
  //   Set the pixel's rgb values. Do nothing if the coordinates are out of
  //   the buffer.
  void (*setPixel)(
    CapyX11DisplayPos const* const pos,
           CapyX11RGB const* const rgb);

  // Set an antialiased pixel color in the buffer.
  // Input:
  //   pos: the pixel coordinates (0,0 at top left corner, y downward)
  //   rgb: the color to be set
  // Output:
  //   Set the pixel's rgb values. Do nothing if the coordinates are out of
  //   the buffer. The antialiasing is done using the Xiaolin Wu's method
  void (*setAntialiasedPixel)(
    CapyPoint2D const* const pos,
     CapyX11RGB const* const rgb);

  // Fill a rectangle with a color.
  // Input:
  //   rect: the rectangle
  //   rgb: the color to be set
  // Output:
  //   Set the pixel's rgb values inside the rectangle (clipped to the buffer
  //   dimensions).
  void (*drawFilledRectangle)(
    CapyRectangle const* const rect,
       CapyX11RGB const* const rgb);

  // Clear the buffer.
  // Input:
  //   rgb: the color to used to clear the buffer
  // Output:
  //   The whole buffer is set to the given color.
  void (*clear)(CapyX11RGB const* const rgb);

  // Reset the buffer.
  // Input:
  //   flagColor: if true the buffer is reset to white, else it is reset to
  //              black
  // Output:
  //   The whole buffer is reset.
  void (*reset)(bool const flagColor);

  // Draw a line (Bresenham algorithm).
  // Input:
  //   seg: the segment to draw
  //   rgb: the color of the line
  // Output:
  //   Draw a one pixel line (clipped to the buffer dimensions).
  void (*drawLine)(
    CapySegment const* const seg,
     CapyX11RGB const* const rgb);

  // Draw a triangle.
  // Input:
  //   tri: the triangle to draw
  //   rgb: the color of the line
  // Output:
  //   Draw a one pixel line triangle (clipped to the buffer dimensions).
  void (*drawTriangle)(
    CapyTriangle const* const tri,
      CapyX11RGB const* const rgb);

  // Draw a line with a gradient color.
  // Input:
  //   seg: the segment to draw
  //   rgbFrom: color at teh start of the segment
  //   rgbTo: color at the end of the sgment
  // Output:
  //   Draw a one pixel line (clipped to the buffer dimensions).
  void (*drawLineGradientColor)(
    CapySegment const* const seg,
     CapyX11RGB const* const rgbFrom,
     CapyX11RGB const* const rgbTo);

  // Draw a triangle with a gradient color.
  // Input:
  //   tri: the triangle to draw
  //   rgbA: the color of the first corner
  //   rgbB: the color of the second corner
  //   rgbC: the color of the third corner
  // Output:
  //   Draw a one pixel line triangle (clipped to the buffer dimensions).
  void (*drawTriangleGradientColor)(
    CapyTriangle const* const tri,
      CapyX11RGB const* const rgbA,
      CapyX11RGB const* const rgbB,
      CapyX11RGB const* const rgbC);

  // Draw a circle (Midpoint circle algorithm).
  // Input:
  //   circle: the circle
  //   rgb: the color of the line
  // Output:
  //   Draw a one pixel circle line (clipped to the buffer dimensions).
  void (*drawCircle)(
    CapyCircle const* const circle,
    CapyX11RGB const* const rgb);

  // Draw a filled circle (Midpoint circle algorithm).
  // Input:
  //   circle: the circle
  //   rgb: the color of the line
  // Output:
  //   Draw a filled circle (clipped to the buffer dimensions).
  void (*drawFilledCircle)(
    CapyCircle const* const circle,
    CapyX11RGB const* const rgb);

  // Draw a bezier.
  // Input:
  //   bezier: the bezier
  //   rgb: the color of the line
  // Output:
  //   Draw a one pixel bezier line (clipped to the buffer dimensions).
  void (*drawBezier)(
    CapyBezier const* const bezier,
    CapyX11RGB const* const rgb);

  // Convert the buffer into an image
  // Output:
  //   Return a CapyImg.
  CapyImg* (*toImg)(void);
} CapyX11DisplayBuffer;

// Allocate memory and create a new CapyX11Display instance
// Input:
//         width: initial width of the display
//        height: initial height of the display
//   displayName: the display name as hostname:server_number.screen_number, if
//                NULL ":0.0" is used instead
//         title: the title of the window
// Output:
//   Return a new CapyX11Display instance, or NULL if it couldn't be created
CapyX11Display* CapyX11DisplayAlloc(
  CapyX11Dims_t const width,
  CapyX11Dims_t const height,
    char const* const displayName,
    char const* const title);

// Free the resource used by a CapyX11Display instance
// Input:
//   that: pointer to the CapyX11Display instance
// Output:
//   *that is set to NULL, the display is closed if it has been shown, all
//   memory allocated by the instance is released
void CapyX11DisplayFree(CapyX11Display** const that);

// Display the CapyX11Display instance and starts its main loop in a separate
// thread. The thread will end when the window has focus and the user presses
// ctrl-c (the key combination, not the signal), or CapyX11DisplayHide() is called,
// or CapyX11DisplayFree() is called.
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayShow(CapyX11Display* const that);

// Hide the CapyX11Display instance. The display is unmapped and its thread is
// terminated but its reources are not freed and it is ready to be shown again
// anytime.
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayHide(CapyX11Display* const that);

// Check if a display is currently shown
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return true if the display is shown, else false
bool CapyX11DisplayIsShown(CapyX11Display const* const that);

// Get the average refresh rate (number of times the window content is updated
// per second) since the last call to CapyX11DisplayGetRefreshRate() or
// CapyX11DisplayShow().
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the refresh rate
double CapyX11DisplayGetRefreshRate(CapyX11Display* const that);

// Get the average frame rate (number of times the image buffer is updated
// per second)  since the last call to CapyX11DisplayGetFrameRate() or
// CapyX11DisplayShow().
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the frame rate
double CapyX11DisplayGetFrameRate(CapyX11Display* const that);

// Return the next event in the queue of events
// Input:
//   that: the CapyX11Display instance
// Output:
//   Pop the next event from the queue and return it. If there was no event in
//   the queue an event with type == 0 is returned instead.
CapyX11DisplayEvt CapyX11DisplayGetNextEvent(CapyX11Display* const that);

// Flush the event queue
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayFlushEvent(CapyX11Display* const that);

// Get the las known position of the pointer in the display
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the last known position.
CapyX11DisplayPos CapyX11DisplayGetPointerPos(CapyX11Display const* const that);

// Get the current rendering buffer.
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the buffer to write on. The content of the buffer is undefined.
//   The width and height of the buffer may be different from the previous
//   buffer. 
CapyX11DisplayBuffer CapyX11DisplayGetBuffer(CapyX11Display* const that);

// Publish the current editing buffer
// Input:
//   that: the CapyX11Display instance
void CapyX11DisplayPublish(CapyX11Display* const that);

// Get the elapsed time in millisecond since the display is shown
// Input:
//   that: the CapyX11Display instance
// Output:
//   Return the elapsed time.
CapyChronoTime_t CapyX11DisplayGetTimeMs(CapyX11Display const* const that);

// Set the title of the window
// Input:
//    that: pointer to the CapyX11Display instance
//   title: the window's new title
void CapyX11DisplaySetTitle(
  CapyX11Display const* const that,
            char const* const title);

// Take a screen shot of the display
// Input:
//   that: pointer to the CapyX11Display instance
// Output:
//   Return a CapyImg.
CapyImg* CapyX11DisplayScreenshot(CapyX11Display const* const that);
#endif
