#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "Creation/Destruction of CapyArgParser") {
  char* argv_[1];
  argv_[0] = strCreate("cutest");
  CapyArg args_[] = {
    {.lbl = argv_[0], .nbVal = capyArg_noVal, .help = "test"},
  };
  CapyArgParser* parser =
    CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
  CUTEST_ASSERT(parser, "parser unallocated");
  CapyArgParserFree(&parser);
  CUTEST_ASSERT(parser == NULL, "parser not reset to NULL");
  free(argv_[0]);
}

CUTEST(test002, "Missing label and shortcut") {
  CapyArg args_[] = {
    {.nbVal = capyArg_noVal, .help = "test"},
  };
  char* argv_[1];
  argv_[0] = strCreate("cutest");
  CapyArgParser* parser = NULL;
  try {
    parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    CUTEST_ASSERT(false, "missing label and shortcut undetected");
  } catch(CapyExc_InvalidCLIArg) {
    CUTEST_ASSERT(true, "missing label and shortcut detected");
    CapyCancelExc();
  } endCatch;
  CapyForwardExc();
  CapyArgParserFree(&parser);
  free(argv_[0]);
}

CUTEST(test003, "Using --help as label") {
  CapyArg args_[] = {
    {.lbl = "--help", .nbVal = capyArg_noVal, .help = "test"},
  };
  char* argv_[1];
  argv_[0] = strCreate("cutest");
  CapyArgParser* parser = NULL;
  try {
    parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    CUTEST_ASSERT(false, "using --help as label is not allowed");
  } catch(CapyExc_InvalidCLIArg) {
    CUTEST_ASSERT(true, "using --help as label detected");
    CapyCancelExc();
  } endCatch;
  CapyForwardExc();
  CapyArgParserFree(&parser);
  free(argv_[0]);
}

CUTEST(test004, "Using --help as shortcut") {
  CapyArg args_[] = {
    {.shortLbl = "--help", .nbVal = capyArg_noVal, .help = "test"},
  };
  char* argv_[1];
  argv_[0] = strCreate("cutest");
  CapyArgParser* parser = NULL;
  try {
    parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    CUTEST_ASSERT(false, "using --help as shortcut is not allowed");
  } catch(CapyExc_InvalidCLIArg) {
    CUTEST_ASSERT(true, "using --help as shortcut detected");
    CapyCancelExc();
  } endCatch;
  CapyArgParserFree(&parser);
  free(argv_[0]);
}

CUTEST(test005, "Duplicate label") {
  CapyArg args_[] = {
    {.lbl = "--a", .nbVal = capyArg_noVal, .help = "test"},
    {.lbl = "--a", .nbVal = capyArg_noVal, .help = "test"},
  };
  char* argv_[1];
  argv_[0] = strCreate("cutest");
  CapyArgParser* parser = NULL;
  try {
    parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    CUTEST_ASSERT(false, "duplicate label is not allowed");
  } endCatch;
  CapyArgParserFree(&parser);
  free(argv_[0]);
}

CUTEST(test006, "Duplicate shortcut") {
  CapyArg args_[] = {
    {.shortLbl = "--a", .nbVal = capyArg_noVal, .help = "test"},
    {.shortLbl = "--a", .nbVal = capyArg_noVal, .help = "test"},
  };
  char* argv_[1];
  argv_[0] = strCreate("cutest");
  CapyArgParser* parser = NULL;
  try {
    parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    CUTEST_ASSERT(false, "duplicate shortcut is not allowed");
  } endCatch;
  CapyArgParserFree(&parser);
  free(argv_[0]);
}

CUTEST(test007, "Validation of the number of arguments") {
  CapyArg args_[] = {
    {.lbl = "cutest", .nbVal = 2, .help = "test"},
    {.lbl = "--opt", .nbVal = capyArg_noVal, .help = "test"},
  };
  {
    char* argv_[1];
    argv_[0] = strCreate("cutest");
    CapyArgParser* parser = NULL;
    try {
      parser =
        CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
      CUTEST_ASSERT(false, "invalid number of argument is not allowed");
    } endCatch;
    CapyArgParserFree(&parser);
    free(argv_[0]);
  }
  {
    char* argv_[2];
    argv_[0] = strCreate("cutest");
    argv_[1] = strCreate("a");
    CapyArgParser* parser = NULL;
    try {
      parser =
        CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
      CUTEST_ASSERT(false, "invalid number of argument is not allowed");
    } endCatch;
    CapyArgParserFree(&parser);
    free(argv_[0]);
    free(argv_[1]);
  }
  {
    char* argv_[4];
    argv_[0] = strCreate("cutest");
    argv_[1] = strCreate("a");
    argv_[2] = strCreate("b");
    argv_[3] = strCreate("c");
    CapyArgParser* parser = NULL;
    try {
      parser =
        CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
      CUTEST_ASSERT(false, "invalid number of argument is not allowed");
    } endCatch;
    CapyArgParserFree(&parser);
    free(argv_[0]);
    free(argv_[1]);
    free(argv_[2]);
    free(argv_[3]);
  }
  {
    char* argv_[3];
    argv_[0] = strCreate("cutest");
    argv_[1] = strCreate("a");
    argv_[2] = strCreate("b");
    CapyArgParser* parser = NULL;
    try {
      parser =
        CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    } catch(CapyExc_InvalidCLIArg) {
      CUTEST_ASSERT(false, "invalid parsing of the argument values");
    } endCatch;
    CapyArgParserFree(&parser);
    free(argv_[0]);
    free(argv_[1]);
    free(argv_[2]);
  }
  {
    char* argv_[4];
    argv_[0] = strCreate("cutest");
    argv_[1] = strCreate("a");
    argv_[2] = strCreate("--opt");
    argv_[3] = strCreate("b");
    CapyArgParser* parser = NULL;
    try {
      parser =
        CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    } catch(CapyExc_InvalidCLIArg) {
      CUTEST_ASSERT(false, "invalid parsing of the argument values");
    } endCatch;
    CapyArgParserFree(&parser);
    free(argv_[0]);
    free(argv_[1]);
    free(argv_[2]);
    free(argv_[3]);
  }
}

CUTEST(test008, "Parsing int value") {
  CapyArg args_[] = {
    {.shortLbl = "--a", .nbVal = 2, .help = "test"},
  };
  char* argv_[4];
  argv_[0] = strCreate("cutest");
  argv_[1] = strCreate("--a");
  argv_[2] = strCreate("123");
  argv_[3] = strCreate("-234");
  CapyArgParser* parser =
    CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
  int64_t val = $(parser, getAsInt)("--a", 0);
  CUTEST_ASSERT(val == 123, "val != 123 (%ld)", val);
  val = $(parser, getAsInt)("--a", 1);
  CUTEST_ASSERT(val == -234, "val != -234 (%ld)", val);
  CapyArgParserFree(&parser);
  free(argv_[0]);
  free(argv_[1]);
  free(argv_[2]);
  free(argv_[3]);
}

CUTEST(test009, "Parsing string value") {
  CapyArg args_[] = {
    {.shortLbl = "--a", .nbVal = 2, .help = "test"},
  };
  char* argv_[4];
  argv_[0] = strCreate("cutest");
  argv_[1] = strCreate("--a");
  argv_[2] = strCreate("123");
  argv_[3] = strCreate("-234");
  CapyArgParser* parser =
    CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
  char const* val = $(parser, getAsStr)("--a", 0);
  CUTEST_ASSERT(strcmp(val, "123") == 0, "val != 123 (%s)", val);
  val = $(parser, getAsStr)("--a", 1);
  CUTEST_ASSERT(strcmp(val, "-234") == 0, "val != -234 (%s)", val);
  CapyArgParserFree(&parser);
  free(argv_[0]);
  free(argv_[1]);
  free(argv_[2]);
  free(argv_[3]);
}

CUTEST(test010, "Number of argument value") {
  CapyArg args_[] = {
    {.lbl = "cutest", .nbVal = capyArg_variableNbVal, .help = "test"},
    {.shortLbl = "--a", .nbVal = capyArg_variableNbVal, .help = "test"},
  };
  char* argv_[7];
  argv_[0] = strCreate("cutest");
  argv_[1] = strCreate("1");
  argv_[2] = strCreate("2");
  argv_[3] = strCreate("--a");
  argv_[4] = strCreate("123");
  argv_[5] = strCreate("-234");
  argv_[6] = strCreate("345");
  CapyArgParser* parser =
    CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
  size_t nb = $(parser, getNbVal)("--a");
  CUTEST_ASSERT(nb == 3, "nb != 3 (%ld)", nb);
  nb = $(parser, getNbVal)("cutest");
  CUTEST_ASSERT(nb == 2, "nb != 2 (%ld)", nb);
  CapyArgParserFree(&parser);
  free(argv_[0]);
  free(argv_[1]);
  free(argv_[2]);
  free(argv_[3]);
  free(argv_[4]);
  free(argv_[5]);
  free(argv_[6]);
}

CUTEST(test011, "Optional argument") {
  CapyArg args_[] = {
    {.lbl = "cutest", .nbVal = capyArg_variableNbVal, .help = "test"},
    {.shortLbl = "--a", .nbVal = capyArg_variableNbVal, .help = "test"},
  };
  {
    char* argv_[3];
    argv_[0] = strCreate("cutest");
    argv_[1] = strCreate("1");
    argv_[2] = strCreate("2");
    CapyArgParser* parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    bool isSet = $(parser, isSet)("--a");
    CUTEST_ASSERT(isSet == false, "missing argument reported as present");
    CapyArgParserFree(&parser);
    free(argv_[0]);
    free(argv_[1]);
    free(argv_[2]);
  }
  {
    char* argv_[4];
    argv_[0] = strCreate("cutest");
    argv_[1] = strCreate("1");
    argv_[2] = strCreate("2");
    argv_[3] = strCreate("--a");
    CapyArgParser* parser =
      CapyArgParserAllocSilent(sizeof(argv_) / sizeof(*argv_), argv_, args_);
    bool isSet = $(parser, isSet)("--a");
    CUTEST_ASSERT(isSet, "present argument reported as missing");
    CapyArgParserFree(&parser);
    free(argv_[0]);
    free(argv_[1]);
    free(argv_[2]);
    free(argv_[3]);
  }
}
