#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "Creation/destruction of a CapyBezier") {
  {
    CapyBezier* bezier = CapyBezierAlloc(1, 2, 3);
    CUTEST_ASSERT(bezier, "creation failed");
    CUTEST_ASSERT(
      bezier->order == 1,
      "initialisation failed, order = %d != 1", bezier->order);
    CUTEST_ASSERT(
      bezier->dimIn == 2,
      "initialisation failed, order = %lu != 2", bezier->dimIn);
    CUTEST_ASSERT(
      bezier->dimOut == 3,
      "initialisation failed, order = %lu != 3", bezier->dimOut);
    CUTEST_ASSERT(
      bezier->nbCtrls == 4,
      "initialisation failed, nbCtrl = %lu != 4", bezier->nbCtrls);
    CapyBezierFree(&bezier);
    CUTEST_ASSERT(bezier == NULL, "bezier not reset to NULL");
  }
  {
    CapyBezier* bezier = CapyBezierAlloc(2, 3, 4);
    CUTEST_ASSERT(
      bezier->nbCtrls == 27,
      "initialisation failed, nbCtrl = %lu != 27", bezier->nbCtrls);
    CapyBezierFree(&bezier);
  }
}

CUTEST(test002, "setCtrl, getCtrl and eval") {
  CapyBezier* bezier = CapyBezierAlloc(1, 2, 3);
  size_t idx[2];
  loop(i, 2) idx[i] = 0;
  CapyVec val = CapyVecCreateLocal3D;
  loop(i, bezier->dimOut) val.vals[i] = (double)i;
  $(bezier, setCtrl)(idx, val.vals);
  CapyVec const* ctrl = $(bezier, getCtrl)(idx);
  loop(i, bezier->dimOut) {
    CUTEST_ASSERT(
      equal(ctrl->vals[i], (double)i),
      "setCtrl failed [%lu]=%lf != %lu", i, ctrl->vals[i], i);
  }
  idx[0] = 1;
  val.vals[1] = -1.0;
  $(bezier, setCtrl)(idx, val.vals);
  double check[25][3] = {
    {0.000000, 1.000000, 2.000000},
    {0.000000, 0.750000, 1.500000},
    {0.000000, 0.500000, 1.000000},
    {0.000000, 0.250000, 0.500000},
    {0.000000, 0.000000, 0.000000},
    {0.000000, 0.500000, 2.000000},
    {0.000000, 0.375000, 1.500000},
    {0.000000, 0.250000, 1.000000},
    {0.000000, 0.125000, 0.500000},
    {0.000000, 0.000000, 0.000000},
    {0.000000, 0.000000, 2.000000},
    {0.000000, 0.000000, 1.500000},
    {0.000000, 0.000000, 1.000000},
    {0.000000, 0.000000, 0.500000},
    {0.000000, 0.000000, 0.000000},
    {0.000000, -0.500000, 2.000000},
    {0.000000, -0.375000, 1.500000},
    {0.000000, -0.250000, 1.000000},
    {0.000000, -0.125000, 0.500000},
    {0.000000, 0.000000, 0.000000},
    {0.000000, -1.000000, 2.000000},
    {0.000000, -0.750000, 1.500000},
    {0.000000, -0.500000, 1.000000},
    {0.000000, -0.250000, 0.500000},
    {0.000000, 0.000000, 0.000000},
  };
  loop(i, (size_t)5) loop(j, (size_t)5) {
    val.vals[0] = (double)i * 0.25;
    val.vals[1] = (double)j * 0.25;
    double out[3];
    $(bezier, eval)(val.vals, out);
    loop(k, bezier->dimOut) {
      CUTEST_ASSERT(
        equal(check[i*5+j][k], out[k]),
        "eval failed [%lu,%lu,%lu]=%lf != %lf",
        i, j, k, check[i*5+j][k], out[k]);
    }
  }
  ctrl = $(bezier, getCtrl)(idx);
  CUTEST_ASSERT(
    equal(ctrl->vals[1], -1.0), "getCtrl failed %lf != -1.0", ctrl->vals[1]);
  CUTEST_ASSERT(
    equal(ctrl->vals[2], 2.0), "getCtrl failed %lf != 2.0", ctrl->vals[2]);
  CapyBezierFree(&bezier);
}

CUTEST(test003, "save/load") {
  CapyBezier* bezier = CapyBezierAlloc(1, 2, 3);
  size_t idx[2];
  loop(i, 2) idx[i] = 0;
  CapyVec val = CapyVecCreateLocal3D;
  loop(i, bezier->dimOut) val.vals[i] = (double)i;
  $(bezier, setCtrl)(idx, val.vals);
  idx[0] = 1;
  val.vals[1] = -1.0;
  $(bezier, setCtrl)(idx, val.vals);
  FILE* stream = safeFOpen("UnitTests/TestBezier/bezier.bin", "wb");
  $(bezier, save)(stream);
  fclose(stream);
  stream = safeFOpen("UnitTests/TestBezier/bezier.bin", "rb");
  CapyBezier* bezierB = CapyBezierLoad(stream);
  fclose(stream);
  CUTEST_ASSERT(
    bezier->dimIn == bezierB->dimIn,
    "save/load failed dimIn=%lu != %lu",
    bezierB->dimIn, bezier->dimIn);
  CUTEST_ASSERT(
    bezier->dimOut == bezierB->dimOut,
    "save/load failed dimOut=%lu != %lu",
    bezierB->dimOut, bezier->dimOut);
  CUTEST_ASSERT(
    bezier->order == bezierB->order,
    "save/load failed order=%d != %d",
    bezierB->order, bezier->order);
  CUTEST_ASSERT(
    bezier->nbCtrls == bezierB->nbCtrls,
    "save/load failed nbCtrls=%lu != %lu",
    bezierB->nbCtrls, bezier->nbCtrls);
  loop(iCtrl, bezier->nbCtrls) {
    CapyVec const* u = $(bezier, getCtrlById)(iCtrl);
    CapyVec const* v = $(bezierB, getCtrlById)(iCtrl);
    loop(iVal, u->dim) {
      CUTEST_ASSERT(
        u->vals[iVal] == v->vals[iVal],
        "save/load failed vals[%lu]=%lf != %lf",
        iVal, u->vals[iVal], v->vals[iVal]);
    }
  }
  CapyBezierFree(&bezierB);
  CapyBezierFree(&bezier);
}

CUTEST(test004, "exportToCFun ") {
  CapyBezier* bezier = CapyBezierAlloc(1, 2, 3);
  size_t idx[2];
  loop(i, 2) idx[i] = 0;
  CapyVec val = CapyVecCreateLocal3D;
  loop(i, bezier->dimOut) val.vals[i] = (double)i;
  $(bezier, setCtrl)(idx, val.vals);
  idx[0] = 1;
  val.vals[1] = -1.0;
  $(bezier, setCtrl)(idx, val.vals);
  FILE* stream = safeFOpen("UnitTests/TestBezier/bezierFun.c", "w");
  $(bezier, exportToCFun)(stream, "bezier");
  fclose(stream);
  CUTEST_ASSERT(
    system(
      "diff UnitTests/TestBezier/bezierFun.c "
      "UnitTests/TestBezier/bezierFun_check.c") == 0,
    "exportToCFun failed");
  CUTEST_ASSERT(
    system(
      "gcc -o UnitTests/TestBezier/bezier UnitTests/TestBezier/bezier.c -I./ "
      "-lm && UnitTests/TestBezier/bezier") == 0,
    "exportToCFun failed");
  CUTEST_ASSERT(
    system("UnitTests/TestBezier/bezier") == 0,
    "exportToCFun failed");
  CapyBezierFree(&bezier);
}

CUTEST(test005, "Creation/destruction of a CapyBezierSpline") {
  {
    size_t nbSegment = 2;
    CapyBezierSpline* spline = CapyBezierSplineAlloc(&nbSegment, 1, 3);
    CUTEST_ASSERT(spline, "creation failed");
    CUTEST_ASSERT(
      spline->nbSegment[0] == 2,
      "initialisation failed, nbSegment = %lu != 2", spline->nbSegment[0]);
    CUTEST_ASSERT(
      spline->dimIn == 1,
      "initialisation failed, order = %lu != 1", spline->dimIn);
    CUTEST_ASSERT(
      spline->dimOut == 3,
      "initialisation failed, order = %lu != 3", spline->dimOut);
    CUTEST_ASSERT(
      spline->nbCurve == 2,
      "initialisation failed, nbCurve = %lu != 2", spline->nbCurve);
    CapyBezierSplineFree(&spline);
    CUTEST_ASSERT(spline == NULL, "spline not reset to NULL");
  }
  {
    size_t nbSegment[2] = {2, 2};
    CapyBezierSpline* spline = CapyBezierSplineAlloc(nbSegment, 2, 3);
    CUTEST_ASSERT(
      spline->nbCurve == 4,
      "initialisation failed, nbCurve = %lu != 4", spline->nbCurve);
    CapyBezierSplineFree(&spline);
  }
}

CUTEST(test006, "setCurve, getCurve and eval") {
  size_t nbSegment[2] = {2, 2};
  CapyBezierSpline* spline = CapyBezierSplineAlloc(nbSegment, 2, 3);
  {
    size_t idx[2] = {0, 0};
    CapyBezier* curve = $(spline, getCurve)(idx);
    CUTEST_ASSERT(
      curve == spline->curves[0],
      "getCurveFailed for (0, 0)");
  }
  {
    size_t idx[2] = {1, 0};
    CapyBezier* curve = $(spline, getCurve)(idx);
    CUTEST_ASSERT(
      curve == spline->curves[1],
      "getCurveFailed for (1, 0)");
  }
  {
    size_t idx[2] = {0, 1};
    CapyBezier* curve = $(spline, getCurve)(idx);
    CUTEST_ASSERT(
      curve == spline->curves[2],
      "getCurveFailed for (0, 1)");
  }
  {
    size_t idx[2] = {1, 1};
    CapyBezier* curve = $(spline, getCurve)(idx);
    CUTEST_ASSERT(
      curve == spline->curves[3],
      "getCurveFailed for (1, 1)");
  }
  double in[2];
  double out[3];
  in[0] = 0.5;
  in[1] = 0.5;
  $(spline, eval)(in, out);
  CUTEST_ASSERT(
    equal(out[0], 0.0) && equal(out[1], 0.0) && equal(out[2], 0.0),
    "eval failed for (%lf, %lf)->(%lf, %lf, %lf) != (%lf, %lf, %lf)",
    in[0], in[1], out[0], out[1], out[2], 0.0, 0.0, 0.0);
  CapyBezier* curve = CapyBezierAlloc(1, 2, 3);
  curve->ctrls[0].vals[0] = 1.0;
  curve->ctrls[1].vals[0] = 2.0;
  curve->ctrls[2].vals[0] = 3.0;
  curve->ctrls[3].vals[0] = 4.0;
  size_t idx[2] = {1, 0};
  $(spline, setCurve)(idx, curve);
  $(spline, eval)(in, out);
  CUTEST_ASSERT(
    equal(out[0], 0.0) && equal(out[1], 0.0) && equal(out[2], 0.0),
    "eval failed for (%lf, %lf)->(%lf, %lf, %lf) != (%lf, %lf, %lf)",
    in[0], in[1], out[0], out[1], out[2], 0.0, 0.0, 0.0);
  in[0] = 1.5;
  in[1] = 0.5;
  $(spline, eval)(in, out);
  CUTEST_ASSERT(
    equal(out[0], 2.5) && equal(out[1], 0.0) && equal(out[2], 0.0),
    "eval failed for (%lf, %lf)->(%lf, %lf, %lf) != (%lf, %lf, %lf)",
    in[0], in[1], out[0], out[1], out[2], 2.5, 0.0, 0.0);
  CapyBezierSplineFree(&spline);
}

CUTEST(test007, "Bezier iterator") {
  CapyBezier* curve = CapyBezierAlloc(2, 1, 2);
  curve->ctrls[0].vals[0] = 1.0;
  curve->ctrls[1].vals[0] = 2.0;
  curve->ctrls[2].vals[0] = 1.0;
  curve->ctrls[0].vals[1] = 4.0;
  curve->ctrls[1].vals[1] = 3.0;
  curve->ctrls[2].vals[1] = 4.0;
  CapyBezierIterator* iter = CapyBezierIteratorAlloc(curve);
  double check[19][3] = {
    {0.000000, 1.000000, 4.000000},
    {0.031250, 1.060547, 3.939453},
    {0.062500, 1.117188, 3.882812},
    {0.093750, 1.169922, 3.830078},
    {0.125000, 1.218750, 3.781250},
    {0.156250, 1.263672, 3.736328},
    {0.187500, 1.304688, 3.695312},
    {0.250000, 1.375000, 3.625000},
    {0.312500, 1.429688, 3.570312},
    {0.375000, 1.468750, 3.531250},
    {0.500000, 1.500000, 3.500000},
    {0.625000, 1.468750, 3.531250},
    {0.687500, 1.429688, 3.570312},
    {0.750000, 1.375000, 3.625000},
    {0.812500, 1.304688, 3.695312},
    {0.843750, 1.263672, 3.736328},
    {0.875000, 1.218750, 3.781250},
    {0.906250, 1.169922, 3.830078},
    {0.937500, 1.117188, 3.882812},
  };
  forEach(pos, *iter) {
    CUTEST_ASSERT(
      fabs(pos.in[0] - check[iter->idx][0]) < 1e-6 &&
      fabs(pos.out[0] - check[iter->idx][1]) < 1e-6 &&
      fabs(pos.out[1] - check[iter->idx][2]) < 1e-6,
      "[%lu]{%.6lf, %.6lf, %.6lf} != {%.6lf, %.6lf, %.6lf}",
      iter->idx, pos.in[0], pos.out[0], pos.out[1],
      check[iter->idx][0], check[iter->idx][1], check[iter->idx][2]);
  }
  CapyBezierIteratorFree(&iter);
  CapyBezierFree(&curve);
}
