#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
static void Fun(
  double const* const in,
        double* const out) {
  out[0] = in[0] * in[0] + 2.0 * in[1] * in[1];
}

static void Fun2(
  double const* const in,
        double* const out) {
  out[0] = in[0] * in[0] + 2.0;
}

#endif
CUTEST(test001, "Gcd") {
  uint64_t v = CapyGcd(0, 0);
  CUTEST_ASSERT(v == 0, "Gcd failed %lu != 0", v);
  v = CapyGcd(0, 1);
  CUTEST_ASSERT(v == 1, "Gcd failed %lu != 1", v);
  v = CapyGcd(1, 0);
  CUTEST_ASSERT(v == 1, "Gcd failed %lu != 1", v);
  v = CapyGcd(UINT64_MAX, UINT64_MAX / 2);
  CUTEST_ASSERT(v == 1, "Gcd failed %lu != 1", v);
  v = CapyGcd((UINT64_MAX / 2) * 2, UINT64_MAX / 2);
  CUTEST_ASSERT(
    v == UINT64_MAX / 2, "Gcd failed %lu != %lu", v, UINT64_MAX / 2);
  loop(i, (uint64_t)255) {
    v = CapyGcd(i, i);
    CUTEST_ASSERT(v == i, "Gcd failed %lu != %lu", v, i);
  }
}

CUTEST(test002, "Peasant multiplication ") {
  loop(i, (uint64_t)100) loop(j, (uint64_t)100) {
    uint64_t v = CapyPeasantMul(i, j);
    CUTEST_ASSERT(
      v == i * j, "Peasant multiplication failed %lu != %lu", v, i * j);
  }
}

CUTEST(test003, "Peasant multiplication and division ") {
  CapyPeasantMulDivRes res;
  #define N 26
  uint64_t v[N][3] = {
    {0, 0, 0},
    {1, 0, 1},
    {0, 1, 1},
    {1, 2, 1},
    {2, 1, 1},
    {1, 2, 3},
    {2, 1, 3},
    {2, 3, 4},
    {UINT64_MAX / 2, 2, 1},
    {3, 5, 7},
    {2, UINT64_MAX, 3},
    {UINT64_MAX, 2, 3},
    {UINT64_MAX, 2, 7},
    {2, UINT64_MAX, 7},
    {100, UINT64_MAX, (UINT64_MAX / 3) * 2},
    {UINT64_MAX, 100, (UINT64_MAX / 3) * 2},
    {10, UINT64_MAX, (UINT64_MAX / 3) * 2},
    {UINT64_MAX, 10, (UINT64_MAX / 3) * 2},
    {UINT64_MAX, 4, 3},
    {4, UINT64_MAX, 3},
    {UINT64_MAX, UINT64_MAX - 1, UINT64_MAX - 2},
    {UINT64_MAX - 1, UINT64_MAX - 2, UINT64_MAX},
    {UINT64_MAX - 2, UINT64_MAX - 1, UINT64_MAX},
    {2, 3, UINT64_MAX},
    {3, 2, UINT64_MAX},
    {1, 4, 2},
  };
  uint64_t check[N][4] = {
    {0, 0, 0, 0},
    {0, 0, 0, 1},
    {0, 0, 0, 1},
    {2, 0, 0, 1},
    {2, 0, 0, 1},
    {0, 0, 2, 3},
    {0, 0, 2, 3},
    {1, 0, 1, 2},
    {UINT64_MAX - 1, 0, 0, 1},
    {2, 0, 1, 7},
    {12297829382473034410UL, 0, 0, 1},
    {12297829382473034410UL, 0, 0, 1},
    {5270498306774157604UL, 0, 2, 7},
    {5270498306774157604UL, 0, 2, 7},
    {150, 0, 0, 1},
    {150, 0, 0, 1},
    {15, 0, 0, 1},
    {15, 0, 0, 1},
    {0, 0, 0, 0},
    {0, 0, 0, 0},
    {0, 0, 0, 0},
    {UINT64_MAX - 3, 0, 2, UINT64_MAX},
    {UINT64_MAX - 3, 0, 2, UINT64_MAX},
    {0, 0, 2, UINT64_MAX / 3},
    {0, 0, 2, UINT64_MAX / 3},
    {0, 2, 0, 1},
  };
  loop(i, N) {
    res = CapyPeasantMulDiv(v[i][0], v[i][1], v[i][2]);
    CUTEST_ASSERT(
      res.base == check[i][0],
      "PeasantMulDiv failed %lu != %lu", res.base, check[i][0]);
    CUTEST_ASSERT(
      (uint64_t)(res.frac.base) == check[i][1],
      "PeasantMulDiv failed %lu != %lu", res.frac.base, check[i][1]);
    CUTEST_ASSERT(
      res.frac.num == check[i][2],
      "PeasantMulDiv failed %lu != %lu", res.frac.num, check[i][2]);
    CUTEST_ASSERT(
      res.frac.den == check[i][3],
      "PeasantMulDiv failed %lu != %lu", res.frac.den, check[i][3]);
  }
}

CUTEST(test004, "SmoothStep") {
  double checka[] = {
    0.0, 0.25, 0.333333, 0.395644, 0.44949,
    0.5, 0.55051, 0.604356, 0.666667, 0.75, 1.0
  };
  double checkb[] = {
    0.0, 0.012195, 0.058824, 0.155172, 0.307692,
    0.5, 0.692308, 0.844828, 0.941176, 0.987805, 1.0
  };
  loop(i, 11) {
    double x = 0.1 * (double)i;
    double y = CapySmoothStep(x, 1.0);
    CUTEST_ASSERT(
      fabs(x - y) < 1e-6, "SmoothStep failed %lf != %lf", x, y);
    y = CapySmoothStep(x, 0.5);
    CUTEST_ASSERT(
      fabs(checka[i] - y) < 1e-6, "SmoothStep failed %lf != %lf", checka[i], y);
    y = CapySmoothStep(x, 2.0);
    CUTEST_ASSERT(
      fabs(checkb[i] - y) < 1e-6, "SmoothStep failed %lf != %lf", checkb[i], y);
  }
}

CUTEST(test005, "Powi") {
  loop(i, 10) loop(j, 10) {
    double x = (double)CapyPowi(i, (uint64_t)j);
    double y = pow(i, j);
    CUTEST_ASSERT(equal(x, y), "Powi failed %lf != %lf", x, y);
  }
}

CUTEST(test006, "Lerp") {
  CapyRangeDouble rangeA = { .min = 1.0, .max = 10.0 };
  CapyRangeDouble rangeB = { .min = 10.0, .max = 100.0 };
  loop(i, 11) {
    double x = CapyLerp(i, &rangeA, &rangeB);
    double y = 10.0 * (double)i;
    CUTEST_ASSERT(equal(x, y), "Lerp failed %lf != %lf", x, y);
  }
}

CUTEST(test007, "evalDerivative/evalJacobian") {
  CapyMathFun fun = CapyMathFunCreate(2, 1);
  fun.eval = Fun;
  double in[2];
  double out[1];
  in[0] = 0.0; in[1] = 0.0;
  $(&fun, evalDerivative)(in, 0, out);
  CUTEST_ASSERT(
    fabs(out[0] - 0.0) < 1e-6, "evalDerivative failed %lf != 0.0", out[0]);
  $(&fun, evalDerivative)(in, 1, out);
  CUTEST_ASSERT(
    fabs(out[0] - 0.0) < 1e-6, "evalDerivative failed %lf != 0.0", out[0]);
  in[0] = 1.0; in[1] = 0.0;
  $(&fun, evalDerivative)(in, 0, out);
  CUTEST_ASSERT(
    fabs(out[0] - 2.0) < 1e-6, "evalDerivative failed %lf != 2.0", out[0]);
  $(&fun, evalDerivative)(in, 1, out);
  CUTEST_ASSERT(
    fabs(out[0] - 0.0) < 1e-6, "evalDerivative failed %lf != 0.0", out[0]);
  in[0] = 0.0; in[1] = 1.0;
  $(&fun, evalDerivative)(in, 0, out);
  CUTEST_ASSERT(
    fabs(out[0] - 0.0) < 1e-6, "evalDerivative failed %lf != 0.0", out[0]);
  $(&fun, evalDerivative)(in, 1, out);
  CUTEST_ASSERT(
    fabs(out[0] - 4.0) < 1e-6, "evalDerivative failed %lf != 4.0", out[0]);
  double jacobian[2];
  $(&fun, evalJacobian)(in, jacobian);
  CUTEST_ASSERT(
    fabs(jacobian[0] - 0.0) < 1e-6,
    "evalJacobian failed %lf != 0.0", jacobian[0]);
  CUTEST_ASSERT(
    fabs(jacobian[1] - 4.0) < 1e-6,
    "evalJacobian failed %lf != 4.0", jacobian[1]);
}

CUTEST(test008, "CapyVec") {
  CapyVec u = {.dim = 3, .vals = (double[3]){1.0, 2.0, 3.0}};
  CapyVec v = {.dim = 3, .vals = (double[3]){4.0, 5.0, 6.0}};
  CapyVec w = {.dim = 3, .vals = (double[3]){0.0, 0.0, 0.0}};
  CapyVecCopy(&u, &w);
  CUTEST_ASSERT(
    equal(w.vals[0], u.vals[0]),
    "VecCopy failed %lf != %lf", w.vals[0], u.vals[0]);
  CUTEST_ASSERT(
    equal(w.vals[1], u.vals[1]),
    "VecCopy failed %lf != %lf", w.vals[1], u.vals[1]);
  CUTEST_ASSERT(
    equal(w.vals[2], u.vals[2]),
    "VecCopy failed %lf != %lf", w.vals[2], u.vals[2]);
  CapyVecAdd(&u, &v, &w);
  CUTEST_ASSERT(equal(w.vals[0], 5.0), "VecAdd failed %lf != 5.0", w.vals[0]);
  CUTEST_ASSERT(equal(w.vals[1], 7.0), "VecAdd failed %lf != 7.0", w.vals[1]);
  CUTEST_ASSERT(equal(w.vals[2], 9.0), "VecAdd failed %lf != 9.0", w.vals[2]);
  CapyVecSub(&u, &v, &w);
  CUTEST_ASSERT(equal(w.vals[0], -3.0), "VecSub failed %lf != -3.0", w.vals[1]);
  CUTEST_ASSERT(equal(w.vals[1], -3.0), "VecSub failed %lf != -3.0", w.vals[2]);
  CUTEST_ASSERT(equal(w.vals[2], -3.0), "VecSub failed %lf != -3.0", w.vals[3]);
  CapyVecCross(&u, &v, &w);
  CUTEST_ASSERT(
    equal(w.vals[0], -3.0), "VecCross failed %lf != -3.0", w.vals[0]);
  CUTEST_ASSERT(
    equal(w.vals[1], 6.0), "VecCross failed %lf != 6.0", w.vals[1]);
  CUTEST_ASSERT(
    equal(w.vals[2], -3.0), "VecCross failed %lf != -3.0", w.vals[2]);
  CapyVecMul(&u, 2.0, &w);
  CUTEST_ASSERT(equal(w.vals[0], 2.0), "VecMul failed %lf != 5.0", w.vals[0]);
  CUTEST_ASSERT(equal(w.vals[1], 4.0), "VecMul failed %lf != 5.0", w.vals[1]);
  CUTEST_ASSERT(equal(w.vals[2], 6.0), "VecMul failed %lf != 5.0", w.vals[2]);
  double a;
  CapyVecDot(&u, &v, &a);
  CUTEST_ASSERT(equal(a, 32.0), "VecDot failed %lf != 32.0", a);
  CapyVecNormalise(&w);
  double n[3] = {2.0 / sqrt(56.0), 4.0 / sqrt(56.0), 6.0 / sqrt(56.0)};
  loop(i, 3) {
    CUTEST_ASSERT(
      equal(w.vals[i], n[i]),
      "VecNormalised failed %lf != %lf", w.vals[i], n[i]);
  }
  CapyVecNormaliseFast(&v);
  double nf[3] = {4.0 / sqrt(77.0), 5.0 / sqrt(77.0), 6.0 / sqrt(77.0)};
  loop(i, 3) {
    CUTEST_ASSERT(
      fabs(v.vals[i] - nf[i]) < 1e-3,
      "VecNormaliseFast failed %lf != %lf", v.vals[i], nf[i]);
  }
  CapyVec x = {.dim = 2, .vals = (double[2]){0.0, 0.0}};
  loop(i, 100) loop(j, 100) {
    x.vals[0] = -10. + .2 * (double)i;
    x.vals[1] = -10. + .2 * (double)j;
    double rn = CapyVecGetNorm(&x);
    a = CapyVec2DApproxNorm(&x);
    double t = 0.04 * fabs(rn) + 1e-9;
    CUTEST_ASSERT(
      fabs(rn - a) <= t, "Vec2DApproxNorm failed %lf > %lf", rn - a, t);
  }
  double xAngles[] =
    {sqrt(2.0), 0.0, 0.0, sqrt(2.0), 1.0, 1.0, -1.0, 1.0, -1.0, -1.0};
  double angles[] =
    {0.0, M_PI / 2.0, M_PI / 4.0, 3.0 * M_PI / 4.0, -3.0 * M_PI / 4.0};
  loop(i, 5) {
    x.vals[0] = xAngles[i * 2];
    x.vals[1] = xAngles[i * 2 + 1];
    double theta = CapyVec2DGetAngle(&x, &capyXAxis2D);
    CUTEST_ASSERT(
      fabs(theta - angles[i]) < 1e-6,
      "Vec2DGetAngle failed %lf != %lf", theta, angles[i]);
    x.vals[0] = sqrt(2.0);
    x.vals[1] = 0.0;
    CapyVec2DSetAngle(&x, angles[i], &x);
    CUTEST_ASSERT(
      fabs(x.vals[0] - xAngles[2 * i]) < 1e-6 &&
      fabs(x.vals[1] - xAngles[2 * i + 1]) < 1e-6,
      "Vec2DSetAngle failed (%lf,%lf) != (%lf,%lf)",
      x.vals[0], x.vals[1], xAngles[2 * i], xAngles[2 * i + 1]);
  }
  CapyVec xRotate = {.dim = 2, .vals = (double[2]){sqrt(2.0), 0.0}};
  CapyVec2DRotate(&xRotate, M_PI_4, &xRotate);
  CUTEST_ASSERT(
    fabs(xRotate.vals[0] - 1.0) < 1e-6 &&
    fabs(xRotate.vals[1] - 1.0) < 1e-6,
    "Vec2DRotate failed (%lf,%lf) != (%lf,%lf)",
    xRotate.vals[0], xRotate.vals[1], 1.0, 1.0);
}

CUTEST(test009, "CapyMat") {
  CapyMat a = {
    .nbCol = 2, .nbRow = 2, .vals = (double[4]){1.0, 2.0, -3.0, 1.0}
  };
  CapyVec u = {.dim = 2, .vals = (double[2]){2.0, 3.0}};
  CapyVec v = {.dim = 2, .vals = (double[2]){0.0, 0.0}};
  CapyMatProdVec(&a, &u, &v);
  CUTEST_ASSERT(
    equal(v.vals[0], 8.0), "MatProdVec failed %lf != 8.0", v.vals[0]);
  CUTEST_ASSERT(
    equal(v.vals[1], -3.0), "MatProdVec failed %lf != -3.0", v.vals[1]);
  CapyMatTransProdVec(&a, &u, &v);
  CUTEST_ASSERT(
    equal(v.vals[0], -7.0), "MatTransProdVec failed %lf != -7.0", v.vals[0]);
  CUTEST_ASSERT(
    equal(v.vals[1], 7.0), "MatTransProdVec failed %lf != 7.0", v.vals[1]);
  double b;
  CapyMatDet(&a, &b);
  CUTEST_ASSERT(equal(b, 7.0), "MatDet failed %lf != 7.0", b);
  CapyMat c = {.nbCol = 2, .nbRow = 2, .vals = (double[4]){0.0, 0.0, 0.0, 0.0}};
  CapyMatInv(&a, &c);
  double inv[4] = {
    0.14285714285714284921, -0.28571428571428569843,
    0.42857142857142854764, 0.14285714285714284921
  };
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(c.vals[i], inv[i]), "MatInv failed %lf != %lf", c.vals[i], inv[i]);
  }
  CapyMat d = {.nbCol = 2, .nbRow = 2, .vals = (double[4]){0.0, 0.0, 0.0, 0.0}};
  CapyMatProdMat(&a, &c, &d);
  double prod[4] = {1.0, 0.0, 0.0, 1.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(d.vals[i], prod[i]),
      "MatProdMat failed %lf != %lf", d.vals[i], prod[i]);
  }
  CapyMatPseudoInv(&a, &c);
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(c.vals[i], inv[i]),
      "MatPseudoInv failed %lf != %lf", c.vals[i], inv[i]);
  }
  CapyMatTransp(&a, &c);
  double trans[4] = {1.0, -3.0, 2.0, 1.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(c.vals[i], trans[i]),
      "MatTrans failed %lf != %lf", c.vals[i], trans[i]);
  }
}

CUTEST(test010, "solveByNewton") {
  CapyMathFun fun = CapyMathFunCreate(1, 1);
  fun.eval = Fun2;
  double in[1];
  double out[1];
  in[0] = 3.0;
  out[0] = 3.0;
  $(&fun, solveByNewtonMethod)(in, out);
  CUTEST_ASSERT(
    fabs(in[0] - 1.0) < 1e-5, "solveByNewton failed %lf != 1.0", in[0]);
  in[0] = -3.0;
  $(&fun, solveByNewtonMethod)(in, out);
  CUTEST_ASSERT(
    fabs(in[0] + 1.0) < 1e-5, "solveByNewton failed %lf != -1.0", in[0]);
}

CUTEST(test011, "PowMod") {
  uint64_t x = CapyPowMod(3, 45, 7);
  CUTEST_ASSERT(x == 6, "PowMod failed %lu != 6", x);
  x = CapyPowMod(23, 373, 747);
  CUTEST_ASSERT(x == 131, "PowMod failed %lu != 131", x);
  x = CapyPowMod(65, 17, 3233);
  CUTEST_ASSERT(x == 2790, "PowMod failed %lu != 2790", x);
  x = CapyPowMod(2790, 413, 3233);
  CUTEST_ASSERT(x == 65, "PowMod failed %lu != 65", x);
  x = CapyPowMod(9223372036854775807, 9223372036854775807, 100000000000);
  CUTEST_ASSERT(x == 61604562943, "PowMod failed %lu != 61604562943", x);
}

CUTEST(test012, "GetQR (1)") {
  CapyMat mat = CapyMatCreate(3, 3);
  double vals[9] = {
    2.0, -2.0, 18.0,
    2.0, 1.0, 0.0,
    1.0, 2.0, 0.0
  };
  loop(i, 9) mat.vals[i] = vals[i];
  CapyMat Q = CapyMatCreate(3, 3);
  CapyMat R = CapyMatCreate(3, 3);
  try {
    CapyMatGetQR(&mat, &Q, &R);
  } catchDefault {
    CUTEST_ASSERT(false, "GetQR failed");
  } endCatch;
  CapyForwardExc();
  CapyMat check = CapyMatCreate(3, 3);
  CapyMatProdMat(&Q, &R, &check);
  loop(i, 9) {
    CUTEST_ASSERT(
      fabs(mat.vals[i] - check.vals[i]) < 1e-6,
      "GetQR failed [%d]=%lf != %lf", i, mat.vals[i], check.vals[i]);
  }
  double checkQ[9] = {
    -2.0 / 3.0, 2.0 / 3.0, -1.0 / 3.0,
    -2.0 / 3.0, -1.0 / 3.0, 2.0 / 3.0,
    -1.0 / 3.0, -2.0 / 3.0, -2.0 / 3.0
  };
  double checkR[9] = {
    -3.0, 0.0, -12.0,
    0.0, -3.0, 12.0,
    0.0, 0.0, -6.0
  };
  loop(i, 9) {
    CUTEST_ASSERT(
      fabs(Q.vals[i] - checkQ[i]) < 1e-6,
      "GetQR failed [%d]=%lf != %lf", i, Q.vals[i], checkQ[i]);
  }
  loop(i, 9) {
    CUTEST_ASSERT(
      fabs(R.vals[i] - checkR[i]) < 1e-6,
      "GetQR failed [%d]=%lf != %lf", i, R.vals[i], checkR[i]);
  }
  CapyMatDestruct(&mat);
  CapyMatDestruct(&Q);
  CapyMatDestruct(&R);
  CapyMatDestruct(&check);
}

CUTEST(test013, "GetQR (2)") {
  CapyMat mat = CapyMatCreate(3, 4);
  double vals[12] = {
    -1.0, -1.0, 1.0,
    1.0, 3.0, 3.0,
    -1.0, -1.0, 5.0,
    1.0, 3.0, 7.0
  };
  loop(i, 12) mat.vals[i] = vals[i];
  CapyMat Q = CapyMatCreate(3, 4);
  CapyMat R = CapyMatCreate(3, 3);
  try {
    CapyMatGetQR(&mat, &Q, &R);
  } catchDefault {
    CUTEST_ASSERT(false, "GetQR failed");
  } endCatch;
  CapyForwardExc();
  CapyMat check = CapyMatCreate(3, 4);
  CapyMatProdMat(&Q, &R, &check);
  loop(i, 12) {
    CUTEST_ASSERT(
      fabs(mat.vals[i] - check.vals[i]) < 1e-6,
      "GetQR failed %lf != %lf", mat.vals[i], check.vals[i]);
  }
  double checkQ[12] = {
    -0.5, -0.5, 0.5,
    0.5, -0.5, 0.5,
    -0.5, -0.5, -0.5,
    0.5, -0.5, -0.5
  };
  double checkR[9] = {
    2.0, 4.0, 2.0,
    0.0, -2.0, -8.0,
    0.0, 0.0, -4.0
  };
  loop(i, 12) {
    CUTEST_ASSERT(
      fabs(Q.vals[i] - checkQ[i]) < 1e-6,
      "GetQR failed %lf != %lf", Q.vals[i], checkQ[i]);
  }
  loop(i, 9) {
    CUTEST_ASSERT(
      fabs(R.vals[i] - checkR[i]) < 1e-6,
      "GetQR failed %lf != %lf", R.vals[i], checkR[i]);
  }
  CapyMatDestruct(&mat);
  CapyMatDestruct(&Q);
  CapyMatDestruct(&R);
  CapyMatDestruct(&check);
}

CUTEST(test014, "GetEigen") {
  CapyMat mat = CapyMatCreate(3, 3);
  double vals[9] = {
    -2.0, -4.0, 2.0,
    -2.0, 1.0, 2.0,
    4.0, 2.0, 5.0
  };
  loop (i, 9) mat.vals[i] = vals[i];
  CapyVec eigenVal = CapyVecCreate(3);
  CapyMat eigenVec = CapyMatCreate(3, 3);
  CapyMatGetEigen(&mat, &eigenVal, &eigenVec);
  double v[3] = {6.0, -5.0, 3.0};
  loop(i, 3) {
    CUTEST_ASSERT(
      fabs(eigenVal.vals[i] - v[i]) < 1e-6,
      "GetEigen failed %lf != %lf", eigenVal.vals[i], v[i]);
  }
  double checks[9] = {
    0.058421, 0.843133, 0.534522,
    0.350524, 0.484021, -0.801784,
    0.934730, -0.234203, 0.267261,
  };
  loop(i, 9) {
    CUTEST_ASSERT(
      fabs(eigenVec.vals[i] - checks[i]) < 1e-6,
      "GetEigen failed %lf != %lf", eigenVal.vals[i], checks[i]);
  }
  CapyMatDestruct(&mat);
  CapyMatDestruct(&eigenVec);
  CapyVecDestruct(&eigenVal);
}

CUTEST(test015, "DegToRad/RadToDeg") {
  double x[3] = {0.0, 90.0, 180.0};
  double y[3] = {0.0, M_PI_2, M_PI};
  loop(i, 3) {
    double v = CapyDegToRad(x[i]);
    CUTEST_ASSERT(equal(v, y[i]), "DegToRad failed %lf != %lf", v, y[i]);
    v = CapyRadToDeg(y[i]);
    CUTEST_ASSERT(equal(v, x[i]), "RadToDeg failed %lf != %lf", v, x[i]);
  }
}

CUTEST(test016, "Padding") {
  CUTEST_ASSERT(
    offsetof(CapyMat, nbCol) == offsetof(CapyMat, dims[0]),
    "offset of nbCol and dims[0] doesn't match %lu!=%lu",
    offsetof(CapyMat, nbCol), offsetof(CapyMat, dims[0]));
  CUTEST_ASSERT(
    offsetof(CapyMat, nbRow) == offsetof(CapyMat, dims[1]),
    "offset of nbRow and dims[1] doesn't match %lu!=%lu",
    offsetof(CapyMat, nbRow), offsetof(CapyMat, dims[1]));
}

CUTEST(test017, "Cosine-similarity") {
  CapyVec u = {.dim = 3, .vals = (double[3]){1.0, 2.0, 3.0}};
  CapyVec v = {.dim = 3, .vals = (double[3]){4.0, 5.0, 6.0}};
  double similarity = CapyVecGetCosineSimilarity(&u, &u);
  CUTEST_ASSERT(equal(similarity, 1.0), "similarity=%lf != 1.0", similarity);
  similarity = CapyVecGetCosineSimilarity(&u, &v);
  CUTEST_ASSERT(
    fabs(similarity - 0.974632) < 1e-6,
    "similarity=%lf != 0.974632", similarity);
}

CUTEST(test018, "CapyMatAddMat") {
  CapyMat a = {
    .nbCol = 2, .nbRow = 2, .vals = (double[4]){1.0, 2.0, -3.0, 1.0}
  };
  CapyMat b = {
    .nbCol = 2, .nbRow = 2, .vals = (double[4]){0.0, 1.0, 2.0, 3.0}
  };
  CapyMat c = {
    .nbCol = 2, .nbRow = 2, .vals = (double[4]){0.0, 0.0, 0.0, 0.0}
  };
  CapyMatAddMat(&a, &b, &c);
  double res[4] = {1.0, 3.0, -1.0, 4.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(c.vals[i], res[i]),
      "MatAddMat failed %lf != %lf", c.vals[i], res[i]);
  }
}

CUTEST(test019, "CapyMatProdScalar") {
  CapyMat a = {
    .nbCol = 2, .nbRow = 2, .vals = (double[4]){0.0, 1.0, 2.0, 3.0}
  };
  double b = 2.0;
  CapyMat c = {
    .nbCol = 2, .nbRow = 2, .vals = (double[4]){0.0, 0.0, 0.0, 0.0}
  };
  CapyMatProdScalar(&a, b, &c);
  double res[4] = {0.0, 2.0, 4.0, 6.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(c.vals[i], res[i]),
      "MatAddMat failed %lf != %lf", c.vals[i], res[i]);
  }
}

CUTEST(test020, "CapyVecSoftmax") {
  CapyVec u = {
    .dim = 3, .vals = (double[3]){1.0, 2.0, 8.0}
  };
  CapyVecSoftmax(&u, 1.0);
  double res[3] = {0.000909, 0.002470, 0.996621};
  loop(i, 3) {
    CUTEST_ASSERT(
      fabs(u.vals[i] - res[i]) < 1e-6,
      "VecSoftmax failed %lf != %lf", u.vals[i], res[i]);
  }
}

CUTEST(test021, "CapyLcm") {
  uint64_t lcm = CapyLcm(21, 6);
  CUTEST_ASSERT(lcm == 42, "lcm = %lu", lcm);
}

CUTEST(test022, "CapyGcdDecomposition") {
  int64_t x = 0;
  int64_t y = 0;
  int64_t gcd = CapyGcdDecomposition(240, 46, &x, &y);
  CUTEST_ASSERT(
    gcd == 2 && x == -9 && y == 47,
    "gcd = %ld, x = %ld, y = %ld", gcd, x, y);
  uint64_t gcdu = CapyGcdDecompositionUnsignedInput(240, 46, &x, &y);
  CUTEST_ASSERT(
    gcdu == 2 && x == -9 && y == 47,
    "gcd = %ld, x = %ld, y = %ld", gcd, x, y);
  gcd = CapyGcdDecomposition(1914, 899, &x, &y);
  CUTEST_ASSERT(
    gcd == 29 && x == 8 && y == -17,
    "gcd = %ld, x = %ld, y = %ld", gcd, x, y);
  gcdu = CapyGcdDecompositionUnsignedInput(1914, 899, &x, &y);
  CUTEST_ASSERT(
    gcdu == 29 && x == 8 && y == -17,
    "gcd = %ld, x = %ld, y = %ld", gcd, x, y);
  gcd = CapyGcdDecomposition(1432, 123211, &x, &y);
  CUTEST_ASSERT(
    gcd == 1 && x == -22973 && y == 267,
    "gcd = %ld, x = %ld, y = %ld", gcd, x, y);
  gcdu = CapyGcdDecompositionUnsignedInput(1432, 123211, &x, &y);
  CUTEST_ASSERT(
    gcd == 1 && x == -22973 && y == 267,
    "gcd = %ld, x = %ld, y = %ld", gcd, x, y);
}

CUTEST(test023, "MultMod") {
  uint64_t x =
    CapyMultMod(9223372036854775807, 9223372036854775807, 100000000000);
  CUTEST_ASSERT(x == 84232501249, "PowMod failed %lu != 84232501249", x);
}

CUTEST(test024, "Fibonacci numbers") {
  uint64_t v[5] = {0};
  uint64_t check[5] = {1, 1, 2, 3, 5};
  loop(i, 5) v[i] = CapyGetFibonacciNumber((uint64_t)i);
  bool isOk = true;
  loop(i, 5) isOk &= (v[i] == check[i]);
  CUTEST_ASSERT(isOk, "CapyGetFibonacciNumber failed");
}

CUTEST(test025, "Vector lerp") {
  CapyVec u = CapyVecCreate(2);
  CapyVec v = CapyVecCreate(2);
  v.vals[0] = 1.0;
  v.vals[1] = 2.0;
  double const t = 0.1;
  CapyVecLerp(&u, &v, &u, t);
  CUTEST_ASSERT(
    fabs(u.vals[0] - 0.1) < 1e-6 &&
    fabs(u.vals[1] - 0.2) < 1e-6,
    "CapyVecLerp failed {%lf, %lf}", u.vals[0], u.vals[1]);
  CapyVecDestruct(&u);
  CapyVecDestruct(&v);
}

CUTEST(test026, "Quadratic easing") {
  double checks[10][4] = {
    {0.000000, 0.000000, 0.000000, 0.000000},
    {0.012346, 0.209877, 0.024691, 0.197531},
    {0.049383, 0.395062, 0.098765, 0.345679},
    {0.111111, 0.555556, 0.222222, 0.444444},
    {0.197531, 0.691358, 0.395062, 0.493827},
    {0.308642, 0.802469, 0.604938, 0.506173},
    {0.444444, 0.888889, 0.777778, 0.555556},
    {0.604938, 0.950617, 0.901235, 0.654321},
    {0.790123, 0.987654, 0.975309, 0.802469},
    {1.000000, 1.000000, 1.000000, 1.000000},
  };
  double vals[10][4];
  bool isOk = true;
  loop(i, 10) {
    vals[i][0] = CapyEaseQuadraticAccel(0.0, 1.0, ((double)i) / 9.0);
    vals[i][1] = CapyEaseQuadraticDecel(0.0, 1.0, ((double)i) / 9.0);
    vals[i][2] = CapyEaseQuadraticAccelDecel(0.0, 1.0, ((double)i) / 9.0);
    vals[i][3] = CapyEaseQuadraticDecelAccel(0.0, 1.0, ((double)i) / 9.0);
    isOk &=
      (fabs(vals[i][0] - checks[i][0]) < 1e-6) &&
      (fabs(vals[i][1] - checks[i][1]) < 1e-6) &&
      (fabs(vals[i][2] - checks[i][2]) < 1e-6) &&
      (fabs(vals[i][3] - checks[i][3]) < 1e-6);
  }
  CUTEST_ASSERT(isOk, "Quadratic easing failed");
}

CUTEST(test027, "CapyVecGetMedian (1)") {
  CapyVec u = CapyVecCreate(10);
  double vals[10] = {2, 3, 5, 0, 1, 6, 2, 7, 4, 5};
  loop(i, 10) u.vals[i] = vals[i];
  double median = CapyVecGetMedian(&u);
  CUTEST_ASSERT(fabs(median - 3.5) < 1e-6, "median = %lf", median);
  CapyVecDestruct(&u);
}

CUTEST(test028, "CapyVecGetMedian (2)") {
  CapyVec u = CapyVecCreate(9);
  double vals[9] = {2, 3, 5, 0, 1, 6, 2, 7, 4};
  loop(i, 9) u.vals[i] = vals[i];
  double median = CapyVecGetMedian(&u);
  CUTEST_ASSERT(fabs(median - 3.0) < 1e-6, "median = %lf", median);
  CapyVecDestruct(&u);
}

CUTEST(test029, "CapySolveQuadratic") {
  double const coeffs[3] = {3.0, -16.0, 16.0};
  double roots[2];
  bool hasSolution = CapySolveQuadratic(coeffs, roots);
  CUTEST_ASSERT(
    hasSolution && equald(roots[0], 0.25) && equald(roots[1], 0.75),
    "CapySolveQuadratic failed");
}

CUTEST(test030, "CapyFastExponential") {
  float const vals[9] = {-3.0, -2.0, -1.0, -0.5, 0.0, 0.5, 1.0, 2.0, 3.0};
  bool isOk = true;
  loop(i, 9) {
    float x = CapyFastExponential(vals[i]);
    isOk &= (fabs(x - expf(vals[i])) < fabs(0.0298 * expf(vals[i])));
  }
  CUTEST_ASSERT(isOk, "CapyFastExponential failed");
}

CUTEST(test031, "CapyFloryDecomposition") {
  CapyMat mat = CapyMatCreateLocal3x3;
  CapyMat vol = CapyMatCreateLocal3x3;
  CapyMat dev = CapyMatCreateLocal3x3;
  mat.vals[0] = 1.25;
  mat.vals[1] = 0.25;
  mat.vals[2] = 0.0;
  mat.vals[3] = -0.5;
  mat.vals[4] = 1.5;
  mat.vals[5] = 0.0;
  mat.vals[6] = 0.0;
  mat.vals[7] = 0.0;
  mat.vals[8] = 1.0;
  CapyFloryDecomposition(&mat, &vol, &dev);
  double checkVol[9] = {1.26, 0.0, 0.0, 0.0, 1.26, 0.0, 0.0, 0.0, 1.26};
  double checkDev[9] = {0.992, 0.198, 0.0, -0.397, 1.191, 0.0, 0.0, 0.0, 0.794};
  bool isOk = true;
  loop(i, 9) {
    isOk &= (fabs(vol.vals[i] - checkVol[i]) < 1e-3);
    isOk &= (fabs(dev.vals[i] - checkDev[i]) < 1e-3);
  }
  CUTEST_ASSERT(isOk, "CapyFloryDecomposition failed");
}
