#include "capy.h"
#ifndef FIXTURE
#define FIXTURE

// Test structure
typedef struct TestMethod {
  int a;
  CapyPad(int, 0);
  int (*get)(void);
} TestMethod;

static int TestMethodGet(void) {
  methodOf(TestMethod);
  return that->a;
}

static TestMethod instance = { .a = 1, .get = TestMethodGet };
#define TestNbArgs(...) CAPY_VA_NB_ARGS(int, __VA_ARGS__)

// Dummy structure to test containerOf
typedef struct TestContainerOf {
  uint64_t a;
  uint64_t b;
} TestContainerOf;

#endif
CUTEST(test001, "loop/loopRange") {
  int a = 0;
  loop(i, 3) a += i;
  CUTEST_ASSERT(a == 3, "loop failed %d != 3", a);
  CapyRangeInt8 range = CapyRangeInt8Create(-3, 1);
  a = 0;
  int8_t check[] = {-3, -2, -1, 0, 1};
  loopRange(i, range) {
    CUTEST_ASSERT(i == check[a], "loopRange failed %d != %d", i, check[a]);
    ++a;
  }
}

CUTEST(test002, "class method") {
  int a = $(&instance, get)();
  CUTEST_ASSERT(a == 1, "Get method failed %d", a);
}

CUTEST(test003, "VA_NB_ARGS") {
  int b, c, d;
  int n = TestNbArgs(b, c, d);
  CUTEST_ASSERT(n == 3, "VA_NB_ARGS failed %d != 3", n);
}

CUTEST(test004, "stream operations") {
  struct {FILE* fp;} volatileVars = {.fp = NULL};
  try {
    volatileVars.fp = safeFOpen("inexistingFile.txt", "r");
    CUTEST_ASSERT(false, "safeFOpen failed to detect inexisting file");
  } catch(CapyExc_StreamOpenError) {
    CUTEST_ASSERT(true, "safeFOpen detected the inexistence of the file");
    CapyCancelExc();
  } endCatch;
  CapyForwardExc();
  CUTEST_ASSERT(volatileVars.fp == NULL, "fp != NULL");
  try {
    volatileVars.fp = safeFOpen("UnitTests/TestCext/cext_test_stream.txt", "r");
    char const* check = "hello\n";
    char c = ' ';
    loop(i, 6) {
      safeFScanf(volatileVars.fp, "%c", &c);
      CUTEST_ASSERT(
        c == check[i], "safeFScanf failed [%d]%c != %c", i, c, check[i]);
    }
    safeFScanf(volatileVars.fp, "%c", &c);
    CUTEST_ASSERT(false, "safeFScanf reading after the EOF");
  } catch(CapyExc_StreamReadError) {
    CUTEST_ASSERT(true, "safeFScanf detected the end of file");
    CapyCancelExc();
  } endCatch;
  CapyForwardExc();
  if(volatileVars.fp != NULL) fclose(volatileVars.fp);
  try {
    volatileVars.fp =
      safeFOpen("UnitTests/TestCext/cext_test_stream.txt", "rb");
    char const* check = "hello\n";
    char c = ' ';
    loop(i, 6) {
      safeFRead(volatileVars.fp, 1, &c);
      CUTEST_ASSERT(
        c == check[i], "safeFRed failed [%d]%c != %c", i, c, check[i]);
    }
    safeFRead(volatileVars.fp, 1, &c);
    CUTEST_ASSERT(false, "safeFRead reading after the EOF");
  } catch(CapyExc_StreamReadError) {
    CUTEST_ASSERT(true, "safeFRead detected the end of file");
    CapyCancelExc();
  } endCatch;
  CapyForwardExc();
  if(volatileVars.fp != NULL) fclose(volatileVars.fp);
  try {
    volatileVars.fp =
      safeFOpen("UnitTests/TestCext/cext_test_stream_out.txt", "w");
    char const* msg = "hello";
    safeFPrintf(volatileVars.fp, "%s\n", msg);
  } catch(CapyExc_StreamWriteError) {
    CUTEST_ASSERT(false, "safeFPrintf failed to write");
  } endCatch;
  CapyForwardExc();
  if(volatileVars.fp != NULL) fclose(volatileVars.fp);
  int ret =
    system(
      "diff UnitTests/TestCext/cext_test_stream.txt "
      "UnitTests/TestCext/cext_test_stream_out.txt");
  CUTEST_ASSERT(ret == 0, "safeFPrintf failed to write");
  try {
    volatileVars.fp =
      safeFOpen("UnitTests/TestCext/cext_test_stream_out.txt", "wb");
    char const* msg = "hello\n";
    safeFWrite(volatileVars.fp, 6, msg);
  } catch(CapyExc_StreamWriteError) {
    CUTEST_ASSERT(false, "safeFWrite failed to write");
  } endCatch;
  CapyForwardExc();
  if(volatileVars.fp != NULL) fclose(volatileVars.fp);
  ret =
    system(
      "diff UnitTests/TestCext/cext_test_stream.txt "
      "UnitTests/TestCext/cext_test_stream_out.txt");
  CUTEST_ASSERT(ret == 0, "safeFWrite failed to write");
}

CUTEST(test005, "String operations") {
  char const* str = "The value of v is";
  int v = 100;
  char* resStr = NULL;
  safeSPrintf(&resStr, "%s %d", str, v);
  int ret = strcmp(resStr, "The value of v is 100");
  CUTEST_ASSERT(
    ret == 0, "safeSPrintf failed \"%s\" != \"The value of v is 100\"", resStr);
  free(resStr);
}

CUTEST(test006, "equal") {
  float a = 0.0;
  float b = 0.0;
  CUTEST_ASSERT(equal(a, b), "equal failed");
  CUTEST_ASSERT(equal(b, a), "equal failed");
  a = 1.2E-38f;
  b = 1.2E-38f;
  loop(i, 253) {
    b += a;
    if(fabsf(b - a) > FLT_EPSILON * 10.0) {
      CUTEST_ASSERT(equal(a, b) == false, "equal failed");
      CUTEST_ASSERT(equal(b, a) == false, "equal failed");
    } else {
      CUTEST_ASSERT(equal(a, b), "equal failed");
      CUTEST_ASSERT(equal(b, a), "equal failed");
    }
    a *= 2.0f;
    CUTEST_ASSERT(equal(a, b), "equal failed");
    CUTEST_ASSERT(equal(b, a), "equal failed");
  }
  a = -1.2E-38f;
  b = -1.2E-38f;
  loop(i, 253) {
    b += a;
    if(fabsf(b - a) > FLT_EPSILON * 10.0) {
      CUTEST_ASSERT(equal(a, b) == false, "equal failed");
      CUTEST_ASSERT(equal(b, a) == false, "equal failed");
    } else {
      CUTEST_ASSERT(equal(a, b), "equal failed");
      CUTEST_ASSERT(equal(b, a), "equal failed");
    }
    a *= 2.0f;
    CUTEST_ASSERT(equal(a, b), "equal failed");
    CUTEST_ASSERT(equal(b, a), "equal failed");
  }
  double c = 0.0;
  double d = 0.0;
  CUTEST_ASSERT(equal(c, d), "equal failed");
  CUTEST_ASSERT(equal(d, c), "equal failed");
  c = 2.3E-308;
  d = 2.3E-308;
  loop(i, 1232) {
    d += c;
    if(fabs(d - c) > DBL_EPSILON * 10.0) {
      CUTEST_ASSERT(equal(c, d) == false, "equal failed");
      CUTEST_ASSERT(equal(d, c) == false, "equal failed");
    } else {
      CUTEST_ASSERT(equal(c, d), "equal failed");
      CUTEST_ASSERT(equal(d, c), "equal failed");
    }
    c *= 2.0;
    CUTEST_ASSERT(equal(c, d), "equal failed");
    CUTEST_ASSERT(equal(d, c), "equal failed");
  }
  c = -2.3E-308;
  d = -2.3E-308;
  loop(i, 1232) {
    d += c;
    if(fabs(d - c) > DBL_EPSILON * 10.0) {
      CUTEST_ASSERT(equal(c, d) == false, "equal failed");
      CUTEST_ASSERT(equal(d, c) == false, "equal failed");
    } else {
      CUTEST_ASSERT(equal(c, d), "equal failed");
      CUTEST_ASSERT(equal(d, c), "equal failed");
    }
    c *= 2.0;
    CUTEST_ASSERT(equal(c, d), "equal failed");
    CUTEST_ASSERT(equal(d, c), "equal failed");
  }
}

CUTEST(test007, "min/max on arrays") {
  int8_t arr[3] = {1, 2, 3};
  int8_t min = min(arr, 3);
  int8_t max = max(arr, 3);
  CUTEST_ASSERT(min == 1, "min failed %d != 1", min);
  CUTEST_ASSERT(max == 3, "max failed %d != 1", max);
}

CUTEST(test008, "IsAccessibleMem, GetMemUsed") {
  CUTEST_ASSERT(
    CapyIsAccessibleMem(NULL, 0) == false, "IsAccessibleMem failed");
  CUTEST_ASSERT(
    CapyIsAccessibleMem((void*)1, 0) == false, "IsAccessibleMem failed");
  char a;
  CUTEST_ASSERT(
    CapyIsAccessibleMem(&a, 1), "IsAccessibleMem failed");
  char* b;
  safeMalloc(b, 2);
  CUTEST_ASSERT(
    CapyIsAccessibleMem(b, 2), "IsAccessibleMem failed");
  CUTEST_ASSERT(
    CapyIsAccessibleMem(b, 3), "IsAccessibleMem failed");
  CUTEST_ASSERT(
    CapyIsAccessibleMem(b, SIZE_MAX) == false, "IsAccessibleMem failed");
  long memUsed = CapyGetMemUsed();
  CUTEST_ASSERT(memUsed > 0, "GetMemUsed failed");
  free(b);
}

CUTEST(test009, "containerOf") {
  TestContainerOf test;
  uint64_t* ptrB = &(test.b);
  TestContainerOf* ptrTest = containerOf(ptrB, TestContainerOf, b);
  CUTEST_ASSERT(ptrTest == &test, "containerOf failed");
}

CUTEST(test010, "IsBigEndian") {
  CUTEST_ASSERT(
    CapyIsBigEndian() == (__BYTE_ORDER__ == __ORDER_BIG_ENDIAN__),
    "IsBigEndian failed");
}
