#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "RGB/HSV conversion") {
  #define NB_TESTS 11
  CapyColorData rgbs[NB_TESTS] = {
    (CapyColorData){.RGB = {1.0, 1.0, 1.0}},
    (CapyColorData){.RGB = {0.5, 0.5, 0.5}},
    (CapyColorData){.RGB = {0.0, 0.0, 0.0}},
    (CapyColorData){.RGB = {1.0, 0.0, 0.0}},
    (CapyColorData){.RGB = {0.0, 1.0, 0.0}},
    (CapyColorData){.RGB = {0.0, 0.0, 1.0}},
    (CapyColorData){.RGB = {0.75, 0.75, 0.0}},
    (CapyColorData){.RGB = {0.0, 0.5, 0.0}},
    (CapyColorData){.RGB = {0.5, 1.0, 1.0}},
    (CapyColorData){.RGB = {0.5, 0.5, 1.0}},
    (CapyColorData){.RGB = {0.75, 0.25, 0.75}},
  };
  CapyColorData hsvs[NB_TESTS] = {
    (CapyColorData){.HSV = {0.0, 0.0, 1.0}},
    (CapyColorData){.HSV = {0.0, 0.0, 0.5}},
    (CapyColorData){.HSV = {0.0, 0.0, 0.0}},
    (CapyColorData){.HSV = {0.0, 1.0, 1.0}},
    (CapyColorData){.HSV = {120.0/360., 1.0, 1.0}},
    (CapyColorData){.HSV = {240.0/360.0, 1.0, 1.0}},
    (CapyColorData){.HSV = {60./360., 1.0, 0.75}},
    (CapyColorData){.HSV = {120./360., 1.0, 0.5}},
    (CapyColorData){.HSV = {180./360., 0.5, 1.0}},
    (CapyColorData){.HSV = {240./360., 0.5, 1.0}},
    (CapyColorData){.HSV = {300./360., 2.0/3.0, 0.75}},
  };
  loop(i, NB_TESTS) {
    CapyColor color = CapyColorCreate(rgbs[i]);
    CapyColorData cvt = $(&color, RGB2HSV)();
    bool ret = $$(&color, &cvt, equalsHSV)(hsvs + i);
    CUTEST_ASSERT(
      ret,
      "Conversion failed for rgb %.3f,%.3f,%.3f -> hsv %.3f,%.3f,%.3f",
      color.vals.RGB[0], color.vals.RGB[1], color.vals.RGB[2],
      cvt.HSV[0], cvt.HSV[1], cvt.HSV[2]);
  }
  loop(i, NB_TESTS) {
    CapyColor color = CapyColorCreate(hsvs[i]);
    CapyColorData cvt = $(&color, HSV2RGB)();
    bool ret = $$(&color, &cvt, equalsRGB)(rgbs + i);
    CUTEST_ASSERT(
      ret,
      "Conversion failed hsv %.3f,%.3f,%.3f -> rgb %.3f,%.3f,%.3f",
      color.vals.HSV[0], color.vals.HSV[1], color.vals.HSV[2],
      cvt.RGB[0], cvt.RGB[1], cvt.RGB[2]);
  }
  int skip = 17;
  int range = 255 / skip + 1;
  loop(r, range) loop(g, range) loop(b, range) {
    CapyColorData orig = {
      .RGB = {
        ((CapyColorValue_t)r * skip) / 256.0,
        ((CapyColorValue_t)g * skip) / 256.0,
        ((CapyColorValue_t)b * skip) / 256.0
      }
    };
    CapyColor color = CapyColorCreate(orig);
    color.vals = $(&color, RGB2HSV)();
    CUTEST_ASSERT(
      color.vals.HSV[0] >= 0.0 && color.vals.HSV[0] <= 1.0 &&
      color.vals.HSV[1] >= 0.0 && color.vals.HSV[1] <= 1.0 &&
      color.vals.HSV[2] >= 0.0 && color.vals.HSV[2] <= 1.0,
      "Conversion rgb->hsv not in [0, 1] for rgb %.3f,%.3f,%.3f",
      orig.RGB[0], orig.RGB[1], orig.RGB[2]);
    color.vals = $(&color, HSV2RGB)();
    bool ret = $(&color, equalsRGB)(&orig);
    CUTEST_ASSERT(
      ret,
      "Conversion rgb->hsv->rgb failed for rgb %.3f,%.3f,%.3f",
      orig.RGB[0], orig.RGB[1], orig.RGB[2]);
  }
}

CUTEST(test002, "RGB/XYZ conversion") {
  #define NB_TESTS 11
  CapyColorData rgbs[NB_TESTS] = {
    (CapyColorData){.RGB = {1.0, 1.0, 1.0}},
    (CapyColorData){.RGB = {0.5, 0.5, 0.5}},
    (CapyColorData){.RGB = {0.0, 0.0, 0.0}},
    (CapyColorData){.RGB = {1.0, 0.0, 0.0}},
    (CapyColorData){.RGB = {0.0, 1.0, 0.0}},
    (CapyColorData){.RGB = {0.0, 0.0, 1.0}},
    (CapyColorData){.RGB = {0.75, 0.75, 0.0}},
    (CapyColorData){.RGB = {0.0, 0.5, 0.0}},
    (CapyColorData){.RGB = {0.5, 1.0, 1.0}},
    (CapyColorData){.RGB = {0.5, 0.5, 1.0}},
    (CapyColorData){.RGB = {0.75, 0.25, 0.75}},
  };
  CapyColorData xyzs[NB_TESTS] = {
    (CapyColorData){.XYZ = {0.950449, 1.000000, 1.088917}},
    (CapyColorData){.XYZ = {0.203435, 0.214041, 0.233073}},
    (CapyColorData){.XYZ = {0.000000, 0.000000, 0.000000}},
    (CapyColorData){.XYZ = {0.412411, 0.212649, 0.019332}},
    (CapyColorData){.XYZ = {0.357585, 0.715169, 0.119195}},
    (CapyColorData){.XYZ = {0.180454, 0.072182, 0.950390}},
    (CapyColorData){.XYZ = {0.402339, 0.484805, 0.072383}},
    (CapyColorData){.XYZ = {0.076538, 0.153076, 0.025513}},
    (CapyColorData){.XYZ = {0.626311, 0.832866, 1.073723}},
    (CapyColorData){.XYZ = {0.345265, 0.270773, 0.980040}},
    (CapyColorData){.XYZ = {0.327977, 0.185215, 0.512765}},
  };
  loop(i, NB_TESTS) {
    CapyColor color = CapyColorCreate(rgbs[i]);
    CapyColorData cvt = $(&color, RGB2XYZ)();
    loop(j, 3) {
      double diff = fabs(cvt.XYZ[j] - xyzs[i].XYZ[j]);
      CUTEST_ASSERT(
        diff < 0.01,
        "Conversion rgb->xyz failed rgb %.3f,%.3f,%.3f -> xyz %.3f,%.3f,%.3f",
        color.vals.RGB[0], color.vals.RGB[1], color.vals.RGB[2],
        cvt.XYZ[0], cvt.XYZ[1], cvt.XYZ[2]);
    }
  }
  loop(i, NB_TESTS) {
    CapyColor color = CapyColorCreate(xyzs[i]);
    CapyColorData cvt = $(&color, XYZ2RGB)();
    loop(j, 3) {
      double diff = fabs(cvt.RGB[j] - rgbs[i].RGB[j]);
      CUTEST_ASSERT(
        diff < 0.01,
        "Conversion xyz->rgb failed xyz %.3f,%.3f,%.3f -> rgb %.3f,%.3f,%.3f",
        color.vals.XYZ[0], color.vals.XYZ[1], color.vals.XYZ[2],
        cvt.RGB[0], cvt.RGB[1], cvt.RGB[2]);
    }
  }
}

CUTEST(test003, "LAB/XYZ conversion") {
  #define NB_TESTS 11
  CapyColorData xyzs[NB_TESTS] = {
    (CapyColorData){.XYZ = {0.950449, 1.000000, 1.088917}},
    (CapyColorData){.XYZ = {0.203435, 0.214041, 0.233073}},
    (CapyColorData){.XYZ = {0.000000, 0.000000, 0.000000}},
    (CapyColorData){.XYZ = {0.412411, 0.212649, 0.019332}},
    (CapyColorData){.XYZ = {0.357585, 0.715169, 0.119195}},
    (CapyColorData){.XYZ = {0.180454, 0.072182, 0.950390}},
    (CapyColorData){.XYZ = {0.402339, 0.484805, 0.072383}},
    (CapyColorData){.XYZ = {0.076538, 0.153076, 0.025513}},
    (CapyColorData){.XYZ = {0.626311, 0.832866, 1.073723}},
    (CapyColorData){.XYZ = {0.345265, 0.270773, 0.980040}},
    (CapyColorData){.XYZ = {0.327977, 0.185215, 0.512765}},
  };
  CapyColorData labs[NB_TESTS] = {
    (CapyColorData){.LAB = {100.000, -0.007, -0.005}},
    (CapyColorData){.LAB = {53.389, -0.004, -0.003}},
    (CapyColorData){.LAB = {0.000, 0.000, 0.000}},
    (CapyColorData){.LAB = {53.238, 80.087, 67.201}},
    (CapyColorData){.LAB = {87.736, -86.186, 83.180}},
    (CapyColorData){.LAB = {32.299, 79.188, -107.863}},
    (CapyColorData){.LAB = {75.127, -17.368, 76.096}},
    (CapyColorData){.LAB = {46.053, -51.555, 49.757}},
    (CapyColorData){.LAB = {93.140, -35.335, -10.898}},
    (CapyColorData){.LAB = {59.046, 33.283, -63.714}},
    (CapyColorData){.LAB = {50.123, 65.688, -41.597}},
  };
  loop(i, NB_TESTS) {
    CapyColor color = CapyColorCreate(xyzs[i]);
    CapyColorData cvt = $(&color, XYZ2LAB)();
    loop(j, 3) {
      double diff = fabs(cvt.LAB[j] - labs[i].LAB[j]);
      CUTEST_ASSERT(
        diff < 0.01,
        "Conversion xyz->lab failed xyz %.3f,%.3f,%.3f -> lab %.3f,%.3f,%.3f",
        color.vals.XYZ[0], color.vals.XYZ[1], color.vals.XYZ[2],
        cvt.LAB[0], cvt.LAB[1], cvt.LAB[2]);
    }
  }
  loop(i, NB_TESTS) {
    CapyColor color = CapyColorCreate(labs[i]);
    CapyColorData cvt = $(&color, LAB2XYZ)();
    loop(j, 3) {
      double diff = fabs(cvt.XYZ[j] - xyzs[i].XYZ[j]);
      CUTEST_ASSERT(
        diff < 0.01,
        "Conversion lab->xyz failed lab %.3f,%.3f,%.3f -> xyz %.3f,%.3f,%.3f",
        color.vals.LAB[0], color.vals.LAB[1], color.vals.LAB[2],
        cvt.XYZ[0], cvt.XYZ[1], cvt.XYZ[2]);
    }
  }
}

CUTEST(test004, "Oklab/XYZ conversion") {
  #define NB_TESTS_4 4
  CapyColorData xyzs[NB_TESTS_4] = {
    (CapyColorData){.XYZ = {0.950, 1.000, 1.089}},
    (CapyColorData){.XYZ = {1.000, 0.000, 0.000}},
    (CapyColorData){.XYZ = {0.000, 1.000, 0.000}},
    (CapyColorData){.XYZ = {0.000, 0.000, 1.000}},
  };
  CapyColorData oklabs[NB_TESTS_4] = {
    (CapyColorData){.Oklab = {1.000, +0.000, +0.000}},
    (CapyColorData){.Oklab = {0.450, +1.236, -0.019}},
    (CapyColorData){.Oklab = {0.922, -0.671, +0.263}},
    (CapyColorData){.Oklab = {0.153, -1.415, -0.449}},
  };
  loop(i, NB_TESTS_4) {
    CapyColor color = CapyColorCreate(xyzs[i]);
    CapyColorData cvt = $(&color, XYZ2Oklab)();
    loop(j, 3) {
      double diff = fabs(cvt.Oklab[j] - oklabs[i].LAB[j]);
      CUTEST_ASSERT(
        diff < 0.01,
        "Conversion xyz->oklab failed xyz %.3f,%.3f,%.3f -> "
        "oklab %.3f,%.3f,%.3f",
        color.vals.XYZ[0], color.vals.XYZ[1], color.vals.XYZ[2],
        cvt.Oklab[0], cvt.Oklab[1], cvt.Oklab[2]);
    }
  }
  loop(i, NB_TESTS_4) {
    CapyColor color = CapyColorCreate(oklabs[i]);
    CapyColorData cvt = $(&color, Oklab2XYZ)();
    loop(j, 3) {
      double diff = fabs(cvt.XYZ[j] - xyzs[i].XYZ[j]);
      CUTEST_ASSERT(
        diff < 0.01,
        "Conversion oklab->xyz failed oklab %.3f,%.3f,%.3f -> "
        "xyz %.3f,%.3f,%.3f",
        color.vals.Oklab[0], color.vals.Oklab[1], color.vals.Oklab[2],
        cvt.XYZ[0], cvt.XYZ[1], cvt.XYZ[2]);
    }
  }
}
