#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
static void ObjectiveFun(
   double const* const in,
         double* const out) {
  double a = 20.0;
  double b = 0.2;
  double c = 2.0 * 3.1415926;
  double d = 2.0;
  double x = (in[0] - 0.5) * 40.0;
  double y = (in[1] - 0.5) * 40.0;
  double v =
    -a * exp(-b * sqrt(1.0 / d * (x * x + y * y))) -
    exp(1.0 / d * (cos(c * x) + cos(c * y))) + a + exp(1.0);
  out[0] = -v;
}

static void ObjectiveFunSlow(
  double const* const in,
        double* const out) {
  ObjectiveFun(in, out);
  CapySleepMs(50);
}

static void ConstraintFun(
  double const* const in,
        double* const out) {
  *out = in[0] + in[1] - 1.2;
}

// Dummy fitness class
typedef struct {
  struct CapyMathFunDef;
  void (*destructCapyMathFun)(void);
} Fitness;

static void FitnessDestruct(void) {
  Fitness* that = (Fitness*)capyThat;
  $(that, destructCapyMathFun)();
}

static Fitness FitnessCreate(void) {
  Fitness that;
  CapyInherits(that, CapyMathFun, (2, 1));
  that.eval = ObjectiveFunSlow;
  that.destruct = FitnessDestruct;
  return that;
}

// Dummy constraint class
typedef struct {
  struct CapyMathFunDef;
  void (*destructCapyMathFun)(void);
} Constraint;

static void ConstraintDestruct(void) {
  Constraint* that = (Constraint*)capyThat;
  $(that, destructCapyMathFun)();
}

static Constraint ConstraintCreate(void) {
  Constraint that;
  CapyInherits(that, CapyMathFun, (2, 1));
  that.eval = ConstraintFun;
  that.destruct = ConstraintDestruct;
  return that;
}

#endif
CUTEST(test001, "run, stop, getBestDna") {
  Fitness fitness = FitnessCreate();
  CapyDiffEvo* optimiser = CapyDiffEvoAlloc();
  optimiser->nbAgent = 20;
  optimiser->seed = 0;
  CUTEST_ASSERT(optimiser->running == false, "unexpected value");
  $(optimiser, run)((CapyMathFun*)&fitness, NULL);
  CUTEST_ASSERT(optimiser->running, "unexpected value");
  CUTEST_ASSERT(
    fabs(optimiser->initialFitness + 19.781001) < 0.001,
    "unexpected initial fitness %lf", optimiser->initialFitness);
  double curFitness;
  double curConstraint;
  double dna[2];
  do {
    $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
    CapySleepMs(2000);
  } while(curFitness < -0.001 && optimiser->nbIter < 100);
  $(optimiser, stop)();
  $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
  CUTEST_ASSERT(optimiser->running == false, "unexpected value");
  CUTEST_ASSERT(fabs(dna[0] - 0.5) < 0.001, "run failed %lf", dna[0]);
  CUTEST_ASSERT(fabs(dna[1] - 0.5) < 0.001, "run failed %lf", dna[1]);
  CUTEST_ASSERT(
    fabs(curFitness - -0.000999) < 0.001, "run failed %lf", curFitness);
  CUTEST_ASSERT(
    fabs(curConstraint - 0.0) < 0.001, "run failed %lf", curFitness);
  CapyDiffEvoFree(&optimiser);
  CUTEST_ASSERT(optimiser == NULL, "optimiser not reset");
  $(&fitness, destruct)();
}

CUTEST(test002, "with seed") {
  Fitness fitness = FitnessCreate();
  CapyDiffEvo* optimiser = CapyDiffEvoAlloc();
  optimiser->nbAgent = 20;
  optimiser->seed = 0;
  double seedDna[] = {0.49, 0.51};
  optimiser->seedDna = seedDna;
  optimiser->nbIterReset = 0;
  $(optimiser, run)((CapyMathFun*)&fitness, NULL);
  double curFitness;
  double curConstraint;
  double dna[2];
  do {
    $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
    CapySleepMs(2000);
  } while(curFitness < -0.001 && optimiser->nbIter < 100);
  $(optimiser, stop)();
  $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
  CUTEST_ASSERT(fabs(dna[0] - 0.5) < 0.001, "run failed %lf", dna[0]);
  CUTEST_ASSERT(fabs(dna[1] - 0.5) < 0.001, "run failed %lf", dna[1]);
  CUTEST_ASSERT(
    fabs(curFitness - -0.000458) < 0.001, "run failed %lf", curFitness);
  CUTEST_ASSERT(
    fabs(curConstraint - 0.0) < 0.001, "run failed %lf", curConstraint);
  CapyDiffEvoFree(&optimiser);
  $(&fitness, destruct)();
}

CUTEST(test003, "with domain") {
  Fitness fitness = FitnessCreate();
  fitness.domains[1].min = 0.6;
  CapyDiffEvo* optimiser = CapyDiffEvoAlloc();
  optimiser->nbAgent = 20;
  optimiser->seed = 0;
  $(optimiser, run)((CapyMathFun*)&fitness, NULL);
  double curFitness;
  double curConstraint;
  double dna[2];
  do {
    $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
    CapySleepMs(2000);
  } while(optimiser->nbIter < 40);
  $(optimiser, stop)();
  $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
  CUTEST_ASSERT(fabs(dna[0] - 0.5) < 0.001, "run failed %lf", dna[0]);
  CUTEST_ASSERT(fabs(dna[1] - 0.6) < 0.001, "run failed %lf", dna[1]);
  CUTEST_ASSERT(
    fabs(curFitness - -8.6) < 0.1, "run failed %lf", curFitness);
  CUTEST_ASSERT(
    fabs(curConstraint - 0.0) < 0.001, "run failed %lf", curConstraint);
  CapyDiffEvoFree(&optimiser);
  $(&fitness, destruct)();
}

CUTEST(test004, "with constraint") {
  Fitness fitness = FitnessCreate();
  Constraint constraint = ConstraintCreate();
  CapyDiffEvo* optimiser = CapyDiffEvoAlloc();
  optimiser->nbAgent = 100;
  optimiser->seed = 0;
  $(optimiser, run)((CapyMathFun*)&fitness, (CapyMathFun*)&constraint);
  double curFitness;
  double curConstraint;
  double dna[2];
  do {
    $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
    CapySleepMs(4000);
  } while(optimiser->nbIter < 15);
  $(optimiser, stop)();
  $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
  CUTEST_ASSERT(fabs(dna[0] - 0.599464) < 0.001, "run failed %lf", dna[0]);
  CUTEST_ASSERT(fabs(dna[1] - 0.600824) < 0.001, "run failed %lf", dna[1]);
  CUTEST_ASSERT(
    fabs(curFitness - -11.064961) < 0.001, "run failed %lf", curFitness);
  CUTEST_ASSERT(
    fabs(curConstraint - 0.000288) < 0.001, "run failed %lf", curConstraint);
  CapyDiffEvoFree(&optimiser);
  $(&fitness, destruct)();
  $(&constraint, destruct)();
}

CUTEST(test005, "init mode as randomised seed") {
  Fitness fitness = FitnessCreate();
  CapyDiffEvo* optimiser = CapyDiffEvoAlloc();
  optimiser->nbAgent = 20;
  optimiser->seed = 0;
  optimiser->initMode = capyDiffEvoInitMode_randomisedSeed;
  double seedDna[] = {0.49, 0.51};
  optimiser->seedDna = seedDna;
  optimiser->nbIterReset = 0;
  $(optimiser, run)((CapyMathFun*)&fitness, NULL);
  double curFitness;
  double curConstraint;
  double dna[2];
  do {
    $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
    CapySleepMs(200);
  } while(curFitness < -0.001 && optimiser->nbIter < 1000);
  $(optimiser, stop)();
  $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
  CUTEST_ASSERT(fabs(dna[0] - 0.5) < 0.001, "run failed %lf", dna[0]);
  CUTEST_ASSERT(fabs(dna[1] - 0.5) < 0.001, "run failed %lf", dna[1]);
  CUTEST_ASSERT(
    fabs(curFitness - -0.000458) < 0.001, "run failed %lf", curFitness);
  CUTEST_ASSERT(
    fabs(curConstraint - 0.0) < 0.001, "run failed %lf", curConstraint);
  CapyDiffEvoFree(&optimiser);
  $(&fitness, destruct)();
}

CUTEST(test006, "sequential mode") {
  Fitness fitness = FitnessCreate();
  CapyDiffEvo* optimiser = CapyDiffEvoAlloc();
  optimiser->nbAgent = 20;
  optimiser->seed = 0;
  optimiser->nbIterReset = 0;
  optimiser->mode = capyDiffEvoMode_sequential;
  optimiser->strideSeq = 2;
  $(optimiser, run)((CapyMathFun*)&fitness, NULL);
  double curFitness;
  double curConstraint;
  double dna[2];
  do {
    $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
    CapySleepMs(2000);
  } while(curFitness < -0.001 && optimiser->nbIter < 100);
  $(optimiser, stop)();
  $(optimiser, getBestDna)(dna, &curFitness, &curConstraint);
  CUTEST_ASSERT(fabs(dna[0] - 0.5) < 0.001, "run failed %lf", dna[0]);
  CUTEST_ASSERT(fabs(dna[1] - 0.5) < 0.001, "run failed %lf", dna[1]);
  CUTEST_ASSERT(
    fabs(curFitness - -0.000458) < 0.001, "run failed %lf", curFitness);
  CUTEST_ASSERT(
    fabs(curConstraint - 0.0) < 0.001, "run failed %lf", curConstraint);
  CapyDiffEvoFree(&optimiser);
  $(&fitness, destruct)();
}
