#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
static void Fun1(
  double const* const in,
        double* const out) {
  out[0] = sin(in[0] * 2.0 * M_PI);
}

static void Fun2(
  double const* const in,
        double* const out) {
  out[0] = 1.0 + sin(2.0 * in[0] * 2.0 * M_PI);
}

static void Fun3(
  double const* const in,
        double* const out) {
  int a = (int)(in[0] * 2.0);
  if((a & 1) == 0) out[0] = 0.0; else out[0] = 1.0;
}

#endif
CUTEST(test001, "DFT on Polynomial1D (1)") {
  CapyVec coeffs = {.dim = 4, .vals = (double[4]){1.0, 2.0, 3.0, 4.0}};
  CapyPolynomial1D poly = CapyPolynomial1DCreate(&coeffs);
  CapyDFT* dft = CapyDFTAlloc();
  CapyDFTCoeffs values = $(dft, fftPolyFromCoeffToVal)(&poly);
  double check[4][2] = {{10.0, 0.0}, {2.0, -2.0}, {2.0, 0.0}, {2.0, 2.0}};
  loop(i, 4) {
    CUTEST_ASSERT(
      fabs(creal(values.vals[i]) - check[i][0]) < 1e-9 ||
      fabs(cimag(values.vals[i]) - check[i][1]) < 1e-9,
      "Unexpected value [%d]=(%lf,%lf) != (%lf,%lf)",
      i, creal(values.vals[i]), cimag(values.vals[i]),
      check[i][0], check[i][1]);
  }
  CapyPolynomial1D* polyB = $(dft, fftPolyFromValToCoeff)(&values);
  loop(i, 4) {
    CUTEST_ASSERT(
      fabs(polyB->coeffs.vals[i] - poly.coeffs.vals[i]) < 1e-9,
      "Unexpected value [%d]=%lf != %lf",
      i, polyB->coeffs.vals[i], poly.coeffs.vals[i]);
  }
  CapyDFTFree(&dft);
  $(&poly, destruct)();
  $(&values, destruct)();
  CapyPolynomial1DFree(&polyB);
}

CUTEST(test002, "DFT on Polynomial1D (2)") {
  CapyVec coeffs =
    {.dim = 8, .vals = (double[8]){1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0}};
  CapyPolynomial1D poly = CapyPolynomial1DCreate(&coeffs);
  CapyDFT* dft = CapyDFTAlloc();
  CapyDFTCoeffs values = $(dft, fftPolyFromCoeffToVal)(&poly);
  double check[8][2] = {
    {10.0, 0.0}, {-4.0, -9.656854}, {-4.0, -4.0}, {-4.0, -1.656854},
    {-4.0, 1.656854}, {-4.0, 4.0}, {-4.0, 9.656854}, {-4.0, 9.656854}
  };
  loop(i, 8) {
    CUTEST_ASSERT(
      fabs(creal(values.vals[i]) - check[i][0]) < 1e-9 ||
      fabs(cimag(values.vals[i]) - check[i][1]) < 1e-9,
      "Unexpected value [%d]=(%lf,%lf) != (%lf,%lf)",
      i, creal(values.vals[i]), cimag(values.vals[i]),
      check[i][0], check[i][1]);
  }
  CapyPolynomial1D* polyB = $(dft, fftPolyFromValToCoeff)(&values);
  loop(i, 8) {
    CUTEST_ASSERT(
      fabs(polyB->coeffs.vals[i] - poly.coeffs.vals[i]) < 1e-9,
      "Unexpected value [%d]=%lf != %lf",
      i, polyB->coeffs.vals[i], poly.coeffs.vals[i]);
  }
  CapyDFTFree(&dft);
  $(&poly, destruct)();
  $(&values, destruct)();
  CapyPolynomial1DFree(&polyB);
}

CUTEST(test003, "DFT on function (1)") {
  CapyMathFun fun = CapyMathFunCreate(1, 1);
  fun.eval = Fun1;
  CapyDFT* dft = CapyDFTAlloc();
  size_t nbSample = 8;
  CapyRangeDouble range = {.min = 0.0, .max = 1.0};
  CapyDFTCoeffs coeffs = $(dft, fftFun)(&fun, &range, nbSample);
  double checkA[8][2] = {
    {0.0, 0.0}, {0.0, -4.0}, {0.0, 0.0}, {0.0, 0.0},
    {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 4.0}
  };
  loop(i, 8) {
    CUTEST_ASSERT(
      fabs(creal(coeffs.vals[i]) - checkA[i][0]) < 1e-9 ||
      fabs(cimag(coeffs.vals[i]) - checkA[i][1]) < 1e-9,
      "Unexpected value [%d]=(%lf,%lf) != (%lf,%lf)",
      i, creal(coeffs.vals[i]), cimag(coeffs.vals[i]),
      checkA[i][0], checkA[i][1]);
  }
  CapyVec amps = $(&coeffs, getAmpFreqBins)(true);
  double checkB[4] = {0.0, 1.0, 0.0, 0.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      fabs(amps.vals[i] - checkB[i]) < 1e-9,
      "Unexpected value [%d]=%lf != %lf",
      i, amps.vals[i], checkB[i]);
  }
  uint8_t n = 16;
  loop(i, n) {
    double in[1];
    CapyRangeDouble rangeFrom = {.min = 0, .max = n - 1};
    in[0] = CapyLerp(i, &rangeFrom, &range);
    double out[1];
    $(&coeffs, eval)(in, out);
    double outCheck[1];
    $(&fun, eval)(in, outCheck);
    CUTEST_ASSERT(
      fabs(out[0] - outCheck[0]) < 1e-9,
      "Unexpected value [%d]=%lf != %lf",
      i, out[0], outCheck[0]);
  }
  CapyDFTFree(&dft);
  $(&fun, destruct)();
  $(&coeffs, destruct)();
  CapyVecDestruct(&amps);
}

CUTEST(test004, "DFT on function (2)") {
  CapyMathFun fun = CapyMathFunCreate(1, 1);
  fun.eval = Fun2;
  CapyDFT* dft = CapyDFTAlloc();
  size_t nbSample = 8;
  CapyRangeDouble range = {.min = 0.0, .max = 1.0};
  CapyDFTCoeffs coeffs = $(dft, fftFun)(&fun, &range, nbSample);
  double checkA[8][2] = {
    {0.0, 0.0}, {0.0, -4.0}, {0.0, 0.0}, {0.0, 0.0},
    {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 4.0}
  };
  loop(i, 8) {
    CUTEST_ASSERT(
      fabs(creal(coeffs.vals[i]) - checkA[i][0]) < 1e-9 ||
      fabs(cimag(coeffs.vals[i]) - checkA[i][1]) < 1e-9,
      "Unexpected value [%d]=(%lf,%lf) != (%lf,%lf)",
      i, creal(coeffs.vals[i]), cimag(coeffs.vals[i]),
      checkA[i][0], checkA[i][1]);
  }
  CapyVec amps = $(&coeffs, getAmpFreqBins)(true);
  double checkB[4] = {1.0, 0.0, 1.0, 0.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      fabs(amps.vals[i] - checkB[i]) < 1e-9,
      "Unexpected value [%d]=%lf != %lf",
      i, amps.vals[i], checkB[i]);
  }
  uint8_t n = 16;
  loop(i, n) {
    double in[1];
    CapyRangeDouble rangeFrom = {.min = 0, .max = n - 1};
    in[0] = CapyLerp(i, &rangeFrom, &range);
    double out[1];
    $(&coeffs, eval)(in, out);
    double outCheck[1];
    $(&fun, eval)(in, outCheck);
    CUTEST_ASSERT(
      fabs(out[0] - outCheck[0]) < 1e-9,
      "Unexpected value [%d]=%lf != %lf",
      i, out[0], outCheck[0]);
  }
  CapyDFTFree(&dft);
  $(&fun, destruct)();
  $(&coeffs, destruct)();
  CapyVecDestruct(&amps);
}

CUTEST(test005, "DFT on function (3)") {
  CapyMathFun fun = CapyMathFunCreate(1, 1);
  fun.eval = Fun3;
  CapyDFT* dft = CapyDFTAlloc();
  size_t nbSample = 128;
  CapyRangeDouble range = {.min = 1.0, .max = 5.0};
  CapyDFTCoeffs coeffs = $(dft, fftFun)(&fun, &range, nbSample);
  CapyVec amps = $(&coeffs, getAmpFreqBins)(true);
  double check[64] = {
    0.500000, 0.000000, 0.000000, 0.000000, 0.637644, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.215306, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.132585,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.098519, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.080853, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.070868, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.065312, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.062802, 0.000000, 0.000000,
    0.000000
  };
  loop(i, nbSample / 2) {
    CUTEST_ASSERT(
      fabs(amps.vals[i] - check[i]) < 1e-6,
      "Unexpected value [%lu]=%lf != %lf",
      i, amps.vals[i], check[i]);
  }
  double checkB[100] = {
    0.000000, 0.067876, -0.051706, 0.015155, 0.017766, -0.031548, 0.021490,
    0.005146, -0.032209, 0.040724, -0.013789, -0.072829, 0.517652, 1.063029,
    1.019336, 0.957947, 1.030314, 0.997938, 0.976471, 1.031109, 0.985092,
    0.981221, 1.053266, 0.936422, 0.978202, -0.135276, 0.069107, -0.013927,
    -0.023003, 0.033622, -0.019412, -0.007861, 0.030703, -0.032984, 0.006070,
    0.050061, -0.141177, 0.867603, 0.962938, 1.054531, 0.967474, 0.998885,
    1.026038, 0.970295, 1.011038, 1.019390, 0.956747, 1.039622, 1.019436,
    0.668437, -0.110798, 0.012389, 0.030664, -0.037020, 0.017757, 0.010668,
    -0.030165, 0.027836, -0.001267, -0.040226, 0.078020, -0.076678, 1.090143,
    0.921594, 1.036654, 1.005031, 0.970484, 1.029107, 0.992163, 0.979657,
    1.037275, 0.972557, 0.981358, 1.118002, 0.297929, -0.009594, -0.043245,
    0.042489, -0.016380, -0.013848, 0.030445, -0.024192, -0.002236, 0.035077,
    -0.053493, 0.029306, 0.161281, 1.139110, 0.955367, 0.989507, 1.034502,
    0.970724, 1.004943, 1.021689, 0.966500, 1.020017, 1.018506, 0.927805,
    1.130155, 0.000000
  };
  uint8_t n = 100;
  loop(i, n) {
    double in[1];
    CapyRangeDouble rangeFrom = {.min = 0, .max = n - 1};
    in[0] = CapyLerp(i, &rangeFrom, &range);
    double out[1];
    $(&coeffs, eval)(in, out);
    CUTEST_ASSERT(
      fabs(out[0] - checkB[i]) < 1e-6,
      "Unexpected value [%d]=%lf != %lf",
      i, out[0], checkB[i]);
  }
  CapyDFTFree(&dft);
  $(&fun, destruct)();
  $(&coeffs, destruct)();
  CapyVecDestruct(&amps);
}

CUTEST(test006, "FFT2D of sinusoid (1)") {
  CapyImgDims dims = {.width = 128, .height = 128};
  CapyImg* input = CapyImgAlloc(capyImgMode_rgb, dims);
  double freq = 2.0;
  forEach(pixel, input->iter) {
    loop(i, 3) {
      pixel.color->rgb[i] =
        0.5 + 0.5 * cos(freq * 2.0 * M_PI * pixel.pos.x / (double)(dims.width));
    }
  }
  $(input, saveToPath)("./UnitTests/TestFft/sinusoid01.png");
  CapyDFT2D* dft = CapyDFT2DAlloc();
  CapyDFT2DCoeffs coeffs = $(dft, fftImage)(input);
  bool center = true;
  bool logScale = true;
  CapyImg* output = $(&coeffs, toAmplitudeImg)(center, logScale);
  $(output, saveToPath)("./UnitTests/TestFft/amplitudeSinusoid01.png");
  CapyImg* check =
    CapyImgLoadFromPath("./UnitTests/TestFft/checkAmplitudeSinusoid01.png");
  CUTEST_ASSERT($(check, isSame)(output), "amplitude differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  output = $(&coeffs, toPhaseImg)(center);
  $(output, saveToPath)("./UnitTests/TestFft/phaseSinusoid01.png");
  check = CapyImgLoadFromPath("./UnitTests/TestFft/checkPhaseSinusoid01.png");
  CUTEST_ASSERT($(check, isSame)(output), "phase differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  CapyDFT2DFree(&dft);
  $(&coeffs, destruct)();
  CapyImgFree(&input);
}

CUTEST(test007, "FFT2D of sinusoid (2)") {
  CapyImgDims dims = {.width = 128, .height = 128};
  CapyImg* input = CapyImgAlloc(capyImgMode_rgb, dims);
  double freq = 16.0;
  forEach(pixel, input->iter) {
    loop(i, 3) {
      pixel.color->rgb[i] =
        0.5 + 0.5 *
        cos(freq * 2.0 * M_PI * pixel.pos.y / (double)(dims.height));
    }
  }
  $(input, saveToPath)("./UnitTests/TestFft/sinusoid02.png");
  CapyDFT2D* dft = CapyDFT2DAlloc();
  CapyDFT2DCoeffs coeffs = $(dft, fftImage)(input);
  bool center = true;
  bool logScale = true;
  CapyImg* output = $(&coeffs, toAmplitudeImg)(center, logScale);
  $(output, saveToPath)("./UnitTests/TestFft/amplitudeSinusoid02.png");
  CapyImg* check =
    CapyImgLoadFromPath("./UnitTests/TestFft/checkAmplitudeSinusoid02.png");
  CUTEST_ASSERT($(check, isSame)(output), "amplitude differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  output = $(&coeffs, toPhaseImg)(center);
  $(output, saveToPath)("./UnitTests/TestFft/phaseSinusoid02.png");
  check = CapyImgLoadFromPath("./UnitTests/TestFft/checkPhaseSinusoid02.png");
  CUTEST_ASSERT($(check, isSame)(output), "phase differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  CapyDFT2DFree(&dft);
  $(&coeffs, destruct)();
  CapyImgFree(&input);
}

CUTEST(test008, "FFT2D of sinusoid (3)") {
  CapyImgDims dims = {.width = 128, .height = 128};
  CapyImg* input = CapyImgAlloc(capyImgMode_rgb, dims);
  double freq[2] = {16.0, 8.0};
  forEach(pixel, input->iter) {
    loop(i, 3) {
      pixel.color->rgb[i] =
        0.5 + 0.5 * cos(
          2.0 * M_PI * (
            freq[0] * pixel.pos.x / (double)(dims.width) +
            freq[1] * pixel.pos.y / (double)(dims.height)
          )
        );
    }
  }
  $(input, saveToPath)("./UnitTests/TestFft/sinusoid03.png");
  CapyDFT2D* dft = CapyDFT2DAlloc();
  CapyDFT2DCoeffs coeffs = $(dft, fftImage)(input);
  bool center = true;
  bool logScale = true;
  CapyImg* output = $(&coeffs, toAmplitudeImg)(center, logScale);
  $(output, saveToPath)("./UnitTests/TestFft/amplitudeSinusoid03.png");
  CapyImg* check =
    CapyImgLoadFromPath("./UnitTests/TestFft/checkAmplitudeSinusoid03.png");
  CUTEST_ASSERT($(check, isSame)(output), "amplitude differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  output = $(&coeffs, toPhaseImg)(center);
  $(output, saveToPath)("./UnitTests/TestFft/phaseSinusoid03.png");
  check = CapyImgLoadFromPath("./UnitTests/TestFft/checkPhaseSinusoid03.png");
  CUTEST_ASSERT($(check, isSame)(output), "phase differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  CapyDFT2DFree(&dft);
  $(&coeffs, destruct)();
  CapyImgFree(&input);
}

CUTEST(test009, "Periodogram of Perlin noise") {
  CapyImg* input = CapyImgLoadFromPath("./UnitTests/TestFft/perlin.png");
  CapyDFT2D* dft = CapyDFT2DAlloc();
  CapyDFT2DCoeffs coeffs = $(dft, fftImage)(input);
  bool center = true;
  bool logScale = true;
  CapyImg* output = $(&coeffs, toPeriodogramImg)(center, logScale);
  $(output, saveToPath)("./UnitTests/TestFft/periodogramPerlin.png");
  CapyImg* check =
    CapyImgLoadFromPath("./UnitTests/TestFft/checkPeriodogramPerlin.png");
  CUTEST_ASSERT($(check, isSame)(output), "periodogram differs");
  CapyImgFree(&check);
  CapyImgFree(&output);
  CapyDFT2DFree(&dft);
  $(&coeffs, destruct)();
  CapyImgFree(&input);
}
