#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "Img operations") {
  CapyImg* img = CapyImgAlloc_RGB_600x800();
  CUTEST_ASSERT(
    img->dims.width == 600 && img->dims.height == 800,
    "unexpected dimension width %d height %d",
    img->dims.width, img->dims.height);
  CUTEST_ASSERT(img->mode == capyImgMode_rgb, "unexpected mode %d", img->mode);
  uint64_t nbPixels = $(img, getNbPixels)();
  CUTEST_ASSERT(nbPixels == 480000, "unexpected nbPixels %lu", nbPixels);
  CUTEST_ASSERT(
    img->colorSpace == capyColorSpace_sRGB,
    "unexpected colorSpace %d", img->colorSpace);
  CapyImgFree(&img);
  CUTEST_ASSERT(img == NULL, "img not reset");
  img = CapyImgLoadFromPath("UnitTests/TestImage/test001.png");
  CUTEST_ASSERT(equal(img->gamma, 2.2), "unexpected gamma %lf", img->gamma);
  CUTEST_ASSERT(
    img->dims.width == 521 && img->dims.height == 757,
    "unexpected dimension width %d height %d",
    img->dims.width, img->dims.height);
  CUTEST_ASSERT(img->mode == capyImgMode_rgba, "unexpected mode %d", img->mode);
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(img->pixels[0].RGBA[i], 1.0),
      "unexpected pixel value %lf", img->pixels[0].RGBA[i]);
  }
  loop(i, 4) {
    double pixels[] = {126.0 / 255.0, 127.0 / 255.0, 129.0 / 255.0, 1.0};
    CUTEST_ASSERT(
      equal(img->pixels[521 * 757 - 1].RGBA[i], pixels[i]),
      "unexpected pixel value [%d]=%lf != %lf",
      i, img->pixels[521 * 757 - 1].RGBA[i], pixels[i]);
  }
  $(img, saveToPath)("UnitTests/TestImage/test001_.png");
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("UnitTests/TestImage/test001_.png", "rb");
  CapyImg* png_ = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(img, isSame)(png_), "saveImg/loadImg failed");
  CapyImg* clone = CapyImgClone(png_);
  CUTEST_ASSERT($(img, isSame)(clone), "clone failed");
  CapyImgFree(&img);
  CUTEST_ASSERT(img == NULL, "img not reset");
  CapyImgFree(&png_);
  CapyImgFree(&clone);
  $(&stream, open)("UnitTests/TestImage/test002.png", "rb");
  CapyImg* pngGrey = $(&png, loadImg)(&stream);
  CUTEST_ASSERT(
    pngGrey->dims.width == 521 && pngGrey->dims.height == 757,
    "unexpected dimension width %d height %d",
    pngGrey->dims.width, pngGrey->dims.height);
  CUTEST_ASSERT(
    pngGrey->mode == capyImgMode_greyscale,
    "unexpected mode %d", pngGrey->mode);
  loop(i, 4) {
    CUTEST_ASSERT(
      equal(pngGrey->pixels[0].RGBA[i], 1.0),
      "unexpected pixel value [%d]=%lf",
      i, pngGrey->pixels[0].RGBA[i]);
  }
  loop(i, 4) {
    double pixels[] = {127.0 / 255.0, 127.0 / 255.0, 127.0 / 255.0, 1.0};
    CUTEST_ASSERT(
      equal(pngGrey->pixels[521 * 757 - 1].RGBA[i], pixels[i]),
      "unexpected pixel value [%d]=%lf != %lf",
      i, pngGrey->pixels[521 * 757 - 1].RGBA[i], pixels[i]);
  }
  $(&stream, open)("UnitTests/TestImage/test002_.png", "wb");
  $(pngGrey, cleanupPixelVals)();
  $(&png, saveImg)(pngGrey, &stream);
  $(&stream, open)("UnitTests/TestImage/test002_.png", "rb");
  CapyImg* pngGrey_ = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(pngGrey, isSame)(pngGrey_), "loadImg/saveImg failed");
  CapyImgFree(&pngGrey);
  CUTEST_ASSERT(pngGrey == NULL, "pngGrey not reset");
  CapyImgFree(&pngGrey_);
  CUTEST_ASSERT(pngGrey_ == NULL, "pngGrey_ not reset");
  $(&stream, open)("UnitTests/TestImage/test003.png", "rb");
  CapyImg* pngAlpha = $(&png, loadImg)(&stream);
  CUTEST_ASSERT(
    pngAlpha->dims.width == 521 && pngAlpha->dims.height == 757,
    "unexpected dimension width %d height %d",
    pngAlpha->dims.width, pngAlpha->dims.height);
  CUTEST_ASSERT(
    pngAlpha->mode == capyImgMode_rgba,
    "unexpected mode %d", pngAlpha->mode);
  loop(i, 4) {
    double pixels[] = {1.0, 1.0, 1.0, 0.49803921568627451677};
    CUTEST_ASSERT(
      equal(pngAlpha->pixels[0].RGBA[i], pixels[i]),
      "unexpected pixel value [%d]=%lf != %lf",
      i, pngAlpha->pixels[0].RGBA[i], pixels[i]);
  }
  loop(i, 4) {
    double pixels[] = {
      126.0 / 255.0, 127.0 / 255.0, 129.0 / 255.0, 0.49803921568627451677
    };
    CUTEST_ASSERT(
      equal(pngAlpha->pixels[521 * 757 - 1].RGBA[i], pixels[i]),
      "unexpected pixel value [%d]=%lf != %lf",
      i, pngAlpha->pixels[521 * 757 - 1].RGBA[i], pixels[i]);
  }
  $(&stream, open)("UnitTests/TestImage/test003_.png", "wb");
  $(&png, saveImg)(pngAlpha, &stream);
  $(&stream, open)("UnitTests/TestImage/test003_.png", "rb");
  CapyImg* pngAlpha_ = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(pngAlpha, isSame)(pngAlpha_), "loadImg/saveImg failed");
  $(&stream, destruct)();
  $(&png, destruct)();
  CapyImgFree(&pngAlpha);
  CUTEST_ASSERT(pngAlpha == NULL, "pngAlpha not reset");
  CapyImgFree(&pngAlpha_);
  CUTEST_ASSERT(pngAlpha_ == NULL, "pngAlpha_ not reset");
}

CUTEST(test002, "gamma") {
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("UnitTests/TestImage/test001.png", "rb");
  CapyImg* img = $(&png, loadImg)(&stream);
  CUTEST_ASSERT(equal(img->gamma, 2.2), "unexpected gamma %lf", img->gamma);
  CapyImg* imgGamma = CapyImgClone(img);
  $(imgGamma, setGamma)(1.0);
  CUTEST_ASSERT(
    equal(imgGamma->gamma, 1.0), "setGamma failed %lf", imgGamma->gamma);
  $(imgGamma, cleanupPixelVals)();
  $(&stream, open)("UnitTests/TestImage/test001gamma.png", "rb");
  CapyImg* img_ = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(img_, isSame)(imgGamma), "load gamma corrected image failed");
  $(&stream, destruct)();
  $(&png, destruct)();
  CapyImgFree(&img);
  CapyImgFree(&imgGamma);
  CapyImgFree(&img_);
}

CUTEST(test003, "brightness") {
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("UnitTests/TestImage/test004.png", "rb");
  CapyImg* img = $(&png, loadImg)(&stream);
  CapyImgPixel brightestPixel = $(img, getBrightestPixel)();
  CUTEST_ASSERT(
    brightestPixel.idx == 833002,
    "getBrightestPixel failed %lu", brightestPixel.idx);
  CUTEST_ASSERT(
    brightestPixel.pos.x == 202,
    "getBrightestPixel failed %d", brightestPixel.pos.x);
  CUTEST_ASSERT(
    brightestPixel.pos.y == 694,
    "getBrightestPixel failed %d", brightestPixel.pos.y);
  loop (i, 3) {
    CUTEST_ASSERT(
      equal(brightestPixel.color->RGB[i], 1.0),
      "getBrightestPixel failed [%d]=%lf", i, brightestPixel.color->RGB[i]);
  }
  CapyImgFree(&img);
  $(&stream, destruct)();
  $(&png, destruct)();
}

CUTEST(test004, "edge map") {
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("UnitTests/TestImage/test005.png", "rb");
  CapyImg* img = $(&png, loadImg)(&stream);
  $(img, fromRGBToGreyScale)();
  CapyImg* edgeMap = $(img, getEdgeMap)(2.0, 0.5, 0);
  $(edgeMap, cleanupPixelVals)();
  $(&stream, open)("UnitTests/TestImage/test005edgemap.png", "rb");
  CapyImg* check = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(edgeMap, isSame)(check), "getEdgeMap failed");
  CapyImgFree(&check);
  CapyImgFree(&edgeMap);
  edgeMap = $(img, getCannyEdgeMap3Chan)(2.0, 0.5, 0.1, 0.25);
  $(edgeMap, cleanupPixelVals)();
  $(&stream, open)("UnitTests/TestImage/test005cannyedgemap.png", "rb");
  check = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(edgeMap, isSame)(check), "getCannyEdgeMap3Chan");
  CapyImgFree(&check);
  CapyImgFree(&edgeMap);
  CapyImgFree(&img);
  $(&stream, destruct)();
  $(&png, destruct)();
}

CUTEST(test005, "conversion") {
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("UnitTests/TestImage/test005.png", "rb");
  CapyImg* img = $(&png, loadImg)(&stream);
  $(img, convertToColorSpace)(capyColorSpace_rgb);
  $(&stream, open)("UnitTests/TestImage/test005rgb.png", "rb");
  CapyImg* check = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(img, isSame)(check), "convertToColorSpace rgb failed");
  CapyImgFree(&check);
  CapyImgFree(&img);
  $(&stream, open)("UnitTests/TestImage/test005.png", "rb");
  img = $(&png, loadImg)(&stream);
  $(img, convertToColorSpace)(capyColorSpace_HSV);
  $(&stream, open)("UnitTests/TestImage/test005HSV.png", "rb");
  check = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(img, isSame)(check), "convertToColorSpace hsv failed");
  CapyImgFree(&check);
  CapyImgFree(&img);
  $(&stream, open)("UnitTests/TestImage/test005.png", "rb");
  img = $(&png, loadImg)(&stream);
  $(img, convertToColorSpace)(capyColorSpace_l1l2l3);
  $(&stream, open)("UnitTests/TestImage/test005l1l2l3.png", "rb");
  check = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(img, isSame)(check), "convertToColorSpace l1l2l3 failed");
  CapyImgFree(&check);
  CapyImgFree(&img);
  $(&stream, open)("UnitTests/TestImage/test005.png", "rb");
  img = $(&png, loadImg)(&stream);
  $(img, convertToColorSpace)(capyColorSpace_c1c2c3);
  $(&stream, open)("UnitTests/TestImage/test005c1c2c3.png", "rb");
  check = $(&png, loadImg)(&stream);
  CUTEST_ASSERT($(img, isSame)(check), "convertToColorSpace c1c2c3 failed");
  CapyImgFree(&check);
  CapyImgFree(&img);
  $(&stream, destruct)();
  $(&png, destruct)();
}

CUTEST(test006, "colors as point cloud") {
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("UnitTests/TestImage/test005cannyedgemap.png", "rb");
  CapyImg* img = $(&png, loadImg)(&stream);
  CapyPointCloud* pointCloud = $(img, getColorsAsPointCloud)();
  CUTEST_ASSERT(pointCloud->dim == 4, "unexpected dim %lu", pointCloud->dim);
  CUTEST_ASSERT(
    pointCloud->size == 1080000, "unexpected size %lu", pointCloud->size);
  loop(i, 3) {
    CUTEST_ASSERT(
      pointCloud->points[0].vals[i] < DBL_EPSILON,
      "getColorsAsPointCloud failed [%d]=%lf",
      i, pointCloud->points[0].vals[i]);
  }
  CapyImgFree(&img);
  $(&stream, destruct)();
  $(&png, destruct)();
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test007, "Padding (1)") {
  CUTEST_ASSERT(
    offsetof(CapyImgDims, width) == offsetof(CapyImgDims, vals[0]),
    "offset of width and dims[0] doesn't match %lu!=%lu",
    offsetof(CapyImgDims, width), offsetof(CapyImgDims, vals[0]));
  CUTEST_ASSERT(
    offsetof(CapyImgDims, height) == offsetof(CapyImgDims, vals[1]),
    "offset of height and dims[1] doesn't match %lu!=%lu",
    offsetof(CapyImgDims, height), offsetof(CapyImgDims, vals[1]));
}

CUTEST(test008, "Padding (2)") {
  CUTEST_ASSERT(
    offsetof(CapyImgPos, x) == offsetof(CapyImgPos, coords[0]),
    "offset of x and coords[0] doesn't match %lu!=%lu",
    offsetof(CapyImgPos, x), offsetof(CapyImgPos, coords[0]));
  CUTEST_ASSERT(
    offsetof(CapyImgPos, y) == offsetof(CapyImgPos, coords[1]),
    "offset of y and coords[1] doesn't match %lu!=%lu",
    offsetof(CapyImgPos, y), offsetof(CapyImgPos, coords[1]));
}

CUTEST(test009, "Image pasting") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test001.png");
  CapyImg* imgPasted =
    CapyImgLoadFromPath("UnitTests/TestImage/test001gamma.png");
  CapyImgPos posA = {.x = -100, .y = -100};
  CapyImgPos posB = {.x = 100, .y = 100};
  $(imgPasted, pasteInto)(img, &posA);
  $(imgPasted, pasteInto)(img, &posB);
  CapyImg* imgCheck = CapyImgLoadFromPath("UnitTests/TestImage/checkPaste.png");
  CUTEST_ASSERT($(img, isSame)(imgCheck), "pasteInto failed");
  CapyImgFree(&imgCheck);
  CapyImgFree(&imgPasted);
}

CUTEST(test010, "Conversion to black and white with horizontal lines") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, convertToBlackWhiteLightnessLine)(img->dims.height / 100);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkBlackWhiteLightLine.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "convertToBlackWhiteLightnessLine failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test011, "Rotate (1)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(M_PI / 2.0, &capyColorRGBAWhite);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test012, "Rotate (2)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(-M_PI / 2.0, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate02.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test013, "Rotate (3)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(M_PI_2 / 3.0, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate03.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test014, "Rotate (4)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(4.0 * M_PI_2 / 3.0, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate04.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test015, "Rotate (5)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(-4.0 * M_PI_2 / 3.0, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate05.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test016, "Rotate (6)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(-5.0 * M_PI_2 / 3.0, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate06.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test017, "Rotate (7)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(5.0 * M_PI_2 / 3.0, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate07.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test018, "Rotate (8)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(M_PI, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate08.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test019, "Rotate (9)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(1.5 * M_PI, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate09.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test020, "Rotate (10)") {
  CapyImg* img = CapyImgLoadFromPath("UnitTests/TestImage/test005.png");
  $(img, rotate)(-1.5 * M_PI, &capyColorRGBABlack);
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestImage/checkRotate10.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "rotate failed");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test021, "IntersectingAlpha") {
  CapyImg* imgA = CapyImgLoadFromPath("UnitTests/TestImage/test006.png");
  CapyImg* imgB = CapyImgLoadFromPath("UnitTests/TestImage/test007.png");
  CapyImgPos pos[5] = {
    {.x = 0, .y = 0},
    {.x = -100, .y = -100},
    {.x = 100, .y = 100},
    {.x = -200, .y = -200},
    {.x = 200, .y = 200},
  };
  bool intersect[5];
  loop(i, 5) intersect[i] = $(imgA, isIntersectingAlpha)(imgB, pos + i);
  CUTEST_ASSERT(
    intersect[0] && intersect[1] == false && intersect[2] == false &&
    intersect[3] == false && intersect[4] == false,
    "isIntersectingAlpha failed");
  CapyImgFree(&imgA);
  CapyImgFree(&imgB);
}

