#include "capy.h"
#ifndef FIXTURE
#define FIXTURE

// Structure to solve the tic-tac-toe game using CapyMinimax.
typedef struct TicTacToe {
  struct CapyMiniMaxWorldDef;
  uint8_t board[3][3];
  CapyPad(uint8_t[3][3], board);
  uint8_t action[2];
  CapyPad(uint8_t[2], action);
  void (*destructCapyMiniMaxWorld)(void);
} TicTacToe;

static void Destruct(void) {}
static TicTacToe TicTacToeCreate(void);
static CapyMiniMaxWorld* Clone(CapyMiniMaxWorldMemPool* const mempool) {
  methodOf(TicTacToe);
  TicTacToe initElem = TicTacToeCreate();
  TicTacToe* clone = (TicTacToe*)$(mempool, alloc)(&initElem);
  if(clone != NULL) {
    clone->nbActor = that->nbActor;
    clone->sente = that->sente;
    clone->depth = that->depth;
    loop(i, that->nbActor) clone->values[i] = that->values[i];
    loop(i, 3) loop(j, 3) clone->board[i][j] = that->board[i][j];
    loop(i, 2) clone->action[i] = that->action[i];
  }
  return (CapyMiniMaxWorld*)clone;
}

static void SetValues(TicTacToe* const that) {
  loop(i, 2) {
    double value = 0.0;
    TicTacToe* const t = that;
    uint8_t j = (uint8_t)i + 1;
    if(
      (t->board[0][0] == j && t->board[0][1] == j && t->board[0][2] == j) ||
      (t->board[1][0] == j && t->board[1][1] == j && t->board[1][2] == j) ||
      (t->board[2][0] == j && t->board[2][1] == j && t->board[2][2] == j) ||
      (t->board[0][0] == j && t->board[1][0] == j && t->board[2][0] == j) ||
      (t->board[0][1] == j && t->board[1][1] == j && t->board[2][1] == j) ||
      (t->board[0][2] == j && t->board[1][2] == j && t->board[2][2] == j) ||
      (t->board[0][0] == j && t->board[1][1] == j && t->board[2][2] == j) ||
      (t->board[2][0] == j && t->board[1][1] == j && t->board[0][2] == j)
    ) {
      value = 1.0;
    }
    that->values[i] = value;
  }
}

static void CreateChilds(CapyMiniMaxWorldMemPool* const mempool) {
  methodOf(TicTacToe);
  if(that->child != NULL) return;
  if(that->values[0] > 0.5 || that->values[1] > 0.5) return;
  loop(i, 3) loop(j, 3) {
    if(that->board[i][j] == 0) {
      TicTacToe* child = (TicTacToe*)$(that, clone)(mempool);
      if(child != NULL) {
        child->parent = (CapyMiniMaxWorld*)that;
        child->board[i][j] = (uint8_t)(that->sente + 1);
        child->sente = (that->sente + 1) % that->nbActor;
        child->depth = that->depth + 1;
        child->action[0] = (uint8_t)i;
        child->action[1] = (uint8_t)j;
        SetValues(child);
        child->sibling = that->child;
        that->child = (CapyMiniMaxWorld*)child;
      }
    }
  }
}

static bool IsSame(CapyMiniMaxWorld const* const world) {
  methodOf(TicTacToe);
  bool sameBoard = true;
  TicTacToe const* other = (TicTacToe const*)world;
  loop(i, 3) loop(j, 3) sameBoard &= (that->board[i][j] == other->board[i][j]);
  return (
    that->sente == other->sente &&
    that->depth == other->depth &&
    that->action[0] == other->action[0] &&
    that->action[1] == other->action[1] &&
    sameBoard
  );
}

static TicTacToe TicTacToeCreate(void) {
  TicTacToe that;
  size_t nbActor = 2;
  CapyInherits(that, CapyMiniMaxWorld, (nbActor));
  that.destruct = Destruct;
  that.clone = Clone;
  that.createChilds = CreateChilds;
  that.isSame = IsSame;
  loop(i, 3) loop(j, 3) that.board[i][j] = 0;
  return that;
}

#endif
CUTEST(test001, "tic-tac-toe") {
  CapyMiniMax* minimax = CapyMiniMaxAlloc(sizeof(TicTacToe));
  size_t nbMaxWorld = 10000;
  minimax->mempool.sizeMax = nbMaxWorld;
  minimax->maxDepthExplore = 10;
  minimax->pruningThreshold = 0.5;
  TicTacToe initialWorld = TicTacToeCreate();
  $(minimax, setCurrentWorld)((CapyMiniMaxWorld*)(&initialWorld));
  $(&initialWorld, destruct)();
  $(minimax, start)();
  TicTacToe* best = NULL;
  size_t nbIter = 0;
  do {
    CapySleepMs(10);
    best = (TicTacToe*)$(minimax, getNextBest)();
    if(best != NULL) {
      $(minimax, setCurrentWorld)((CapyMiniMaxWorld*)best);
    }
    ++nbIter;
  } while(best != NULL && nbIter < 11);
  $(minimax, stop)();
  CUTEST_ASSERT(
    nbIter == 10 &&
    fabs(minimax->currentWorld->values[0] - 0.0) < 1e-6 &&
    fabs(minimax->currentWorld->values[1] - 0.0) < 1e-6,
    "unexpected result");
  CapyMiniMaxFree(&minimax);
}
