#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "PointCloud operations") {
  CapyPlyFormat ply = CapyPlyFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("./UnitTests/TestPointCloud/stanford_bunny.ply", "rb");
  CapyPointCloud* pointCloud = $(&ply, loadPointCloud)(&stream);
  CUTEST_ASSERT(pointCloud->dim == 5, "unexpected dim %lu", pointCloud->dim);
  CUTEST_ASSERT(
    pointCloud->size == 35947, "unexpected size %lu", pointCloud->size);
  CUTEST_ASSERT(
    fabs(pointCloud->points[0].vals[0] - -0.0378297) < 1e-6,
    "loadPointCloud failed %lf", pointCloud->points[0].vals[0]);
  CUTEST_ASSERT(
    fabs(pointCloud->points[1].vals[1] - 0.128887) < 1e-6,
    "loadPointCloud failed %lf", pointCloud->points[0].vals[0]);
  CapyPointCloudFree(&pointCloud);
  $(&stream, destruct)();
  $(&ply, destruct)();
}

CUTEST(test002, "getApproxBezier") {
  CapyBezier* bezier = CapyBezierAlloc(1, 2, 3);
  loop(iCtrl, bezier->nbCtrls) loop(iOut, bezier->dimOut) {
    bezier->ctrls[iCtrl].vals[iOut] = (double)(iCtrl + iOut);
  }
  CapyPointCloud* pointCloud =
    CapyPointCloudAlloc(bezier->dimIn + bezier->dimOut);
  size_t nbPoint = 100;
  safeMalloc(pointCloud->points, nbPoint);
  pointCloud->size = nbPoint;
  uint64_t iPoint = 0;
  loop(i, 10) loop(j, 10) {
    pointCloud->points[iPoint] = CapyVecCreate(pointCloud->dim);
    pointCloud->points[iPoint].vals[0] = 0.1 * (double)i;
    pointCloud->points[iPoint].vals[1] = 0.1 * (double)j;
    $(bezier, eval)(
      pointCloud->points[iPoint].vals,
      pointCloud->points[iPoint].vals + 2);
    ++iPoint;
  }
  CapyBezier* approx = $(pointCloud, getApproxBezier)(2, 2);
  CUTEST_ASSERT(approx != NULL, "getApproxBezier failed");
  loop(jPoint, pointCloud->size) {
    double out[3];
    $(approx, eval)(pointCloud->points[jPoint].vals, out);
    loop(iOut, 3) {
      CUTEST_ASSERT(
        fabs(out[iOut] - pointCloud->points[jPoint].vals[2 + iOut]) < 1e-6,
        "getApproxBezier failed [%lu,%d]=%lf != %lf",
        jPoint, iOut,
        out[iOut], pointCloud->points[jPoint].vals[2 + iOut]);
    }
  }
  CapyPointCloudFree(&pointCloud);
  CapyBezierFree(&bezier);
  CapyBezierFree(&approx);
}

CUTEST(test003, "updateCovariance") {
  CapyPlyFormat ply = CapyPlyFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("./UnitTests/TestPointCloud/stanford_bunny.ply", "rb");
  CapyPointCloud* pointCloud = $(&ply, loadPointCloud)(&stream);
  $(pointCloud, updateCovariance)();
  double checkMean[5] = {-0.026760, 0.095216, 0.008947, 0.577079, 0.481677};
  double checkCovariance[25] = {
    0.001680, -0.000577, 0.000046, 0.000732, 0.000070,
    -0.000577, 0.001725, -0.000259, -0.000465, 0.000001,
    0.000046, -0.000259, 0.000793, 0.000144, 0.000018,
    0.000732, -0.000465, 0.000144, 0.028462, 0.003204,
    0.000070, 0.000001, 0.000018, 0.003204, 0.002984,
  };
  loop(i, 5) {
    CUTEST_ASSERT(
      fabs(pointCloud->mean.vals[i] - checkMean[i]) < 1e-6,
      "updateCovariance failed [%d]=%lf != %lf",
      i, pointCloud->mean.vals[i], checkMean[i]);
  }
  loop(i, 25) {
    CUTEST_ASSERT(
      fabs(pointCloud->covariance.vals[i] - checkCovariance[i]) < 1e-6,
      "updateCovariance failed [%d]=%lf != %lf",
      i, pointCloud->covariance.vals[i], checkCovariance[i]);
  }
  CapyPointCloudFree(&pointCloud);
  $(&stream, destruct)();
  $(&ply, destruct)();
}

CUTEST(test004, "updatePrincipalComponent") {
  CapyPlyFormat ply = CapyPlyFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)("./UnitTests/TestPointCloud/stanford_bunny.ply", "rb");
  CapyPointCloud* pointCloud = $(&ply, loadPointCloud)(&stream);
  $(pointCloud, updatePrincipalComponent)();
  double checkEigenValue[5] = {
    0.028888, 0.002598, 0.002275, 0.001173, 0.000710
  };
  double checkPrincComp[25] = {
    0.027371, -0.107702, 0.660079, -0.727601, -0.150143, -0.017624,
    0.140889, -0.714983, -0.603446, -0.323252, 0.005373, -0.022210,
    0.142850, 0.325875, -0.934279, 0.991887, -0.115345, -0.053085,
    0.005747, 0.002334, 0.122769, 0.977114, 0.172831, 0.014896, 0.009099
  };
  loop(i, 5) {
    CUTEST_ASSERT(
      fabs(pointCloud->eigenValue.vals[i] - checkEigenValue[i]) < 1e-6,
      "updatePrincipalComponent failed [%d]=%lf != %lf",
      i, pointCloud->eigenValue.vals[i], checkEigenValue[i]);
  }
  loop(i, 25) {
    CUTEST_ASSERT(
      fabs(pointCloud->principalComponent.vals[i] - checkPrincComp[i]) < 1e-6,
      "updatePrincipalComponent failed [%d]=%lf != %lf",
      i, pointCloud->principalComponent.vals[i], checkPrincComp[i]);
  }
  CapyPointCloudFree(&pointCloud);
  $(&stream, destruct)();
  $(&ply, destruct)();
}

CUTEST(test005, "mean and standard deviation") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(1);
  pointCloud->size = 8;
  safeMalloc(pointCloud->points, 8);
  double vals[8] = {2, 4, 4, 4, 5, 5, 7, 9};
  loop(i, 8) {
    pointCloud->points[i] = CapyVecCreate(1);
    pointCloud->points[i].vals[0] = vals[i];
  }
  $(pointCloud, updateStdDev)();
  double checkMean[1] = {5.0};
  double checkStdDev[1] = {2.0};
  loop(i, 1) {
    CUTEST_ASSERT(
      fabs(pointCloud->mean.vals[i] - checkMean[i]) < 1e-6,
      "updateMean failed [%d]=%lf != %lf",
      i, pointCloud->mean.vals[i], checkMean[i]);
  }
  loop(i, 1) {
    CUTEST_ASSERT(
      fabs(pointCloud->stdDev.vals[i] - checkStdDev[i]) < 1e-6,
      "updateStdDev failed [%d]=%lf != %lf",
      i, pointCloud->stdDev.vals[i], checkStdDev[i]);
  }
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test006, "pearson correlation") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 10;
  safeMalloc(pointCloud->points, 10);
  double vals[10][2] = {
    {3.63, 53.1},
    {3.02, 49.7},
    {3.82, 48.4},
    {3.42, 54.2},
    {3.59, 54.9},
    {2.87, 43.7},
    {3.03, 47.2},
    {3.46, 45.2},
    {3.36, 54.4},
    {3.3, 50.4}
  };
  loop(i, 10) {
    pointCloud->points[i] = CapyVecCreate(2);
    pointCloud->points[i].vals[0] = vals[i][0];
    pointCloud->points[i].vals[1] = vals[i][1];
  }
  $(pointCloud, updatePearsonCorrelation)();
  double check[4] = {1.0, 0.470177, 0.470177, 1.0};
  loop(i, 4) {
    CUTEST_ASSERT(
      fabs(pointCloud->pearsonCorrelation.vals[i] - check[i]) < 1e-6,
      "updatePearsonCorrelation failed [%d]=%lf != %lf",
      i, pointCloud->pearsonCorrelation.vals[i], check[i]);
  }
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test007, "Nearest neighbour (1)") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 1;
  safeMalloc(pointCloud->points, pointCloud->size);
  loop(i, pointCloud->size) {
    pointCloud->points[i] = CapyVecCreate(pointCloud->dim);
    loop(j, pointCloud->dim) pointCloud->points[i].vals[j] = 0.0;
  }
  CapyVec origin = (CapyVec){.dim = 2, .vals = (double[2]){0.0, 0.0}};
  CapyPointCloudNearestNeighbour* pcnn =
    CapyPointCloudNearestNeighbourAlloc(pointCloud, 3, &origin);
  CapyVec point = CapyVecCreate(pointCloud->dim);
  loop(j, pointCloud->dim) point.vals[j] = 1.0;
  CapyPointCloudNearestNeighbourRes res = $(pcnn, query)(&point, 0);
  CUTEST_ASSERT(
    res.iPoint == 0 && fabs(res.dist - sqrt(2.0)) < 1e-6 &&
    res.nbTestedPoint == 1,
    "iPoint=%lu, dist=%lf, nbTestedPoint=%lu",
    res.iPoint, res.dist, res.nbTestedPoint);
  CapyVecDestruct(&point);
  CapyPointCloudNearestNeighbourFree(&pcnn);
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test008, "Nearest neighbour (2)") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 2;
  safeMalloc(pointCloud->points, pointCloud->size);
  loop(i, pointCloud->size) {
    pointCloud->points[i] = CapyVecCreate(pointCloud->dim);
    loop(j, pointCloud->dim) pointCloud->points[i].vals[j] = (double)i;
  }
  CapyVec origin = (CapyVec){.dim = 2, .vals = (double[3]){0.0, 0.0}};
  CapyPointCloudNearestNeighbour* pcnn =
    CapyPointCloudNearestNeighbourAlloc(pointCloud, 3, &origin);
  CapyVec point = CapyVecCreate(pointCloud->dim);
  loop(j, pointCloud->dim) point.vals[j] = 1.0;
  CapyPointCloudNearestNeighbourRes res = $(pcnn, query)(&point, 0);
  CUTEST_ASSERT(
    res.iPoint == 1 && fabs(res.dist - 0.0) < 1e-6 &&
    res.nbTestedPoint == 2,
    "iPoint=%lu, dist=%lf, nbTestedPoint=%lu",
    res.iPoint, res.dist, res.nbTestedPoint);
  CapyVecDestruct(&point);
  CapyPointCloudNearestNeighbourFree(&pcnn);
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test009, "Nearest neighbour (3)") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 4;
  safeMalloc(pointCloud->points, pointCloud->size);
  loop(i, pointCloud->size) {
    pointCloud->points[i] = CapyVecCreate(pointCloud->dim);
    loop(j, pointCloud->dim) pointCloud->points[i].vals[j] = (double)i;
  }
  CapyVec origin = (CapyVec){.dim = 2, .vals = (double[2]){0.0, 0.0}};
  CapyPointCloudNearestNeighbour* pcnn =
    CapyPointCloudNearestNeighbourAlloc(pointCloud, 3, &origin);
  CapyVec point = CapyVecCreate(pointCloud->dim);
  loop(j, pointCloud->dim) point.vals[j] = 0.0;
  CapyPointCloudNearestNeighbourRes res = $(pcnn, query)(&point, 0);
  CUTEST_ASSERT(
    res.iPoint == 0 && fabs(res.dist - 0.0) < 1e-6 &&
    res.nbTestedPoint == 1,
    "iPoint=%lu, dist=%lf, nbTestedPoint=%lu",
    res.iPoint, res.dist, res.nbTestedPoint);
  CapyVecDestruct(&point);
  CapyPointCloudNearestNeighbourFree(&pcnn);
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test010, "Nearest neighbour (4)") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 4;
  safeMalloc(pointCloud->points, pointCloud->size);
  loop(i, pointCloud->size) {
    pointCloud->points[i] = CapyVecCreate(pointCloud->dim);
    loop(j, pointCloud->dim) pointCloud->points[i].vals[j] = (double)i;
  }
  CapyVec origin = (CapyVec){.dim = 2, .vals = (double[2]){0.0, 0.0}};
  CapyPointCloudNearestNeighbour* pcnn =
    CapyPointCloudNearestNeighbourAlloc(pointCloud, 3, &origin);
  CapyVec point = CapyVecCreate(pointCloud->dim);
  loop(j, pointCloud->dim) point.vals[j] = 1.0;
  CapyPointCloudNearestNeighbourRes res = $(pcnn, query)(&point, 0);
  CUTEST_ASSERT(
    res.iPoint == 1 && fabs(res.dist - 0.0) < 1e-6 &&
    res.nbTestedPoint == 1,
    "iPoint=%lu, dist=%lf, nbTestedPoint=%lu",
    res.iPoint, res.dist, res.nbTestedPoint);
  CapyVecDestruct(&point);
  CapyPointCloudNearestNeighbourFree(&pcnn);
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test011, "Nearest neighbour (5)") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 4;
  safeMalloc(pointCloud->points, pointCloud->size);
  loop(i, pointCloud->size) {
    pointCloud->points[i] = CapyVecCreate(pointCloud->dim);
    loop(j, pointCloud->dim) pointCloud->points[i].vals[j] = (double)i;
  }
  CapyVec origin = (CapyVec){.dim = 2, .vals = (double[2]){0.0, 0.0}};
  CapyPointCloudNearestNeighbour* pcnn =
    CapyPointCloudNearestNeighbourAlloc(pointCloud, 3, &origin);
  CapyVec point = CapyVecCreate(pointCloud->dim);
  loop(j, pointCloud->dim) point.vals[j] = 2.0;
  CapyPointCloudNearestNeighbourRes res = $(pcnn, query)(&point, 0);
  CUTEST_ASSERT(
    res.iPoint == 2 && fabs(res.dist - 0.0) < 1e-6 &&
    res.nbTestedPoint == 2,
    "iPoint=%lu, dist=%lf, nbTestedPoint=%lu",
    res.iPoint, res.dist, res.nbTestedPoint);
  CapyVecDestruct(&point);
  CapyPointCloudNearestNeighbourFree(&pcnn);
  CapyPointCloudFree(&pointCloud);
}

CUTEST(test012, "Nearest neighbour (6)") {
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(2);
  pointCloud->size = 4;
  safeMalloc(pointCloud->points, pointCloud->size);
  loop(i, pointCloud->size) {
    pointCloud->points[i] = CapyVecCreate(pointCloud->dim);
    loop(j, pointCloud->dim) pointCloud->points[i].vals[j] = (double)i;
  }
  CapyVec origin = (CapyVec){.dim = 2, .vals = (double[2]){0.0, 0.0}};
  CapyPointCloudNearestNeighbour* pcnn =
    CapyPointCloudNearestNeighbourAlloc(pointCloud, 3, &origin);
  CapyVec point = CapyVecCreate(pointCloud->dim);
  loop(j, pointCloud->dim) point.vals[j] = 3.0;
  CapyPointCloudNearestNeighbourRes res = $(pcnn, query)(&point, 0);
  CUTEST_ASSERT(
    res.iPoint == 3 && fabs(res.dist - 0.0) < 1e-6 &&
    res.nbTestedPoint == 3,
    "iPoint=%lu, dist=%lf, nbTestedPoint=%lu",
    res.iPoint, res.dist, res.nbTestedPoint);
  CapyVecDestruct(&point);
  CapyPointCloudNearestNeighbourFree(&pcnn);
  CapyPointCloudFree(&pointCloud);
}
