#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "quaternion alloc/free") {
  CapyQuaternion* quat = CapyQuaternionAlloc();
  CUTEST_ASSERT(
    fabs(quat->vals[0]) < 1e-9 &&
    fabs(quat->vals[1]) < 1e-9 &&
    fabs(quat->vals[2]) < 1e-9 &&
    fabs(1.0 - quat->vals[3]) < 1e-9,
    "quaternion initialisation failed");
  CapyQuaternionFree(&quat);
}

CUTEST(test002, "quaternion from rotation matrix") {
  CapyMat mat = CapyMatCreate(3, 3);
  double theta = CapyDegToRad(10.0);
  mat.vals[0] = 1.0;
  mat.vals[4] = cos(theta);
  mat.vals[5] = -sin(theta);
  mat.vals[7] = sin(theta);
  mat.vals[8] = cos(theta);
  CapyQuaternion* quat = CapyQuaternionAllocFromRotMat(&mat);
  CapyMat res = CapyMatCreate(3, 3);
  $(quat, toRotMat)(&res);
  CUTEST_ASSERT(
    fabs(mat.vals[0] - res.vals[0]) < 1e-9 &&
    fabs(mat.vals[1] - res.vals[1]) < 1e-9 &&
    fabs(mat.vals[2] - res.vals[2]) < 1e-9 &&
    fabs(mat.vals[3] - res.vals[3]) < 1e-9 &&
    fabs(mat.vals[4] - res.vals[4]) < 1e-9 &&
    fabs(mat.vals[5] - res.vals[5]) < 1e-9 &&
    fabs(mat.vals[6] - res.vals[6]) < 1e-9 &&
    fabs(mat.vals[7] - res.vals[7]) < 1e-9 &&
    fabs(mat.vals[8] - res.vals[8]) < 1e-9,
    "CapyQuaternionAllocFromRotMat failed\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n",
    mat.vals[0], mat.vals[1], mat.vals[2],
    res.vals[0], res.vals[1], res.vals[2],
    mat.vals[3], mat.vals[4], mat.vals[5],
    res.vals[3], res.vals[4], res.vals[5],
    mat.vals[6], mat.vals[7], mat.vals[8],
    res.vals[6], res.vals[7], res.vals[8]);
  CapyMatDestruct(&res);
  CapyMatDestruct(&mat);
  CapyQuaternionFree(&quat);
}

CUTEST(test003, "quaternion from rotation axis") {
  CapyMat mat = CapyMatCreate(3, 3);
  double theta = CapyDegToRad(10.0);
  mat.vals[0] = 1.0;
  mat.vals[4] = cos(theta);
  mat.vals[5] = -sin(theta);
  mat.vals[7] = sin(theta);
  mat.vals[8] = cos(theta);
  double axis[3] = {1, 0, 0};
  CapyQuaternion* quat = CapyQuaternionAllocFromRotAxis(axis, theta);
  CapyMat res = CapyMatCreate(3, 3);
  $(quat, toRotMat)(&res);
  CUTEST_ASSERT(
    fabs(mat.vals[0] - res.vals[0]) < 1e-9 &&
    fabs(mat.vals[1] - res.vals[1]) < 1e-9 &&
    fabs(mat.vals[2] - res.vals[2]) < 1e-9 &&
    fabs(mat.vals[3] - res.vals[3]) < 1e-9 &&
    fabs(mat.vals[4] - res.vals[4]) < 1e-9 &&
    fabs(mat.vals[5] - res.vals[5]) < 1e-9 &&
    fabs(mat.vals[6] - res.vals[6]) < 1e-9 &&
    fabs(mat.vals[7] - res.vals[7]) < 1e-9 &&
    fabs(mat.vals[8] - res.vals[8]) < 1e-9,
    "CapyQuaternionAllocFromRotAxis failed\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n",
    mat.vals[0], mat.vals[1], mat.vals[2],
    res.vals[0], res.vals[1], res.vals[2],
    mat.vals[3], mat.vals[4], mat.vals[5],
    res.vals[3], res.vals[4], res.vals[5],
    mat.vals[6], mat.vals[7], mat.vals[8],
    res.vals[6], res.vals[7], res.vals[8]);
  double resAxis[3];
  $(quat, getRotAxis)(resAxis);
  CUTEST_ASSERT(
    fabs(axis[0] - resAxis[0]) < 1e-9 &&
    fabs(axis[1] - resAxis[1]) < 1e-9 &&
    fabs(axis[2] - resAxis[2]) < 1e-9,
    "getRotAxis failed, (%lf, %lf, %lf)!=(1, 0, 0)",
    resAxis[0], resAxis[1], resAxis[2]);
  double t = $(quat, getRotAngle)();
  CUTEST_ASSERT(
    fabs(theta - t) < 1e-6,
    "getRotAngle failed, %lf!=%lf", t, theta);
  CapyMatDestruct(&res);
  CapyMatDestruct(&mat);
  CapyQuaternionFree(&quat);
}

CUTEST(test004, "quaternion from rotation axis (2)") {
  double theta = CapyDegToRad(90.0);
  double axis[3] = {0, 1, 0};
  CapyQuaternion* quat = CapyQuaternionAllocFromRotAxis(axis, theta);
  CUTEST_ASSERT(
    fabs(quat->vals[0] - 0.0) < 1e-9 &&
    fabs(quat->vals[1] - 1.0 / sqrt(2.0)) < 1e-9 &&
    fabs(quat->vals[2] - 0.0) < 1e-9 &&
    fabs(quat->vals[3] - 1.0 / sqrt(2.0)) < 1e-9,
    "CapyQuaternionAllocFromRotAxis failed\n"
    "%lf %lf %lf %lf != 0 1/sqrt(2) 0 1/sqrt(2)\n",
    quat->vals[0], quat->vals[1], quat->vals[2], quat->vals[3]);
  CapyQuaternionFree(&quat);
}

CUTEST(test005, "quaternion compose") {
  CapyMat mat = CapyMatCreate(3, 3);
  double theta = CapyDegToRad(10.0);
  mat.vals[0] = 1.0;
  mat.vals[4] = cos(2.0 * theta);
  mat.vals[5] = -sin(2.0 * theta);
  mat.vals[7] = sin(2.0 * theta);
  mat.vals[8] = cos(2.0 * theta);
  double axisX[3] = {1, 0, 0};
  CapyQuaternion* quat = CapyQuaternionAllocFromRotAxis(axisX, theta);
  $(quat, compose)(quat, quat);
  CapyMat res = CapyMatCreate(3, 3);
  $(quat, toRotMat)(&res);
  CUTEST_ASSERT(
    fabs(mat.vals[0] - res.vals[0]) < 1e-9 &&
    fabs(mat.vals[1] - res.vals[1]) < 1e-9 &&
    fabs(mat.vals[2] - res.vals[2]) < 1e-9 &&
    fabs(mat.vals[3] - res.vals[3]) < 1e-9 &&
    fabs(mat.vals[4] - res.vals[4]) < 1e-9 &&
    fabs(mat.vals[5] - res.vals[5]) < 1e-9 &&
    fabs(mat.vals[6] - res.vals[6]) < 1e-9 &&
    fabs(mat.vals[7] - res.vals[7]) < 1e-9 &&
    fabs(mat.vals[8] - res.vals[8]) < 1e-9,
    "compose failed\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n",
    mat.vals[0], mat.vals[1], mat.vals[2],
    res.vals[0], res.vals[1], res.vals[2],
    mat.vals[3], mat.vals[4], mat.vals[5],
    res.vals[3], res.vals[4], res.vals[5],
    mat.vals[6], mat.vals[7], mat.vals[8],
    res.vals[6], res.vals[7], res.vals[8]);
  CapyMatDestruct(&res);
  CapyMatDestruct(&mat);
  CapyQuaternionFree(&quat);
}

CUTEST(test006, "quaternion inverse") {
  CapyMat mat = CapyMatCreate(3, 3);
  double theta = CapyDegToRad(10.0);
  mat.vals[0] = 1.0;
  mat.vals[4] = 1.0;
  mat.vals[8] = 1.0;
  double axisX[3] = {1, 0, 0};
  CapyQuaternion* quat = CapyQuaternionAllocFromRotAxis(axisX, theta);
  CapyQuaternion inv = CapyQuaternionCreate();
  $(quat, inverse)(&inv);
  $(quat, compose)(&inv, quat);
  CapyMat res = CapyMatCreate(3, 3);
  $(quat, toRotMat)(&res);
  CUTEST_ASSERT(
    fabs(mat.vals[0] - res.vals[0]) < 1e-3 &&
    fabs(mat.vals[1] - res.vals[1]) < 1e-3 &&
    fabs(mat.vals[2] - res.vals[2]) < 1e-3 &&
    fabs(mat.vals[3] - res.vals[3]) < 1e-3 &&
    fabs(mat.vals[4] - res.vals[4]) < 1e-3 &&
    fabs(mat.vals[5] - res.vals[5]) < 1e-3 &&
    fabs(mat.vals[6] - res.vals[6]) < 1e-3 &&
    fabs(mat.vals[7] - res.vals[7]) < 1e-3 &&
    fabs(mat.vals[8] - res.vals[8]) < 1e-3,
    "inverse failed\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n"
    "%lf %lf %lf != %lf %lf %lf\n",
    mat.vals[0], mat.vals[1], mat.vals[2],
    res.vals[0], res.vals[1], res.vals[2],
    mat.vals[3], mat.vals[4], mat.vals[5],
    res.vals[3], res.vals[4], res.vals[5],
    mat.vals[6], mat.vals[7], mat.vals[8],
    res.vals[6], res.vals[7], res.vals[8]);
  CapyMatDestruct(&res);
  CapyMatDestruct(&mat);
  CapyQuaternionFree(&quat);
}

CUTEST(test007, "quaternion difference") {
  double theta = CapyDegToRad(10.0);
  double axisX[3] = {1, 0, 0};
  double axisY[3] = {0, 1, 0};
  CapyQuaternion* quatA = CapyQuaternionAllocFromRotAxis(axisX, theta);
  CapyQuaternion* quatB = CapyQuaternionAllocFromRotAxis(axisY, theta);
  CapyQuaternion* diff = CapyQuaternionAlloc();
  $(quatA, difference)(quatB, diff);
  $(diff, compose)(quatA, quatA);
  bool isEqual = $(quatA, isEqualTo)(quatB);
  CUTEST_ASSERT(
    isEqual,
    "difference failed\n"
    "%lf %lf %lf %lf != %lf %lf %lf %lf",
    quatA->vals[0], quatA->vals[1], quatA->vals[2], quatA->vals[3],
    quatB->vals[0], quatB->vals[1], quatB->vals[2], quatB->vals[3]);
  CapyQuaternionFree(&quatA);
  CapyQuaternionFree(&quatB);
  CapyQuaternionFree(&diff);
}

CUTEST(test008, "quaternion apply") {
  double theta = CapyDegToRad(90.0);
  double axisY[3] = {0, 1, 0};
  CapyQuaternion* quatA = CapyQuaternionAllocFromRotAxis(axisY, theta);
  double u[3] = {1, 0, 0};
  $(quatA, apply)(u, u);
  CUTEST_ASSERT(
    fabs(u[0] - 0.0) < 1e-3 &&
    fabs(u[1] - 0.0) < 1e-3 &&
    fabs(u[2] + 1.0) < 1e-3,
    "apply failed (%lf, %lf, %lf) != (0, 0, -1)", u[0], u[1], u[2]);
  CapyQuaternionFree(&quatA);
}

CUTEST(test009, "quaternion apply (2)") {
  double theta = CapyDegToRad(90.0);
  double axisX[3] = {1, 0, 0};
  double axisZ[3] = {0, 0, 1};
  CapyQuaternion* quatA = CapyQuaternionAllocFromRotAxis(axisX, theta);
  CapyQuaternion* quatB = CapyQuaternionAllocFromRotAxis(axisZ, theta);
  double u[3] = {0, 0, 2};
  $(quatA, apply)(u, u);
  $(quatB, apply)(u, u);
  CUTEST_ASSERT(
    fabs(u[0] - 2.0) < 1e-3 &&
    fabs(u[1] - 0.0) < 1e-3 &&
    fabs(u[2] - 0.0) < 1e-3,
    "apply failed (%lf, %lf, %lf) != (2, 0, 0)", u[0], u[1], u[2]);
  CapyQuaternionFree(&quatA);
  CapyQuaternionFree(&quatB);
}

CUTEST(test010, "quaternion rotation a vector into another") {
  double from[3] = {1, 0, 0};
  double to[3] = {0, 0, 1};
  CapyQuaternion* quat = CapyQuaternionAllocRotFromVecToVec(from, to);
  $(quat, apply)(from, from);
  CUTEST_ASSERT(
    fabs(from[0] - to[0]) < 1e-3 &&
    fabs(from[1] - to[1]) < 1e-3 &&
    fabs(from[2] - to[2]) < 1e-3,
    "apply failed (%lf, %lf, %lf) != (%lf, %lf, %lf)",
    from[0], from[1], from[2], to[0], to[1], to[2]);
  CapyQuaternionFree(&quat);
}
