#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
static void OscillatorDerivative(
  double const* const in,
        double* const out) {
  double const k = 1.5;
  double const m = 0.5;
  out[0] = -k / m * in[0] * in[1];
}

static double Oscillator(double const t) {
  double const k = 1.5;
  double const m = 0.5;
  double const a = 1.0;
  double const phi = M_PI_2;
  return a * sin(sqrt(k / m) * t + phi);
}

static void PendulumDerivative(
  double const* const in,
        double* const out) {
  double const M = 1.0;
  double const mp = 0.1;
  double const g = 9.8;
  double const L = 1.0;
  double const F = (in[0] < 0.001 ? 1.0 : 0.0);
  out[0] = (
    -mp * L * sin(in[1]) * cos(in[1]) * in[3] * in[3] +
    (M + mp) * g * sin(in[1]) + cos(in[1]) * F
  ) / (
    (M + mp * (1.0 - cos(in[1]) * cos(in[1]))) * L
  );
  out[1] = (
    -mp * L * sin(in[1]) * in[3] * in[3] +
    mp * g * sin(in[1]) * cos(in[1]) + F
  ) / (
    M + mp * (1.0 - cos(in[1]) * cos(in[1]))
  );
}

#endif
CUTEST(test001, "Harmonic oscillator, eval") {
  CapyMathFun derivative = CapyMathFunCreate(2, 1);
  derivative.eval = OscillatorDerivative;
  size_t const order = 1;
  CapyRungeKutta* rk = CapyRungeKuttaAlloc(&derivative, order);
  rk->initVal.vals[1] = 1.0;
  double const t = rk->deltaT * 100.0;
  $(rk, eval)(t, rk->initVal.vals);
  double check = Oscillator(t);
  CUTEST_ASSERT(
    fabs(rk->initVal.vals[1] - check) < 1e-4,
    "%lf!=%lf\n", rk->initVal.vals[1], check);
  CapyRungeKuttaFree(&rk);
  $(&derivative, destruct)();
}

CUTEST(test002, "Harmonic oscillator, step") {
  CapyMathFun derivative = CapyMathFunCreate(2, 1);
  derivative.eval = OscillatorDerivative;
  size_t const order = 1;
  CapyRungeKutta* rk = CapyRungeKuttaAlloc(&derivative, order);
  rk->initVal.vals[1] = 1.0;
  bool isOk = true;
  loop(i, 100) {
    $(rk, step)(rk->initVal.vals);
    double check = Oscillator(rk->initVal.vals[0]);
    isOk &= (fabs(rk->initVal.vals[1]) - check < 1e-4);
  }
  CUTEST_ASSERT(isOk, "isOk=%d\n", isOk);
  CapyRungeKuttaFree(&rk);
  $(&derivative, destruct)();
}

CUTEST(test003, "Pendulum, step") {
  CapyMathFun derivative = CapyMathFunCreate(5, 2);
  derivative.eval = PendulumDerivative;
  size_t const order = 2;
  CapyRungeKutta* rk = CapyRungeKuttaAlloc(&derivative, order);
  rk->deltaT = 0.01;
  rk->initVal.vals[1] = 0.001;
  rk->initVal.vals[2] = 0.0;
  bool isOk = true;
  loop(i, 100) {
    $(rk, step)(rk->initVal.vals);
  }
  isOk &=
    (fabs(rk->initVal.vals[1] - 0.021605) < 1e-6) &&
    (fabs(rk->initVal.vals[2] - 0.003388) < 1e-6) &&
    (fabs(rk->initVal.vals[3] - 0.071754) < 1e-6) &&
    (fabs(rk->initVal.vals[4] - 0.008037) < 1e-6);
  CUTEST_ASSERT(isOk, "isOk=%d\n", isOk);
  CapyRungeKuttaFree(&rk);
  $(&derivative, destruct)();
}
