#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
static CapyImg* FieldToImg(
  CapySdf const* const shape,
          double const zoom) {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_100x100);
  forEach(pixel, img->iter) {
    double pos[3] = {
      ((double)(pixel.pos.x) - 50.0) * zoom,
      ((double)(99 - pixel.pos.y) - 50.0) * zoom,
      0.0
    };
    double dist = $(shape, dist)(pos);
    if(dist > 0) pixel.color->rgb[0] = 1.0;
    else pixel.color->rgb[2] = 1.0;
    pixel.color->rgb[1] = fabs(dist);
  }
  $(img, normalise3Chan)();
  return img;
}

static CapyImg* RayMarchingToImg(
  CapySdf const* const shape,
          double const zoom) {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_100x100);
  forEach(pixel, img->iter) {
    double pos[3] = {
      ((double)(pixel.pos.x) - 50.0) * zoom,
      ((double)(99 - pixel.pos.y) - 50.0) * zoom,
      -10.0
    };
    double dir[3] = {0.0, 0.0, 1.0};
    CapyRayMarchingRes res = {0};
    res.dist = 20.0;
    $(shape, rayMarching)(pos, dir, &res);
    if(res.flagIntersect) {
      double p = 0.0;
      loop(i, 3) p += res.normal[i];
      loop(i, 3) pixel.color->rgb[i] = p;
    } else {
      loop(i, 3) pixel.color->rgb[i] = 0.0;
    }
  }
  $(img, normalise3Chan)();
  return img;
}

static CapyImg* RayMarchingToColorImg(
  CapySdf const* const shape,
          double const zoom) {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_100x100);
  forEach(pixel, img->iter) {
    double pos[3] = {
      ((double)(pixel.pos.x) - 50.0) * zoom,
      ((double)(99 - pixel.pos.y) - 50.0) * zoom,
      -10.0
    };
    double dir[3] = {0.0, 0.0, 1.0};
    CapyRayMarchingRes res = {0};
    res.dist = 20.0;
    $(shape, rayMarching)(pos, dir, &res);
    if(res.flagIntersect) {
      double* color = $(res.shape, getData)();
      double p = 0.0;
      loop(i, 3) p += res.normal[i];
      loop(i, 3) pixel.color->rgb[i] = p * color[i];
    } else {
      loop(i, 3) pixel.color->rgb[i] = 0.0;
    }
  }
  $(img, normalise3Chan)();
  return img;
}

static double SdfUser01(
  CapySdf* const that,
    double const pos[3]) {
  double* data = $(that, getData)();
  double dist =
    sqrt(pos[0] * pos[0] + pos[1] * pos[1] + pos[2] * pos[2]) - *data;
  return dist;
}

#endif
CUTEST(test001, "box") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  double check[18][4] = {
    {0.0, 0.0, 0.0, -0.5},
    {1.0, 0.0, 0.0, 0.5},
    {0.0, 1.0, 0.0, 0.5},
    {0.0, 0.0, 1.0, 0.5},
    {0.0, 1.0, 1.0, 0.707107},
    {0.0, 0.5, 0.0, 0.0},
    {0.0, 0.25, 0.0, -0.25},
    {0.4, 0.3, 0.0, -0.1},
    {0.7, 0.0, 0.4, 0.2},
    {0.0, 0.0, 0.0, -0.5},
    {-1.0, 0.0, 0.0, 0.5},
    {0.0, -1.0, 0.0, 0.5},
    {0.0, 0.0, -1.0, 0.5},
    {0.0, -1.0, -1.0, 0.707107},
    {0.0, -0.5, 0.0, 0.0},
    {0.0, -0.25, 0.0, -0.25},
    {-0.4, -0.3, 0.0, -0.1},
    {-0.7, 0.0, -0.6, 0.223607},
  };
  loop(iCheck, 18) {
    double p[3] = {0};
    loop(i, 3) p[i] = check[iCheck][i];
    double d = $(box, dist)(p);
    CUTEST_ASSERT(
      fabs(d - check[iCheck][3]) < 1e-6,
      "sdf(%lf, %lf, %lf)=%lf != %lf", p[0], p[1], p[2], d, check[iCheck][3]);
  }
  CapySdfFree(&box);
}

CUTEST(test002, "box transformed") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, translate)(0, (double[3]){0, -0.5, 0});
  $(box, rotate)(1, (double[3]){0, 0, 1}, M_PI_2);
  $(box, scale)(2, 0.5);
  CapyImg* img = FieldToImg(box, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/box01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkBox01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/box01.png differs from "
    "UnitTests/TestSdfCsg/checkBox01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&box);
}

CUTEST(test003, "box with thickness and roundness") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  box->roundness = 0.1;
  box->thickness = 0.1;
  CapyImg* img = FieldToImg(box, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/box02.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkBox02.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/box02.png differs from "
    "UnitTests/TestSdfCsg/checkBox02.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&box);
}

CUTEST(test004, "sphere") {
  CapySdf* sphere = CapySdfAllocSphere(.5);
  sphere->params[capySdfParam_radius] = 0.25;
  CapyImg* img = FieldToImg(sphere, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/sphere01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkSphere01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/sphere01.png differs from "
    "UnitTests/TestSdfCsg/checkSphere01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&sphere);
}

CUTEST(test005, "merge") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, translate)(0, (double[3]){-0.25, 0, 0});
  CapySdf* sphere = CapySdfAllocSphere(0.5);
  $(sphere, translate)(0, (double[3]){0.25, 0, 0});
  CapySdf* shape = CapySdfAllocMerge(box, sphere);
  CapyImg* img = FieldToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/merge01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkMerge01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/merge01.png differs from "
    "UnitTests/TestSdfCsg/checkMerge01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test006, "difference") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, translate)(0, (double[3]){-0.25, 0, 0});
  CapySdf* sphere = CapySdfAllocSphere(0.5);
  $(sphere, translate)(0, (double[3]){0.25, 0, 0});
  CapySdf* shape = CapySdfAllocDifference(box, sphere);
  CapyImg* img = FieldToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/difference01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkDifference01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/difference01.png differs from "
    "UnitTests/TestSdfCsg/checkDifference01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test007, "intersection") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, translate)(0, (double[3]){-0.25, 0, 0});
  CapySdf* sphere = CapySdfAllocSphere(0.5);
  $(sphere, translate)(0, (double[3]){0.25, 0, 0});
  CapySdf* shape = CapySdfAllocIntersection(box, sphere);
  CapyImg* img = FieldToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/intersection01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkIntersection01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/intersection01.png differs from "
    "UnitTests/TestSdfCsg/checkIntersection01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test008, "merge with smoothness") {
  CapySdf* boxA = CapySdfAllocBox(1, 1, 1);
  $(boxA, translate)(0, (double[3]){-0.25, -0.25, 0});
  CapySdf* boxB = CapySdfAllocBox(1, 1, 1);
  $(boxB, translate)(0, (double[3]){0.25, 0.25, 0});
  CapySdf* shape = CapySdfAllocMerge(boxA, boxB);
  shape->smoothness = 0.5;
  CapyImg* img = FieldToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/merge02.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkMerge02.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/merge02.png differs from "
    "UnitTests/TestSdfCsg/checkMerge02.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test009, "difference with smoothness") {
  CapySdf* boxA = CapySdfAllocBox(1, 1, 1);
  $(boxA, translate)(0, (double[3]){-0.25, -0.25, 0});
  CapySdf* boxB = CapySdfAllocBox(1, 1, 1);
  $(boxB, translate)(0, (double[3]){0.25, 0.25, 0});
  CapySdf* shape = CapySdfAllocDifference(boxA, boxB);
  shape->smoothness = 0.5;
  CapyImg* img = FieldToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/difference02.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkDifference02.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/difference02.png differs from "
    "UnitTests/TestSdfCsg/checkDifference02.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test010, "intersection with smoothness") {
  CapySdf* boxA = CapySdfAllocBox(1, 1, 1);
  $(boxA, translate)(0, (double[3]){-0.25, -0.25, 0});
  CapySdf* boxB = CapySdfAllocBox(1, 1, 1);
  $(boxB, translate)(0, (double[3]){0.25, 0.25, 0});
  CapySdf* shape = CapySdfAllocIntersection(boxA, boxB);
  shape->smoothness = 0.5;
  CapyImg* img = FieldToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/intersection02.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkIntersection02.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/intersection02.png differs from "
    "UnitTests/TestSdfCsg/checkIntersection02.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test011, "epsilon") {
  double epsilon = CapySdfGetEpsilon();
  CUTEST_ASSERT(
    fabs(epsilon - 1e-6) < 1e-9,
    "default value of sdfEpsilon=%lf != 1e-6", epsilon);
  CapySdfSetEpsilon(1e-3);
  epsilon = CapySdfGetEpsilon();
  CUTEST_ASSERT(
    fabs(epsilon - 1e-3) < 1e-9,
    "SdfSetEpsilon failed %lf != 1e-3", epsilon);
}

CUTEST(test012, "normal") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  double normal[3];
  double check[6][6] = {
    {1.0, 0.0, 0.0, 1.0, 0.0, 0.0},
    {1.0, 0.1, 0.0, 1.0, 0.0, 0.0},
    {1.0, 1.0, 0.0, 1.0 / sqrt(2), 1.0 / sqrt(2), 0.0},
    {0.0, 1.0, 0.0, 0.0, 1.0, 0.0},
    {0.0, 0.0, 1.0, 0.0, 0.0, 1.0},
    {0.0, 0.0, 0.1, 0.0, 0.0, 1.0},
  };
  loop(iCheck, 6) {
    double pos[3] = {0};
    loop(i, 3) pos[i] = check[iCheck][i];
    $(box, normal)(pos, normal);
    CUTEST_ASSERT(
      fabs(normal[0] - check[iCheck][3]) < 1e-9 &&
      fabs(normal[1] - check[iCheck][4]) < 1e-9 &&
      fabs(normal[2] - check[iCheck][5]) < 1e-9,
      "normal(%lf, %lf, %lf)={%lf, %lf, %lf} != {%lf, %lf, %lf}",
      pos[0], pos[1], pos[2], normal[0], normal[1], normal[2],
      check[iCheck][3], check[iCheck][4], check[iCheck][5]);
  }
  CapySdfFree(&box);
}

CUTEST(test013, "ray marching cube") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, rotate)(0, (double[3]){1, 1, 1}, M_PI_4);
  CapyImg* img = RayMarchingToImg(box, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/rayMarching01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkRayMarching01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/rayMarching01.png differs from "
    "UnitTests/TestSdfCsg/checkRayMarching01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&box);
}

CUTEST(test014, "ray marching sphere") {
  CapySdf* sphere = CapySdfAllocSphere(0.5);
  CapyImg* img = RayMarchingToImg(sphere, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/rayMarching02.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkRayMarching02.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/rayMarching02.png differs from "
    "UnitTests/TestSdfCsg/checkRayMarching02.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&sphere);
}

CUTEST(test015, "ray marching merge") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, translate)(0, (double[3]){-0.25, -0.25, 0});
  CapySdf* sphere = CapySdfAllocSphere(0.5);
  $(sphere, translate)(0, (double[3]){0.25, 0.25, 0});
  CapySdf* shape = CapySdfAllocMerge(box, sphere);
  $(shape, rotate)(0, (double[3]){1, 1, 1}, M_PI_4);
  shape->smoothness = 0.5;
  CapyImg* img = RayMarchingToImg(shape, 0.03);
  $(img, saveToPath)("UnitTests/TestSdfCsg/rayMarching03.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkRayMarching03.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/rayMarching03.png differs from "
    "UnitTests/TestSdfCsg/checkRayMarching03.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test016, "extrusion") {
  CapySdf* shape = CapySdfAllocSphere(0.5);
  $(shape, extrude)(0, 0.5);
  $(shape, rotate)(1, (double[3]){1, 0, 0}, M_PI_4);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/extrusion01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkExtrusion01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/extrusion01.png differs from "
    "UnitTests/TestSdfCsg/checkExtrusion01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test017, "symmetry") {
  CapySdf* shape = CapySdfAllocSphere(0.5);
  $(shape, translate)(0, (double[3]){0.5, 0, 0});
  $(shape, symmetry)(1);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/symmetry01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkSymmetry01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/symmetry01.png differs from "
    "UnitTests/TestSdfCsg/checkSymmetry01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test018, "revolution") {
  CapySdf* shape = CapySdfAllocBox(0.5, 0.5, 0.5);
  $(shape, translate)(0, (double[3]){0.5, 0, 0});
  $(shape, revolution)(1);
  $(shape, rotate)(2, (double[3]){1, 0, 0}, -M_PI_4);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/revolution01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkRevolution01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/revolution01.png differs from "
    "UnitTests/TestSdfCsg/checkRevolution01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test019, "user defined") {
  CapySdf* shape = CapySdfAllocUserDefined(SdfUser01);
  double param = 1.0;
  $(shape, setData)(&param);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/userDefined01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkUserDefined01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/userDefined01.png differs from "
    "UnitTests/TestSdfCsg/checkUserDefined01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test020, "torus") {
  CapySdf* shape = CapySdfAllocTorus(0.5, 0.1);
  $(shape, rotate)(0, (double[3]){1, 0, 0}, -M_PI_4);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/torus01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkTorus01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/torus01.png differs from "
    "UnitTests/TestSdfCsg/checkTorus01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test021, "capsule") {
  CapySdf* shape = CapySdfAllocCapsule(0.25, 0.25);
  $(shape, rotate)(0, (double[3]){1, 0, 0}, -M_PI_4);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/capsule01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkCapsule01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/capsule01.png differs from "
    "UnitTests/TestSdfCsg/checkCapsule01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test022, "ellipsoid") {
  CapySdf* shape = CapySdfAllocEllipsoid(0.2, 0.3, 0.4);
  $(shape, rotate)(0, (double[3]){1, 0, 0}, -M_PI_4);
  $(shape, rotate)(1, (double[3]){0, 1, 0}, M_PI_4);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/ellipsoid01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkEllipsoid01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/ellipsoid01.png differs from "
    "UnitTests/TestSdfCsg/checkEllipsoid01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test023, "union") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  $(box, translate)(0, (double[3]){-0.25, 0, 0});
  CapySdf* sphere = CapySdfAllocSphere(0.5);
  $(sphere, translate)(0, (double[3]){0.25, 0, 0});
  CapySdf* shape = CapySdfAllocUnion(box, sphere);
  double red[3] = {1, 0, 0};
  $(box, setData)(red);
  double blue[3] = {0, 0, 1};
  $(sphere, setData)(blue);
  CapyImg* img = RayMarchingToColorImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/union01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkUnion01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/union01.png differs from "
    "UnitTests/TestSdfCsg/checkUnion01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test024, "field inversion") {
  CapySdf* box = CapySdfAllocBox(1, 1, 1);
  double from[3] = {0, 0, -0.25};
  double dir[3] = {0, 0, 1.0};
  CapyRayMarchingRes resA = {0};
  resA.dist = 1.0;
  $(box, rayMarching)(from, dir, &resA);
  CUTEST_ASSERT(resA.flagIntersect, "ray marching failed");
  CUTEST_ASSERT(
    fabs(resA.pos[0] - 0.0) < 1e-6 &&
    fabs(resA.pos[1] - 0.0) < 1e-6 &&
    fabs(resA.pos[2] + 0.5) < 1e-6,
    "position of intersection incorrect (%lf, %lf, %lf)!=(0, 0, -0.5)",
    resA.pos[0], resA.pos[1], resA.pos[2]);
  CapyRayMarchingRes resB = {0};
  resB.dist = 1.0;
  dir[2] = -1.0;
  $(box, rayMarching)(from, dir, &resB);
  CUTEST_ASSERT(resB.flagIntersect, "ray marching failed");
  CUTEST_ASSERT(
    fabs(resB.pos[0] - 0.0) < 1e-6 &&
    fabs(resB.pos[1] - 0.0) < 1e-6 &&
    fabs(resB.pos[2] - 0.5) < 1e-6,
    "position of intersection incorrect (%lf, %lf, %lf)!=(0, 0, 0.5)",
    resB.pos[0], resB.pos[1], resB.pos[2]);
  CapySdfFree(&box);
}

CUTEST(test025, "cone") {
  CapySdf* shape = CapySdfAllocCone(0.3, 0.2, 0.4);
  $(shape, rotate)(0, (double[3]){1, 0, 0}, -M_PI_4);
  $(shape, rotate)(1, (double[3]){0, 1, 0}, M_PI_4);
  CapyImg* img = RayMarchingToImg(shape, 0.01);
  $(img, saveToPath)("UnitTests/TestSdfCsg/cone01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkCone01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/cone01.png differs from "
    "UnitTests/TestSdfCsg/checkCone01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test026, "plane") {
  CapySdf* shape = CapySdfAllocPlane((double[3]){1, 1, -1}, 0.1);
  CapyImg* img = FieldToImg(shape, 0.01);
  $(img, saveToPath)("UnitTests/TestSdfCsg/plane01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkPlane01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/plane01.png differs from "
    "UnitTests/TestSdfCsg/checkPlane01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

CUTEST(test027, "twist") {
  CapySdf* shape = CapySdfAllocBox(1, 1, 1);
  $(shape, twist)(0, 2.0);
  $(shape, rotate)(1, (double[3]){1, 0, 0}, -M_PI_4 * 0.5);
  CapyImg* img = RayMarchingToImg(shape, 0.02);
  $(img, saveToPath)("UnitTests/TestSdfCsg/twist01.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestSdfCsg/checkTwist01.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg),
    "UnitTests/TestSdfCsg/twist01.png differs from "
    "UnitTests/TestSdfCsg/checkTwist01.png");
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
  CapySdfFree(&shape);
}

