#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "Alloc and free") {
  size_t const dim = 2;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  CUTEST_ASSERT(true, " ");
  CapySpringMassFree(&springmass);
}

CUTEST(test002, "Add/get a mass") {
  size_t const dim = 2;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  size_t const id = 1;
  CapySpringMassNode* mass = $(springmass, addMass)(id);
  CapySpringMassNode* massCheck = $(springmass, getMass)(id);
  massCheck->pos.vals[0] = 1.0;
  bool isOk =
    (mass != NULL) &&
    (mass == massCheck) &&
    (mass->pos.dim == dim) &&
    (fabs(mass->pos.vals[0] - 1.0) < 1e-6) &&
    (mass->speed.dim == dim) &&
    (fabs(mass->speed.vals[0]) < 1e-6) &&
    (mass->acc.dim == dim) &&
    (fabs(mass->acc.vals[0]) < 1e-6) &&
    (mass->flagFixed == false);
  CUTEST_ASSERT(isOk, "add mass failed");
  CapySpringMassFree(&springmass);
}

CUTEST(test003, "Add/get a spring") {
  size_t const dim = 2;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  size_t const ids[2] = {1, 2};
  CapySpringMassNode* masses[2];
  masses[0] = $(springmass, addMass)(ids[0]);
  CapySpringMassLink* link = $(springmass, addSpring)(ids[0], ids[1]);
  masses[1] = $(springmass, getMass)(ids[1]);
  CapySpringMassLink* linkCheck = $(springmass, getSpring)(ids[0], ids[1]);
  bool isOk =
    (link != NULL) &&
    (link == linkCheck) &&
    (fabs(link->restLength - 1.0) < 1e-6) &&
    (link->masses[0] = masses[0]) &&
    (link->masses[1] = masses[1]);
  CUTEST_ASSERT(isOk, "add spring failed");
  CapySpringMassFree(&springmass);
}

CUTEST(
  test004, "Step a simple 1D system with one spring and two masses, "
  "no dampening") {
  size_t const dim = 1;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  springmass->dampening = 1.0;
  CapySpringMassLink* link = $(springmass, addSpring)(0, 1);
  (void)link;
  CapySpringMassNode* masses[2];
  loop (i, (size_t)2) masses[i] = $(springmass, getMass)(i);
  masses[0]->flagFixed = true;
  masses[1]->pos.vals[0] = 1.1;
  double const deltaT = 0.01;
  double positions[2][3];
  double speeds[2][3];
  double accs[2][3];
  double checkPositions[2][3] =
    {{0.0, 0.0, 0.0}, {1.099995, 1.099980, 1.099955}};
  double checkSpeeds[2][3] =
    {{0.0, 0.0, 0.0}, {-0.001000, -0.002000, -0.003000}};
  double checkAccs[2][3] =
    {{0.0, 0.0, 0.0}, {-0.100000, -0.099995, -0.099980}};
  bool isOk = true;
  loop(i, 3) {
    $(springmass, step)(deltaT);
    loop(j, 2) {
      positions[j][i] = masses[j]->pos.vals[0];
      isOk &= (fabs(positions[j][i] - checkPositions[j][i]) < 1e-6);
      speeds[j][i] = masses[j]->speed.vals[0];
      isOk &= (fabs(speeds[j][i] - checkSpeeds[j][i]) < 1e-6);
      accs[j][i] = masses[j]->acc.vals[0];
      isOk &= (fabs(accs[j][i] - checkAccs[j][i]) < 1e-6);
    }
  }
  CUTEST_ASSERT(isOk, "step failed");
  CapySpringMassFree(&springmass);
}

CUTEST(
  test005, "Step a simple 1D system with one spring and two masses, "
  "dampening") {
  size_t const dim = 1;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  springmass->dampening = 0.9;
  CapySpringMassLink* link = $(springmass, addSpring)(0, 1);
  (void)link;
  CapySpringMassNode* masses[2];
  loop (i, (size_t)2) masses[i] = $(springmass, getMass)(i);
  masses[0]->flagFixed = true;
  masses[1]->pos.vals[0] = 1.1;
  double const deltaT = 0.01;
  double positions[2][3];
  double speeds[2][3];
  double accs[2][3];
  double checkPositions[2][3] =
    {{0.0, 0.0, 0.0}, {1.099995, 1.099981, 1.099959}};
  double checkSpeeds[2][3] =
    {{0.0, 0.0, 0.0}, {-0.001000, -0.001900, -0.002710}};
  double checkAccs[2][3] =
    {{0.0, 0.0, 0.0}, {-0.100000, -0.099995, -0.099981}};
  bool isOk = true;
  loop(i, 3) {
    $(springmass, step)(deltaT);
    loop(j, 2) {
      positions[j][i] = masses[j]->pos.vals[0];
      isOk &= (fabs(positions[j][i] - checkPositions[j][i]) < 1e-6);
      speeds[j][i] = masses[j]->speed.vals[0];
      isOk &= (fabs(speeds[j][i] - checkSpeeds[j][i]) < 1e-6);
      accs[j][i] = masses[j]->acc.vals[0];
      isOk &= (fabs(accs[j][i] - checkAccs[j][i]) < 1e-6);
    }
  }
  CUTEST_ASSERT(isOk, "step failed");
  CapySpringMassFree(&springmass);
}

CUTEST(
  test006, "Step a simple 1D system with one spring and two masses "
  "until stability") {
  size_t const dim = 1;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  springmass->dampening = 0.99;
  springmass->nbMaxStep = 100000;
  CapySpringMassLink* link = $(springmass, addSpring)(0, 1);
  (void)link;
  CapySpringMassNode* masses[2];
  loop (i, (size_t)2) masses[i] = $(springmass, getMass)(i);
  masses[0]->flagFixed = true;
  masses[1]->pos.vals[0] = 1.1;
  double stresses[2];
  stresses[0] = $(springmass, getStress)();
  double const deltaT = 0.01;
  $(springmass, stepToStableState)(deltaT);
  stresses[1] = $(springmass, getStress)();
  bool isOk =
    (fabs(stresses[0] - 0.1) < 1e-6) &&
    (fabs(stresses[1]) < 1e-6) &&
    (fabs(masses[1]->acc.vals[0]) < 1e-6) &&
    (fabs(masses[1]->speed.vals[0]) < 1e-6) &&
    (fabs(masses[1]->pos.vals[0] - 1.0) < 1e-6);
  CUTEST_ASSERT(isOk, "stepToStability failed");
  CapySpringMassFree(&springmass);
}

CUTEST(
  test007, "Step a 2D system with several spring and masses "
  "until stability (1)") {
  size_t const dim = 2;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  springmass->dampening = 0.99;
  springmass->nbMaxStep = 100000;
  CapySpringMassLink* link = $(springmass, addSpring)(0, 1);
  (void)link;
  CapySpringMassNode* masses[2];
  loop (i, (size_t)2) masses[i] = $(springmass, getMass)(i);
  masses[0]->flagFixed = true;
  masses[1]->pos.vals[0] = 1.1;
  masses[1]->pos.vals[1] = 1.1;
  double stresses[2];
  stresses[0] = $(springmass, getStress)();
  double const deltaT = 0.01;
  $(springmass, stepToStableState)(deltaT);
  stresses[1] = $(springmass, getStress)();
  bool isOk =
    (fabs(stresses[0] - 0.555635) < 1e-6) &&
    (fabs(stresses[1] - 0.000001) < 1e-6) &&
    (fabs(masses[1]->acc.vals[0]) < 1e-6) &&
    (fabs(masses[1]->acc.vals[1]) < 1e-6) &&
    (fabs(masses[1]->speed.vals[0]) < 1e-6) &&
    (fabs(masses[1]->speed.vals[1]) < 1e-6) &&
    (fabs(masses[1]->pos.vals[0] - 0.707106) < 1e-6) &&
    (fabs(masses[1]->pos.vals[1] - 0.707106) < 1e-6);
  CUTEST_ASSERT(isOk, "stepToStability failed");
  CapySpringMassFree(&springmass);
}

CUTEST(
  test008, "Step a 2D system with several spring and masses "
  "until stability (2)") {
  size_t const dim = 2;
  CapySpringMass* springmass = CapySpringMassAlloc(dim);
  springmass->dampening = 0.99;
  springmass->nbMaxStep = 100000;
  CapySpringMassLink* links[5];
  links[0] = $(springmass, addSpring)(0, 1);
  links[0]->restLength = 6.308681;
  loop(i, 2) links[0]->coeffs[i] = 1.5;
  links[1] = $(springmass, addSpring)(1, 2);
  links[1]->restLength = 6.082929;
  loop(i, 2) links[1]->coeffs[i] = 1.441619;
  links[2] = $(springmass, addSpring)(3, 1);
  links[2]->restLength = 6.027145;
  loop(i, 2) links[2]->coeffs[i] = 1.427298;
  links[3] = $(springmass, addSpring)(4, 2);
  links[3]->restLength = 5.643969;
  loop(i, 2) links[3]->coeffs[i] = 1.330123;
  links[4] = $(springmass, addSpring)(4, 5);
  links[4]->restLength = 5.643969;
  loop(i, 2) links[4]->coeffs[i] = 1.330123;
  CapySpringMassNode* masses[6];
  CapyRandom rng = CapyRandomCreate(1);
  loop (i, 6) {
    masses[i] = $(springmass, getMass)((size_t)i);
    masses[i]->pos.vals[0] = $(&rng, getDouble)() * 100.0;
    masses[i]->pos.vals[1] = $(&rng, getDouble)() * 100.0;
  }
  $(&rng, destruct)();
  masses[0]->flagFixed = true;
  double stresses[2];
  stresses[0] = $(springmass, getStress)();
  double const deltaT = 0.01;
  $(springmass, stepToStableState)(deltaT);
  stresses[1] = $(springmass, getStress)();
  bool isOk =
    (fabs(stresses[0] - 260.398395) < 1e-6) &&
    (fabs(stresses[1] - 0.000006) < 1e-6) &&
    (fabs(masses[1]->acc.vals[0]) < 1e-6) &&
    (fabs(masses[1]->acc.vals[1]) < 1e-6) &&
    (fabs(masses[1]->speed.vals[0]) < 1e-6) &&
    (fabs(masses[1]->speed.vals[1]) < 1e-6) &&
    (fabs(masses[1]->pos.vals[0] - 11.644001) < 1e-6) &&
    (fabs(masses[1]->pos.vals[1] - 30.175398) < 1e-6);
  CUTEST_ASSERT(isOk, "stepToStability failed");
  CapySpringMassFree(&springmass);
}

